<?php
/**
 * Módulo de Importación y Exportación
 * Permite la carga masiva de rendiciones y órdenes de compra,
 * así como la exportación de datos a Excel o CSV
 */

// Definir el título de la página
$page_title = 'Importación y Exportación de Datos';

// Incluir archivos de configuración y funciones
require_once 'includes/config.php';
require_once 'includes/functions.php';

// Verificar sesión
verificarSesion();
$user_id = $_SESSION['user_id'];
$user_rol = $_SESSION['user_rol'] ?? '';
$user_nombre = $_SESSION['user_nombre'] ?? '';
$user_apellido = $_SESSION['user_apellido'] ?? '';
$is_admin = ($user_rol === 'admin' || $user_rol === 'super_admin');

// Verificar permisos
if (!tienePermiso('carga_masiva') && !$is_admin) {
    $_SESSION['flash_message'] = "No tienes permisos para acceder a esta sección";
    $_SESSION['flash_type'] = 'danger';
    header('Location: dashboard.php');
    exit;
}

// Definir acción a realizar
$accion = isset($_GET['accion']) ? $_GET['accion'] : 'importar';

// Variables para resultados de importación/exportación
$resultados_importacion = null;
$errores_importacion = [];
$archivo_exportacion = null;

// Conexión a la base de datos
try {
    $conn = new PDO("mysql:host=$db_host;dbname=$db_name;charset=utf8mb4", $db_user, $db_pass, $db_options);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // Obtener categorías para rendiciones
    $stmt = $conn->prepare("SELECT id, nombre FROM categorias_gastos WHERE activo = 1 ORDER BY nombre");
    $stmt->execute();
    $categorias = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Obtener usuarios
    if ($is_admin) {
        $stmt = $conn->prepare("SELECT id, nombre, apellido FROM users WHERE activo = 1 ORDER BY nombre, apellido");
        $stmt->execute();
        $usuarios = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // Procesar formulario de importación
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && $accion === 'importar') {
        $tipo_documento = $_POST['tipo_documento'] ?? '';
        
        // Verificar si se ha subido un archivo
        if (isset($_FILES['archivo_importacion']) && $_FILES['archivo_importacion']['error'] === UPLOAD_ERR_OK) {
            $archivo = $_FILES['archivo_importacion'];
            $nombre_archivo = $archivo['name'];
            $ruta_temporal = $archivo['tmp_name'];
            $tipo_archivo = $archivo['type'];
            
            // Verificar extensión
            $extension = strtolower(pathinfo($nombre_archivo, PATHINFO_EXTENSION));
            if (!in_array($extension, ['csv', 'xlsx', 'xls'])) {
                $errores_importacion[] = "El archivo debe ser de tipo CSV o Excel (.xlsx/.xls)";
            } else {
                // Procesar según el tipo de archivo
                if ($extension === 'csv') {
                    // Procesar CSV
                    $registros = procesarCSV($ruta_temporal, $tipo_documento);
                } else {
                    // Procesar Excel
                    $registros = procesarExcel($ruta_temporal, $tipo_documento);
                }
                
                // Si hay registros válidos, procesarlos
                if (!empty($registros) && empty($errores_importacion)) {
                    $resultados_importacion = importarDatos($conn, $tipo_documento, $registros, $user_id);
                }
            }
        } else {
            $errores_importacion[] = "Debes seleccionar un archivo para importar";
        }
    }
    
    // Procesar solicitud de exportación
    if ($accion === 'exportar' && isset($_GET['tipo_documento'])) {
        $tipo_documento = $_GET['tipo_documento'];
        $formato = $_GET['formato'] ?? 'xlsx';
        $fecha_desde = $_GET['fecha_desde'] ?? '';
        $fecha_hasta = $_GET['fecha_hasta'] ?? '';
        $usuario_id = isset($_GET['usuario_id']) ? intval($_GET['usuario_id']) : 0;
        $estado_id = isset($_GET['estado_id']) ? intval($_GET['estado_id']) : 0;
        
        // Si no hay fechas específicas, usar el año actual
        if (empty($fecha_desde)) {
            $fecha_desde = date('Y-01-01');
        }
        if (empty($fecha_hasta)) {
            $fecha_hasta = date('Y-12-31');
        }
        
        // Exportar datos
        exportarDatos($conn, $tipo_documento, $formato, $fecha_desde, $fecha_hasta, $usuario_id, $estado_id);
    }
    
} catch (PDOException $e) {
    $error_message = 'Error de conexión a la base de datos: ' . $e->getMessage();
}

/**
 * Procesa un archivo CSV para importación
 * 
 * @param string $ruta_archivo Ruta al archivo CSV
 * @param string $tipo_documento Tipo de documento (rendicion o orden_compra)
 * @return array Registros procesados
 */
function procesarCSV($ruta_archivo, $tipo_documento) {
    global $errores_importacion;
    $registros = [];
    
    // Abrir el archivo
    $handle = fopen($ruta_archivo, 'r');
    if ($handle === false) {
        $errores_importacion[] = "No se pudo abrir el archivo CSV";
        return $registros;
    }
    
    // Leer encabezados
    $encabezados = fgetcsv($handle);
    if ($encabezados === false) {
        $errores_importacion[] = "El archivo CSV está vacío o no tiene el formato correcto";
        fclose($handle);
        return $registros;
    }
    
    // Verificar encabezados según tipo de documento
    $encabezados_esperados = [];
    
    if ($tipo_documento === 'rendicion') {
        $encabezados_esperados = ['fecha', 'descripcion', 'categoria', 'monto', 'detalle'];
    } elseif ($tipo_documento === 'orden_compra') {
        $encabezados_esperados = ['fecha', 'proveedor', 'rut_proveedor', 'numero_documento', 'producto', 'cantidad', 'precio_unitario', 'condiciones_pago', 'descripcion'];
    }
    
    // Validar encabezados
    $encabezados = array_map('trim', $encabezados);
    $encabezados = array_map('strtolower', $encabezados);
    
    foreach ($encabezados_esperados as $encabezado) {
        if (!in_array($encabezado, $encabezados)) {
            $errores_importacion[] = "El archivo no contiene el encabezado '$encabezado' que es obligatorio";
        }
    }
    
    if (!empty($errores_importacion)) {
        fclose($handle);
        return $registros;
    }
    
    // Procesar filas
    $numero_fila = 2; // Empezamos en 2 porque la fila 1 son los encabezados
    
    while (($fila = fgetcsv($handle)) !== false) {
        // Omitir filas vacías
        if (count($fila) === 1 && empty($fila[0])) {
            continue;
        }
        
        // Convertir a UTF-8 si es necesario
        $fila = array_map(function($valor) {
            if (!mb_check_encoding($valor, 'UTF-8')) {
                return mb_convert_encoding($valor, 'UTF-8', 'auto');
            }
            return $valor;
        }, $fila);
        
        // Crear un registro asociativo usando los encabezados
        $registro = [];
        foreach ($encabezados as $indice => $encabezado) {
            if (isset($fila[$indice])) {
                $registro[$encabezado] = trim($fila[$indice]);
            } else {
                $registro[$encabezado] = '';
            }
        }
        
        // Validar datos según tipo de documento
        $errores_fila = validarRegistro($registro, $tipo_documento, $numero_fila);
        
        if (empty($errores_fila)) {
            $registros[] = $registro;
        } else {
            $errores_importacion = array_merge($errores_importacion, $errores_fila);
        }
        
        $numero_fila++;
    }
    
    fclose($handle);
    return $registros;
}

/**
 * Procesa un archivo Excel para importación
 * 
 * @param string $ruta_archivo Ruta al archivo Excel
 * @param string $tipo_documento Tipo de documento (rendicion o orden_compra)
 * @return array Registros procesados
 */
function procesarExcel($ruta_archivo, $tipo_documento) {
    global $errores_importacion;
    $registros = [];
    
    // Verificar si la extensión PhpSpreadsheet está instalada
    if (!class_exists('PhpOffice\PhpSpreadsheet\IOFactory')) {
        // Utilizar método alternativo con funciones nativas
        if (function_exists('xlsx_to_array')) {
            $datos = xlsx_to_array($ruta_archivo);
        } else {
            $errores_importacion[] = "No se pueden procesar archivos Excel. Por favor, usa archivos CSV.";
            return $registros;
        }
    } else {
        // Usar PhpSpreadsheet si está disponible
        try {
            $reader = \PhpOffice\PhpSpreadsheet\IOFactory::createReaderForFile($ruta_archivo);
            $reader->setReadDataOnly(true);
            $spreadsheet = $reader->load($ruta_archivo);
            $worksheet = $spreadsheet->getActiveSheet();
            $datos = $worksheet->toArray();
        } catch (Exception $e) {
            $errores_importacion[] = "Error al leer el archivo Excel: " . $e->getMessage();
            return $registros;
        }
    }
    
    // Validar que hay datos
    if (empty($datos) || count($datos) < 2) { // Al menos encabezados y una fila de datos
        $errores_importacion[] = "El archivo Excel está vacío o no contiene datos suficientes";
        return $registros;
    }
    
    // Extraer encabezados (primera fila)
    $encabezados = array_map('trim', $datos[0]);
    $encabezados = array_map('strtolower', $encabezados);
    
    // Verificar encabezados según tipo de documento
    $encabezados_esperados = [];
    
    if ($tipo_documento === 'rendicion') {
        $encabezados_esperados = ['fecha', 'descripcion', 'categoria', 'monto', 'detalle'];
    } elseif ($tipo_documento === 'orden_compra') {
        $encabezados_esperados = ['fecha', 'proveedor', 'rut_proveedor', 'numero_documento', 'producto', 'cantidad', 'precio_unitario', 'condiciones_pago', 'descripcion'];
    }
    
    // Validar encabezados
    foreach ($encabezados_esperados as $encabezado) {
        if (!in_array($encabezado, $encabezados)) {
            $errores_importacion[] = "El archivo no contiene el encabezado '$encabezado' que es obligatorio";
        }
    }
    
    if (!empty($errores_importacion)) {
        return $registros;
    }
    
    // Procesar filas (a partir de la segunda)
    for ($i = 1; $i < count($datos); $i++) {
        $fila = $datos[$i];
        
        // Omitir filas vacías
        if (count(array_filter($fila)) === 0) {
            continue;
        }
        
        // Crear un registro asociativo usando los encabezados
        $registro = [];
        foreach ($encabezados as $indice => $encabezado) {
            if (isset($fila[$indice])) {
                $registro[$encabezado] = trim((string)$fila[$indice]);
            } else {
                $registro[$encabezado] = '';
            }
        }
        
        // Validar datos según tipo de documento
        $errores_fila = validarRegistro($registro, $tipo_documento, $i + 1);
        
        if (empty($errores_fila)) {
            $registros[] = $registro;
        } else {
            $errores_importacion = array_merge($errores_importacion, $errores_fila);
        }
    }
    
    return $registros;
}

/**
 * Valida un registro según el tipo de documento
 * 
 * @param array $registro Registro a validar
 * @param string $tipo_documento Tipo de documento
 * @param int $numero_fila Número de fila para mensajes de error
 * @return array Errores encontrados
 */
function validarRegistro($registro, $tipo_documento, $numero_fila) {
    $errores = [];
    
    if ($tipo_documento === 'rendicion') {
        // Validar fecha
        if (empty($registro['fecha'])) {
            $errores[] = "Fila $numero_fila: La fecha es obligatoria";
        } elseif (!validarFecha($registro['fecha'])) {
            $errores[] = "Fila $numero_fila: La fecha no tiene un formato válido (debe ser YYYY-MM-DD)";
        }
        
        // Validar descripción
        if (empty($registro['descripcion'])) {
            $errores[] = "Fila $numero_fila: La descripción es obligatoria";
        }
        
        // Validar categoría
        if (empty($registro['categoria'])) {
            $errores[] = "Fila $numero_fila: La categoría es obligatoria";
        } else {
            global $categorias;
            $categoria_valida = false;
            
            foreach ($categorias as $categoria) {
                if (strtolower($categoria['nombre']) === strtolower($registro['categoria'])) {
                    $categoria_valida = true;
                    break;
                }
            }
            
            if (!$categoria_valida) {
                $errores[] = "Fila $numero_fila: La categoría '{$registro['categoria']}' no es válida";
            }
        }
        
        // Validar monto
        if (empty($registro['monto'])) {
            $errores[] = "Fila $numero_fila: El monto es obligatorio";
        } elseif (!is_numeric(str_replace(['$', '.', ','], '', $registro['monto']))) {
            $errores[] = "Fila $numero_fila: El monto debe ser un número";
        } elseif (floatval(str_replace(['$', '.', ','], '', $registro['monto'])) <= 0) {
            $errores[] = "Fila $numero_fila: El monto debe ser mayor que cero";
        }
    } 
    elseif ($tipo_documento === 'orden_compra') {
        // Validar fecha
        if (empty($registro['fecha'])) {
            $errores[] = "Fila $numero_fila: La fecha es obligatoria";
        } elseif (!validarFecha($registro['fecha'])) {
            $errores[] = "Fila $numero_fila: La fecha no tiene un formato válido (debe ser YYYY-MM-DD)";
        }
        
        // Validar proveedor
        if (empty($registro['proveedor'])) {
            $errores[] = "Fila $numero_fila: El nombre del proveedor es obligatorio";
        }
        
        // Validar RUT proveedor
        if (empty($registro['rut_proveedor'])) {
            $errores[] = "Fila $numero_fila: El RUT del proveedor es obligatorio";
        } elseif (!validarRut($registro['rut_proveedor'])) {
            $errores[] = "Fila $numero_fila: El RUT del proveedor no es válido";
        }
        
        // Validar número de documento
        if (empty($registro['numero_documento'])) {
            $errores[] = "Fila $numero_fila: El número de documento es obligatorio";
        }
        
        // Validar producto
        if (empty($registro['producto'])) {
            $errores[] = "Fila $numero_fila: El nombre del producto es obligatorio";
        }
        
        // Validar cantidad
        if (empty($registro['cantidad'])) {
            $errores[] = "Fila $numero_fila: La cantidad es obligatoria";
        } elseif (!is_numeric($registro['cantidad'])) {
            $errores[] = "Fila $numero_fila: La cantidad debe ser un número";
        } elseif (intval($registro['cantidad']) <= 0) {
            $errores[] = "Fila $numero_fila: La cantidad debe ser mayor que cero";
        }
        
        // Validar precio unitario
        if (empty($registro['precio_unitario'])) {
            $errores[] = "Fila $numero_fila: El precio unitario es obligatorio";
        } elseif (!is_numeric(str_replace(['$', '.', ','], '', $registro['precio_unitario']))) {
            $errores[] = "Fila $numero_fila: El precio unitario debe ser un número";
        } elseif (floatval(str_replace(['$', '.', ','], '', $registro['precio_unitario'])) <= 0) {
            $errores[] = "Fila $numero_fila: El precio unitario debe ser mayor que cero";
        }
    }
    
    return $errores;
}

/**
 * Valida el formato de una fecha
 * 
 * @param string $fecha Fecha a validar
 * @return bool True si la fecha es válida
 */
function validarFecha($fecha) {
    // Intentar diferentes formatos comunes
    $formatos = ['Y-m-d', 'd/m/Y', 'd-m-Y', 'Y/m/d'];
    
    foreach ($formatos as $formato) {
        $fecha_obj = DateTime::createFromFormat($formato, $fecha);
        if ($fecha_obj && $fecha_obj->format($formato) === $fecha) {
            return true;
        }
    }
    
    return false;
}

/**
 * Importa los datos a la base de datos
 * 
 * @param PDO $conn Conexión a la base de datos
 * @param string $tipo_documento Tipo de documento
 * @param array $registros Registros a importar
 * @param int $user_id ID del usuario actual
 * @return array Resultados de la importación
 */
function importarDatos($conn, $tipo_documento, $registros, $user_id) {
    $resultados = [
        'total' => count($registros),
        'exitosos' => 0,
        'fallidos' => 0,
        'errores' => []
    ];
    
    if ($tipo_documento === 'rendicion') {
        // Obtener mapeo de categorías
        $stmt = $conn->prepare("SELECT id, nombre FROM categorias_gastos WHERE activo = 1");
        $stmt->execute();
        $categorias_db = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $mapa_categorias = [];
        foreach ($categorias_db as $categoria) {
            $mapa_categorias[strtolower($categoria['nombre'])] = $categoria['id'];
        }
        
        // Importar cada rendición
        foreach ($registros as $indice => $registro) {
            try {
                $conn->beginTransaction();
                
                // Convertir fecha al formato MySQL si es necesario
                $fecha = $registro['fecha'];
                if (strpos($fecha, '/') !== false) {
                    $partes = explode('/', $fecha);
                    if (count($partes) === 3) {
                        // Asumir formato dd/mm/yyyy
                        if (strlen($partes[2]) === 4) {
                            $fecha = "{$partes[2]}-{$partes[1]}-{$partes[0]}";
                        }
                    }
                }
                
                // Limpiar monto
                $monto = str_replace(['$', '.', ','], '', $registro['monto']);
                
                // Insertar rendición
                $stmt = $conn->prepare("
                    INSERT INTO rendiciones 
                    (user_id, descripcion, fecha, monto_total, estado_id, fecha_creacion, ultima_modificacion) 
                    VALUES (:user_id, :descripcion, :fecha, :monto_total, 1, NOW(), NOW())
                ");
                
                $stmt->bindParam(':user_id', $user_id);
                $stmt->bindParam(':descripcion', $registro['descripcion']);
                $stmt->bindParam(':fecha', $fecha);
                $stmt->bindParam(':monto_total', $monto);
                $stmt->execute();
                
                $rendicion_id = $conn->lastInsertId();
                
                // Buscar categoría
                $categoria_nombre = strtolower(trim($registro['categoria']));
                if (!isset($mapa_categorias[$categoria_nombre])) {
                    throw new Exception("La categoría '{$registro['categoria']}' no existe en el sistema");
                }
                
                $categoria_id = $mapa_categorias[$categoria_nombre];
                
                // Insertar detalle de rendición
                $stmt = $conn->prepare("
                    INSERT INTO rendicion_categoria 
                    (rendicion_id, categoria_id, monto, detalle) 
                    VALUES (:rendicion_id, :categoria_id, :monto, :detalle)
                ");
                
                $stmt->bindParam(':rendicion_id', $rendicion_id);
                $stmt->bindParam(':categoria_id', $categoria_id);
                $stmt->bindParam(':monto', $monto);
                $stmt->bindParam(':detalle', $registro['detalle']);
                $stmt->execute();
                
                // Registrar actividad
                registrarActividad($conn, $user_id, 'crear', 'Importación de rendición', $rendicion_id, 'rendicion');
                
                $conn->commit();
                $resultados['exitosos']++;
                
            } catch (Exception $e) {
                $conn->rollBack();
                $resultados['fallidos']++;
                $resultados['errores'][] = "Error en fila " . ($indice + 2) . ": " . $e->getMessage();
            }
        }
    } 
    elseif ($tipo_documento === 'orden_compra') {
        // Importar cada orden de compra
        foreach ($registros as $indice => $registro) {
            try {
                $conn->beginTransaction();
                
                // Convertir fecha al formato MySQL si es necesario
                $fecha = $registro['fecha'];
                if (strpos($fecha, '/') !== false) {
                    $partes = explode('/', $fecha);
                    if (count($partes) === 3) {
                        // Asumir formato dd/mm/yyyy
                        if (strlen($partes[2]) === 4) {
                            $fecha = "{$partes[2]}-{$partes[1]}-{$partes[0]}";
                        }
                    }
                }
                
                // Limpiar precio unitario
                $precio_unitario = str_replace(['$', '.', ','], '', $registro['precio_unitario']);
                
                // Calcular subtotal
                $cantidad = intval($registro['cantidad']);
                $subtotal = $cantidad * $precio_unitario;
                
                // Insertar orden de compra
                $stmt = $conn->prepare("
                    INSERT INTO ordenes_compra 
                    (user_id, proveedor, rut_proveedor, fecha, descripcion, numero_documento, 
                    condiciones_pago, monto_total, estado_id, fecha_creacion, ultima_modificacion) 
                    VALUES (:user_id, :proveedor, :rut_proveedor, :fecha, :descripcion, :numero_documento, 
                    :condiciones_pago, :monto_total, 1, NOW(), NOW())
                ");
                
                $stmt->bindParam(':user_id', $user_id);
                $stmt->bindParam(':proveedor', $registro['proveedor']);
                $stmt->bindParam(':rut_proveedor', $registro['rut_proveedor']);
                $stmt->bindParam(':fecha', $fecha);
                $stmt->bindParam(':descripcion', $registro['descripcion']);
                $stmt->bindParam(':numero_documento', $registro['numero_documento']);
                $stmt->bindParam(':condiciones_pago', $registro['condiciones_pago']);
                $stmt->bindParam(':monto_total', $subtotal);
                $stmt->execute();
                
                $orden_id = $conn->lastInsertId();
                
                // Insertar detalle de orden
                $stmt = $conn->prepare("
                    INSERT INTO detalle_orden_compra 
                    (orden_compra_id, guia_entrada_id, producto, cantidad, precio_unitario, subtotal) 
                    VALUES (:orden_compra_id, :guia_entrada_id, :producto, :cantidad, :precio_unitario, :subtotal)
                ");
                
                $stmt->bindParam(':orden_compra_id', $orden_id);
                $stmt->bindParam(':guia_entrada_id', $orden_id); // En la DB guia_entrada_id apunta a orden_compra_id
                $stmt->bindParam(':producto', $registro['producto']);
                $stmt->bindParam(':cantidad', $cantidad);
                $stmt->bindParam(':precio_unitario', $precio_unitario);
                $stmt->bindParam(':subtotal', $subtotal);
                $stmt->execute();
                
                // Registrar actividad
                registrarActividad($conn, $user_id, 'crear', 'Importación de orden de compra', $orden_id, 'orden_compra');
                
                $conn->commit();
                $resultados['exitosos']++;
                
            } catch (Exception $e) {
                $conn->rollBack();
                $resultados['fallidos']++;
                $resultados['errores'][] = "Error en fila " . ($indice + 2) . ": " . $e->getMessage();
            }
        }
    }
    
    return $resultados;
}

/**
 * Exporta datos a un archivo
 * 
 * @param PDO $conn Conexión a la base de datos
 * @param string $tipo_documento Tipo de documento
 * @param string $formato Formato de exportación (xlsx o csv)
 * @param string $fecha_desde Fecha inicial
 * @param string $fecha_hasta Fecha final
 * @param int $usuario_id ID del usuario para filtrar (0 para todos)
 * @param int $estado_id ID del estado para filtrar (0 para todos)
 */
function exportarDatos($conn, $tipo_documento, $formato, $fecha_desde, $fecha_hasta, $usuario_id, $estado_id) {
    global $user_id, $is_admin;
    
    // Preparar nombres de archivo y encabezados
    $fecha_actual = date('Y-m-d_H-i-s');
    $nombre_archivo = $tipo_documento . '_' . $fecha_actual;
    $encabezados = [];
    
    // Construir la consulta según el tipo de documento
    if ($tipo_documento === 'rendicion') {
        $sql = "
            SELECT r.id, r.fecha, r.descripcion, r.monto_total, 
                   u.nombre as usuario_nombre, u.apellido as usuario_apellido, 
                   e.nombre as estado_nombre, 
                   rc.categoria_id, cg.nombre as categoria_nombre, rc.monto, rc.detalle
            FROM rendiciones r
            JOIN users u ON r.user_id = u.id
            JOIN estados e ON r.estado_id = e.id
            JOIN rendicion_categoria rc ON r.id = rc.rendicion_id
            JOIN categorias_gastos cg ON rc.categoria_id = cg.id
            WHERE r.fecha BETWEEN :fecha_desde AND :fecha_hasta
        ";
        
        $encabezados = [
            'Código', 'Fecha', 'Descripción', 'Categoría', 'Monto', 'Detalle', 
            'Estado', 'Usuario', 'Fecha Creación'
        ];
    } 
    elseif ($tipo_documento === 'orden_compra') {
        $sql = "
            SELECT oc.id, oc.fecha, oc.proveedor, oc.rut_proveedor, 
                   oc.numero_documento, oc.monto_total, oc.condiciones_pago, oc.descripcion,
                   u.nombre as usuario_nombre, u.apellido as usuario_apellido, 
                   e.nombre as estado_nombre,
                   doc.producto, doc.cantidad, doc.precio_unitario, doc.subtotal
            FROM ordenes_compra oc
            JOIN users u ON oc.user_id = u.id
            JOIN estados e ON oc.estado_id = e.id
            JOIN detalle_orden_compra doc ON oc.id = doc.orden_compra_id
            WHERE oc.fecha BETWEEN :fecha_desde AND :fecha_hasta
        ";
        
        $encabezados = [
            'Código', 'Fecha', 'Proveedor', 'RUT Proveedor', 'N° Documento', 
            'Producto', 'Cantidad', 'Precio Unitario', 'Subtotal',
            'Condiciones de Pago', 'Descripción', 'Estado', 'Usuario', 'Fecha Creación'
        ];
    } else {
        // Tipo no válido
        header('HTTP/1.1 400 Bad Request');
        echo "Tipo de documento no válido";
        exit;
    }
    
    // Aplicar filtro de usuario según permisos
    if (!$is_admin) {
        // Si no es admin, solo ver sus propios documentos
        $sql .= " AND " . ($tipo_documento === 'rendicion' ? "r" : "oc") . ".user_id = :user_id";
    } 
    elseif ($usuario_id > 0) {
        // Si es admin y se seleccionó un usuario específico
        $sql .= " AND " . ($tipo_documento === 'rendicion' ? "r" : "oc") . ".user_id = :user_id";
    }
    
    // Filtro por estado
    if ($estado_id > 0) {
        $sql .= " AND " . ($tipo_documento === 'rendicion' ? "r" : "oc") . ".estado_id = :estado_id";
    }
    
    // Ordenar
    $sql .= " ORDER BY " . ($tipo_documento === 'rendicion' ? "r" : "oc") . ".fecha_creacion DESC";
    
    // Preparar los parámetros
    $params = [':fecha_desde' => $fecha_desde, ':fecha_hasta' => $fecha_hasta];
    
    if (!$is_admin || $usuario_id > 0) {
        $params[':user_id'] = !$is_admin ? $user_id : $usuario_id;
    }
    
    if ($estado_id > 0) {
        $params[':estado_id'] = $estado_id;
    }
    
    // Ejecutar consulta
    $stmt = $conn->prepare($sql);
    foreach ($params as $param => $value) {
        $stmt->bindValue($param, $value);
    }
    $stmt->execute();
    
    $resultados = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Si es rendición, agrupar por ID para tener una fila por rendición
    $datos_exportacion = [];
    
    if ($tipo_documento === 'rendicion') {
        $rendiciones_agrupadas = [];
        
        foreach ($resultados as $fila) {
            $id = $fila['id'];
            $codigo = generarCodigo('REN', $id);
            
            // Si es la primera vez que vemos esta rendición, inicializar
            if (!isset($rendiciones_agrupadas[$id])) {
                $rendiciones_agrupadas[$id] = [
                    'Código' => $codigo,
                    'Fecha' => formatoFecha($fila['fecha']),
                    'Descripción' => $fila['descripcion'],
                    'Categoría' => $fila['categoria_nombre'],
                    'Monto' => formatoMoneda($fila['monto']),
                    'Detalle' => $fila['detalle'],
                    'Estado' => $fila['estado_nombre'],
                    'Usuario' => $fila['usuario_nombre'] . ' ' . $fila['usuario_apellido'],
                    'Fecha Creación' => isset($fila['fecha_creacion']) ? formatoFecha($fila['fecha_creacion']) : ''
                ];
            } else {
                // Si ya existe, agregar categoría adicional
                $rendiciones_agrupadas[$id]['Categoría'] .= ', ' . $fila['categoria_nombre'];
                $rendiciones_agrupadas[$id]['Monto'] .= ', ' . formatoMoneda($fila['monto']);
                $rendiciones_agrupadas[$id]['Detalle'] .= ', ' . $fila['detalle'];
            }
        }
        
        $datos_exportacion = array_values($rendiciones_agrupadas);
    } 
    elseif ($tipo_documento === 'orden_compra') {
        $ordenes_agrupadas = [];
        
        foreach ($resultados as $fila) {
            $id = $fila['id'];
            $codigo = generarCodigo('OC', $id);
            
            // Si es la primera vez que vemos esta orden, inicializar
            if (!isset($ordenes_agrupadas[$id])) {
                $ordenes_agrupadas[$id] = [
                    'Código' => $codigo,
                    'Fecha' => formatoFecha($fila['fecha']),
                    'Proveedor' => $fila['proveedor'],
                    'RUT Proveedor' => $fila['rut_proveedor'],
                    'N° Documento' => $fila['numero_documento'],
                    'Producto' => $fila['producto'],
                    'Cantidad' => $fila['cantidad'],
                    'Precio Unitario' => formatoMoneda($fila['precio_unitario']),
                    'Subtotal' => formatoMoneda($fila['subtotal']),
                    'Condiciones de Pago' => $fila['condiciones_pago'],
                    'Descripción' => $fila['descripcion'],
                    'Estado' => $fila['estado_nombre'],
                    'Usuario' => $fila['usuario_nombre'] . ' ' . $fila['usuario_apellido'],
                    'Fecha Creación' => isset($fila['fecha_creacion']) ? formatoFecha($fila['fecha_creacion']) : ''
                ];
            } else {
                // Si ya existe, agregar producto adicional
                $ordenes_agrupadas[$id]['Producto'] .= ', ' . $fila['producto'];
                $ordenes_agrupadas[$id]['Cantidad'] .= ', ' . $fila['cantidad'];
                $ordenes_agrupadas[$id]['Precio Unitario'] .= ', ' . formatoMoneda($fila['precio_unitario']);
                $ordenes_agrupadas[$id]['Subtotal'] .= ', ' . formatoMoneda($fila['subtotal']);
            }
        }
        
        $datos_exportacion = array_values($ordenes_agrupadas);
    }
    
    // Exportar según formato seleccionado
    if ($formato === 'csv') {
        // Exportar a CSV
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $nombre_archivo . '.csv"');
        
        $output = fopen('php://output', 'w');
        
        // Escribir encabezados
        fputcsv($output, $encabezados);
        
        // Escribir datos
        foreach ($datos_exportacion as $fila) {
            fputcsv($output, $fila);
        }
        
        fclose($output);
        exit;
    } 
    elseif ($formato === 'xlsx') {
        // Exportar a Excel (XLSX)
        if (class_exists('PhpOffice\PhpSpreadsheet\Spreadsheet')) {
            // Usar PhpSpreadsheet si está disponible
            $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
            $sheet = $spreadsheet->getActiveSheet();
            
            // Escribir encabezados
            for ($i = 0; $i < count($encabezados); $i++) {
                $sheet->setCellValueByColumnAndRow($i + 1, 1, $encabezados[$i]);
            }
            
            // Escribir datos
            $fila = 2;
            foreach ($datos_exportacion as $data) {
                $columna = 1;
                foreach ($data as $valor) {
                    $sheet->setCellValueByColumnAndRow($columna, $fila, $valor);
                    $columna++;
                }
                $fila++;
            }
            
            // Aplicar estilos a encabezados
            $styleArray = [
                'font' => [
                    'bold' => true,
                ],
                'fill' => [
                    'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                    'startColor' => [
                        'rgb' => 'E0E0E0',
                    ],
                ],
            ];
            
            $sheet->getStyle('A1:' . \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex(count($encabezados)) . '1')->applyFromArray($styleArray);
            
            // Configurar cabeceras para descarga
            header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
            header('Content-Disposition: attachment;filename="' . $nombre_archivo . '.xlsx"');
            header('Cache-Control: max-age=0');
            
            $writer = \PhpOffice\PhpSpreadsheet\IOFactory::createWriter($spreadsheet, 'Xlsx');
            $writer->save('php://output');
            exit;
        } else {
            // Si PhpSpreadsheet no está disponible, redirigir a CSV
            header('Location: importacion_exportacion.php?accion=exportar&tipo_documento=' . $tipo_documento . '&formato=csv&fecha_desde=' . $fecha_desde . '&fecha_hasta=' . $fecha_hasta . '&usuario_id=' . $usuario_id . '&estado_id=' . $estado_id);
            exit;
        }
    }
}

// Incluir el header
require_once 'includes/header.php';
?>

<style>
    /* Estilos personalizados para importación/exportación */
    .tab-container {
        margin-bottom: 20px;
    }
    
    .tab-navigation {
        display: flex;
        border-bottom: 1px solid #ddd;
        margin-bottom: 20px;
    }
    
    .tab-link {
        padding: 10px 20px;
        border: 1px solid transparent;
        border-radius: 4px 4px 0 0;
        margin-right: 5px;
        cursor: pointer;
        transition: all 0.3s;
        font-weight: 500;
        text-decoration: none;
        color: #607d8b;
    }
    
    .tab-link:hover {
        background-color: #f8f9fa;
        border-color: #ddd #ddd transparent;
    }
    
    .tab-link.active {
        background-color: white;
        border-color: #ddd #ddd white;
        color: #1e88e5;
    }
    
    .import-export-card {
        background-color: white;
        border-radius: 8px;
        box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        padding: 20px;
        margin-bottom: 20px;
    }
    
    .import-steps {
        margin-bottom: 20px;
    }
    
    .step {
        display: flex;
        margin-bottom: 15px;
    }
    
    .step-number {
        width: 30px;
        height: 30px;
        border-radius: 50%;
        background-color: #1e88e5;
        color: white;
        display: flex;
        align-items: center;
        justify-content: center;
        font-weight: 600;
        margin-right: 10px;
        flex-shrink: 0;
    }
    
    .step-content {
        flex: 1;
    }
    
    .step h4 {
        margin: 0 0 5px 0;
        font-size: 16px;
    }
    
    .step p {
        margin: 0;
        color: #6c757d;
        font-size: 14px;
    }
    
    .dropzone {
        border: 2px dashed #ddd;
        border-radius: 6px;
        padding: 30px;
        text-align: center;
        margin-bottom: 20px;
        transition: all 0.3s;
    }
    
    .dropzone:hover {
        border-color: #1e88e5;
    }
    
    .dropzone i {
        font-size: 48px;
        color: #1e88e5;
        margin-bottom: 10px;
    }
    
    .dropzone h3 {
        margin: 0 0 10px 0;
        font-size: 18px;
    }
    
    .dropzone p {
        margin: 0;
        color: #6c757d;
    }
    
    .templates-section {
        margin-top: 20px;
    }
    
    .template-card {
        display: flex;
        align-items: center;
        border: 1px solid #ddd;
        border-radius: 6px;
        padding: 15px;
        margin-bottom: 10px;
        transition: all 0.3s;
    }
    
    .template-card:hover {
        border-color: #1e88e5;
        box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
    }
    
    .template-icon {
        margin-right: 15px;
        font-size: 24px;
        color: #1e88e5;
    }
    
    .template-info {
        flex: 1;
    }
    
    .template-title {
        font-weight: 600;
        margin: 0 0 5px 0;
    }
    
    .template-description {
        margin: 0;
        color: #6c757d;
        font-size: 14px;
    }
    
    .filtros-container {
        display: flex;
        flex-wrap: wrap;
        gap: 10px;
        margin-bottom: 20px;
    }
    
    .filtro-item {
        flex: 1;
        min-width: 200px;
    }
    
    .resultado-importacion {
        margin-top: 20px;
        padding: 15px;
        border-radius: 6px;
    }
    
    .resultado-success {
        background-color: rgba(102, 187, 106, 0.1);
        border: 1px solid rgba(102, 187, 106, 0.3);
    }
    
    .resultado-warning {
        background-color: rgba(255, 167, 38, 0.1);
        border: 1px solid rgba(255, 167, 38, 0.3);
    }
    
    .resultado-danger {
        background-color: rgba(239, 83, 80, 0.1);
        border: 1px solid rgba(239, 83, 80, 0.3);
    }
    
    .formato-opcion {
        display: inline-block;
        padding: 8px 16px;
        border: 1px solid #ddd;
        border-radius: 4px;
        margin-right: 10px;
        cursor: pointer;
        transition: all 0.3s;
    }
    
    .formato-opcion:hover, .formato-opcion.active {
        border-color: #1e88e5;
        color: #1e88e5;
    }
    
    .formato-opcion i {
        margin-right: 5px;
    }
    
    @media (max-width: 768px) {
        .filtro-item {
            min-width: 100%;
        }
    }
</style>

<!-- Mensaje de error si existe -->
<?php if (isset($error_message)): ?>
<div class="alert alert-danger">
    <?php echo $error_message; ?>
</div>
<?php endif; ?>

<div class="card">
    <div class="panel-header">
        <h2 class="panel-title">Importación y Exportación de Datos</h2>
    </div>
    
    <!-- Pestañas de navegación -->
    <div class="tab-navigation">
        <a href="importacion_exportacion.php?accion=importar" class="tab-link <?php echo $accion === 'importar' ? 'active' : ''; ?>">
            <i class="material-icons" style="vertical-align: middle; font-size: 18px; margin-right: 5px;">cloud_upload</i>
            Importar Datos
        </a>
        <a href="importacion_exportacion.php?accion=exportar" class="tab-link <?php echo $accion === 'exportar' ? 'active' : ''; ?>">
            <i class="material-icons" style="vertical-align: middle; font-size: 18px; margin-right: 5px;">cloud_download</i>
            Exportar Datos
        </a>
    </div>
    
    <?php if ($accion === 'importar'): ?>
    <!-- Sección de Importación -->
    <div class="import-export-card">
        <h3>Importación de Datos</h3>
        <p>Carga archivos CSV o Excel para importar datos de rendiciones u órdenes de compra al sistema.</p>
        
        <div class="import-steps">
            <div class="step">
                <div class="step-number">1</div>
                <div class="step-content">
                    <h4>Descarga una plantilla</h4>
                    <p>Utiliza una de nuestras plantillas para asegurar que el formato sea correcto.</p>
                </div>
            </div>
            
            <div class="step">
                <div class="step-number">2</div>
                <div class="step-content">
                    <h4>Llena los datos</h4>
                    <p>Completa la información en la plantilla siguiendo las instrucciones.</p>
                </div>
            </div>
            
            <div class="step">
                <div class="step-number">3</div>
                <div class="step-content">
                    <h4>Sube el archivo</h4>
                    <p>Selecciona el tipo de documento y sube el archivo completado.</p>
                </div>
            </div>
        </div>
        
        <div class="templates-section">
            <h4>Plantillas Disponibles</h4>
            
            <div class="template-card">
                <div class="template-icon">
                    <i class="material-icons">receipt</i>
                </div>
                <div class="template-info">
                    <div class="template-title">Plantilla de Rendiciones</div>
                    <div class="template-description">Para importar múltiples rendiciones de gastos.</div>
                </div>
                <div>
                    <a href="templates/plantilla_rendiciones.xlsx" class="btn btn-primary btn-sm" download>
                        <i class="material-icons">download</i> XLSX
                    </a>
                    <a href="templates/plantilla_rendiciones.csv" class="btn btn-primary btn-sm" download>
                        <i class="material-icons">download</i> CSV
                    </a>
                </div>
            </div>
            
            <div class="template-card">
                <div class="template-icon">
                    <i class="material-icons">shopping_cart</i>
                </div>
                <div class="template-info">
                    <div class="template-title">Plantilla de Órdenes de Compra</div>
                    <div class="template-description">Para importar múltiples órdenes de compra.</div>
                </div>
                <div>
                    <a href="templates/plantilla_ordenes_compra.xlsx" class="btn btn-primary btn-sm" download>
                        <i class="material-icons">download</i> XLSX
                    </a>
                    <a href="templates/plantilla_ordenes_compra.csv" class="btn btn-primary btn-sm" download>
                        <i class="material-icons">download</i> CSV
                    </a>
                </div>
            </div>
        </div>
        
        <form method="POST" action="importacion_exportacion.php?accion=importar" enctype="multipart/form-data" style="margin-top: 30px;">
            <div class="form-group">
                <label for="tipo_documento" class="form-label">Tipo de Documento</label>
                <select id="tipo_documento" name="tipo_documento" class="form-control" required>
                    <option value="">Seleccione el tipo de documento</option>
                    <option value="rendicion">Rendiciones de Gastos</option>
                    <option value="orden_compra">Órdenes de Compra</option>
                </select>
            </div>
            
            <div class="dropzone" id="dropzone">
                <input type="file" name="archivo_importacion" id="archivo_importacion" style="display: none;" accept=".csv, .xlsx, .xls">
                <i class="material-icons">cloud_upload</i>
                <h3>Arrastra y suelta tu archivo aquí</h3>
                <p>o haz clic para seleccionar un archivo (CSV, XLSX, XLS)</p>
            </div>
            
            <div style="text-align: center;">
                <button type="submit" class="btn btn-primary" id="btn-importar">
                    <i class="material-icons">upload</i> Importar Datos
                </button>
            </div>
        </form>
        
        <?php if ($resultados_importacion): ?>
            <?php 
            $clase_resultado = 'resultado-success';
            if ($resultados_importacion['fallidos'] > 0 && $resultados_importacion['exitosos'] > 0) {
                $clase_resultado = 'resultado-warning';
            } elseif ($resultados_importacion['fallidos'] > 0 && $resultados_importacion['exitosos'] === 0) {
                $clase_resultado = 'resultado-danger';
            }
            ?>
            <div class="resultado-importacion <?php echo $clase_resultado; ?>">
                <h4>Resultado de Importación</h4>
                <p>
                    Se procesaron <strong><?php echo $resultados_importacion['total']; ?></strong> registros: 
                    <strong><?php echo $resultados_importacion['exitosos']; ?></strong> exitosos y 
                    <strong><?php echo $resultados_importacion['fallidos']; ?></strong> fallidos.
                </p>
                
                <?php if (!empty($resultados_importacion['errores'])): ?>
                    <h5 style="margin-top: 15px;">Errores Encontrados:</h5>
                    <ul>
                        <?php foreach ($resultados_importacion['errores'] as $error): ?>
                            <li><?php echo $error; ?></li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            </div>
        <?php endif; ?>
        
        <?php if (!empty($errores_importacion)): ?>
            <div class="resultado-importacion resultado-danger">
                <h4>Errores en el Archivo</h4>
                <p>Se encontraron errores que impidieron procesar el archivo:</p>
                <ul>
                    <?php foreach ($errores_importacion as $error): ?>
                        <li><?php echo $error; ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>
    </div>
    
    <?php elseif ($accion === 'exportar'): ?>
    <!-- Sección de Exportación -->
    <div class="import-export-card">
        <h3>Exportación de Datos</h3>
        <p>Exporta datos de rendiciones u órdenes de compra a archivos Excel o CSV para su análisis o respaldo.</p>
        
        <form method="GET" action="importacion_exportacion.php">
            <input type="hidden" name="accion" value="exportar">
            
            <div class="form-group">
                <label for="tipo_documento" class="form-label">Tipo de Documento</label>
                <select id="tipo_documento" name="tipo_documento" class="form-control" required>
                    <option value="">Seleccione el tipo de documento</option>
                    <option value="rendicion">Rendiciones de Gastos</option>
                    <option value="orden_compra">Órdenes de Compra</option>
                </select>
            </div>
            
            <div class="filtros-container">
                <div class="filtro-item">
                    <label for="fecha_desde" class="form-label">Fecha Desde</label>
                    <input type="date" id="fecha_desde" name="fecha_desde" class="form-control" 
                           value="<?php echo date('Y-01-01'); ?>">
                </div>
                
                <div class="filtro-item">
                    <label for="fecha_hasta" class="form-label">Fecha Hasta</label>
                    <input type="date" id="fecha_hasta" name="fecha_hasta" class="form-control" 
                           value="<?php echo date('Y-12-31'); ?>">
                </div>
                
                <?php if ($is_admin): ?>
                <div class="filtro-item">
                    <label for="usuario_id" class="form-label">Usuario</label>
                    <select id="usuario_id" name="usuario_id" class="form-control">
                        <option value="0">Todos los usuarios</option>
                        <?php foreach ($usuarios as $usuario): ?>
                            <option value="<?php echo $usuario['id']; ?>">
                                <?php echo htmlspecialchars($usuario['nombre'] . ' ' . $usuario['apellido']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <?php endif; ?>
                
                <div class="filtro-item">
                    <label for="estado_id" class="form-label">Estado</label>
                    <select id="estado_id" name="estado_id" class="form-control">
                        <option value="0">Todos los estados</option>
                        <?php foreach ($estados as $estado): ?>
                            <option value="<?php echo $estado['id']; ?>">
                                <?php echo htmlspecialchars($estado['nombre']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>
            
            <div class="form-group">
                <label class="form-label">Formato de Exportación</label>
                <div>
                    <label class="formato-opcion active">
                        <input type="radio" name="formato" value="xlsx" checked style="display: none;">
                        <i class="material-icons">table_chart</i> Excel (.xlsx)
                    </label>
                    
                    <label class="formato-opcion">
                        <input type="radio" name="formato" value="csv" style="display: none;">
                        <i class="material-icons">description</i> CSV (.csv)
                    </label>
                </div>
            </div>
            
            <div style="text-align: center; margin-top: 20px;">
                <button type="submit" class="btn btn-primary">
                    <i class="material-icons">download</i> Exportar Datos
                </button>
            </div>
        </form>
    </div>
    <?php endif; ?>
</div>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Configurar dropzone para importación
        const dropzone = document.getElementById('dropzone');
        const fileInput = document.getElementById('archivo_importacion');
        const btnImportar = document.getElementById('btn-importar');
        
        if (dropzone && fileInput) {
            // Prevenir comportamiento predeterminado de arrastrar
            ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
                dropzone.addEventListener(eventName, preventDefaults, false);
            });
            
            function preventDefaults(e) {
                e.preventDefault();
                e.stopPropagation();
            }
            
            // Efectos visuales al arrastrar
            ['dragenter', 'dragover'].forEach(eventName => {
                dropzone.addEventListener(eventName, highlight, false);
            });
            
            ['dragleave', 'drop'].forEach(eventName => {
                dropzone.addEventListener(eventName, unhighlight, false);
            });
            
            function highlight() {
                dropzone.style.borderColor = '#1e88e5';
                dropzone.style.backgroundColor = 'rgba(30, 136, 229, 0.05)';
            }
            
            function unhighlight() {
                dropzone.style.borderColor = '#ddd';
                dropzone.style.backgroundColor = '';
            }
            
            // Manejar cuando se suelta un archivo
            dropzone.addEventListener('drop', handleDrop, false);
            
            function handleDrop(e) {
                const dt = e.dataTransfer;
                const files = dt.files;
                
                if (files.length) {
                    fileInput.files = files;
                    updateFileName(files[0].name);
                }
            }
            
            // Manejar clic en el dropzone
            dropzone.addEventListener('click', function() {
                fileInput.click();
            });
            
            // Actualizar nombre del archivo cuando se selecciona
            fileInput.addEventListener('change', function() {
                if (this.files.length) {
                    updateFileName(this.files[0].name);
                }
            });
            
            function updateFileName(name) {
                const nameElement = dropzone.querySelector('h3');
                const infoElement = dropzone.querySelector('p');
                
                nameElement.textContent = name;
                infoElement.textContent = 'Archivo seleccionado';
                
                // Cambiar estilo para indicar que se ha seleccionado un archivo
                dropzone.style.borderColor = '#66bb6a';
                dropzone.style.backgroundColor = 'rgba(102, 187, 106, 0.05)';
                dropzone.querySelector('i').textContent = 'check_circle';
                dropzone.querySelector('i').style.color = '#66bb6a';
            }
        }
        
        // Configurar opciones de formato para exportación
        const formatoOpciones = document.querySelectorAll('.formato-opcion');
        
        formatoOpciones.forEach(opcion => {
            opcion.addEventListener('click', function() {
                // Quitar clase active de todas las opciones
                formatoOpciones.forEach(op => op.classList.remove('active'));
                
                // Marcar esta opción como activa
                this.classList.add('active');
                
                // Marcar el radio button
                const radio = this.querySelector('input[type="radio"]');
                if (radio) {
                    radio.checked = true;
                }
            });
        });
    });
</script>

<?php
// Incluir el footer
require_once 'includes/footer.php';
?>