<?php
/**
 * Gestión de rendiciones de gastos
 * CRUD completo para rendiciones con soporte para subir archivos
 */

// Definir el título de la página
$page_title = 'Gestión de Rendiciones';

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Incluir archivos de configuración y funciones
require_once 'includes/config.php';
require_once 'includes/functions.php';

// Verificar sesión
verificarSesion();
$user_id = $_SESSION['user_id'];
$user_rol = $_SESSION['user_rol'] ?? '';
$user_nombre = $_SESSION['user_nombre'] ?? '';
$user_apellido = $_SESSION['user_apellido'] ?? '';
$is_admin = ($user_rol === 'admin' || $user_rol === 'super_admin');

// Determinar la acción a realizar
$action = $_GET['action'] ?? 'list';
$id = isset($_GET['id']) ? intval($_GET['id']) : 0;

// Conexión a la base de datos
try {
    $conn = new PDO("mysql:host=$db_host;dbname=$db_name;charset=utf8mb4", $db_user, $db_pass, $db_options);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // Obtener categorías de gastos
    $stmt = $conn->prepare("SELECT id, CONCAT(codigo, ' * ', nombre) as nombre FROM categorias_gastos WHERE activo = 1 ORDER BY codigo");
    $stmt->execute();
    $categorias = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Obtener estados
    $estados = obtenerEstados($conn);
    $estados_map = [];
    foreach ($estados as $estado) {
        $estados_map[$estado['id']] = [
            'nombre' => $estado['nombre'],
            'color' => $estado['color']
        ];
    }
    
    // Procesar formulario de creación o edición (incluyendo archivos)
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($action === 'new' || $action === 'edit')) {
        // Datos generales de la rendición
        $descripcion = trim($_POST['descripcion']);
        $area_origen = trim($_POST['area_origen']);
        $fecha = $_POST['fecha'];
        $tipo_rendicion = $_POST['tipo_rendicion'] ?? 'propia';
        $tercero_id = null;
        
        // Procesar usuario tercero si corresponde
        if ($tipo_rendicion === 'tercero') {
            if ($_POST['tercero_id'] === 'nuevo') {
                // Crear nuevo usuario tercero
                $stmt = $conn->prepare("
                    INSERT INTO users (rut, nombre, apellido, cargo, email, password, rol, activo, created_at, updated_at)
                    VALUES (:rut, :nombre, :apellido, :cargo, :email, :password, 'usuario', 1, NOW(), NOW())
                ");
                
                $email = strtolower(str_replace(' ', '.', $_POST['tercero_nombre'])) . '@santavictoria.cl';
                $password = password_hash('123456', PASSWORD_DEFAULT);
                
                $stmt->bindParam(':rut', $_POST['tercero_rut']);
                $stmt->bindParam(':nombre', $_POST['tercero_nombre']);
                $stmt->bindParam(':apellido', $_POST['tercero_apellido']);
                $stmt->bindParam(':cargo', $_POST['tercero_cargo']);
                $stmt->bindParam(':email', $email);
                $stmt->bindParam(':password', $password);
                
                $stmt->execute();
                $tercero_id = $conn->lastInsertId();
            } else {
                $tercero_id = intval($_POST['tercero_id']);
            }
        }
        
        $lineas = isset($_POST['categoria']) ? $_POST['categoria'] : [];
        $montos = isset($_POST['monto']) ? $_POST['monto'] : [];
        $detalles = isset($_POST['detalle']) ? $_POST['detalle'] : [];
        $tipos_documento = isset($_POST['tipo_documento']) ? $_POST['tipo_documento'] : [];
        $numeros_documento = isset($_POST['numero_documento']) ? $_POST['numero_documento'] : [];
        $folios_factura = isset($_POST['folio_factura']) ? $_POST['folio_factura'] : [];
        $proveedores_factura = isset($_POST['proveedor_factura']) ? $_POST['proveedor_factura'] : [];
        $ruts_proveedor = isset($_POST['rut_proveedor']) ? $_POST['rut_proveedor'] : [];
        $fechas_documento = isset($_POST['fecha_documento']) ? $_POST['fecha_documento'] : [];
        $fondo_a_rendir = isset($_POST['fondo_a_rendir']) ? floatval($_POST['fondo_a_rendir']) : 0;
        $archivos = isset($_FILES['soportes']) ? $_FILES['soportes'] : null;
        
        // Validación básica
        $errores = [];
        
        if (empty($descripcion)) {
            $errores[] = "La descripción es obligatoria";
        }
        
        if (empty($fecha)) {
            $errores[] = "La fecha es obligatoria";
        }
        
        if (empty($lineas) || count($lineas) < 1) {
            $errores[] = "Debe agregar al menos una línea de gasto";
        }
        
        if (empty($area_origen)) {
            $errores[] = "El área de origen es obligatoria";
        }
        
        if ($tipo_rendicion === 'tercero' && empty($tercero_id)) {
            $errores[] = "Debe seleccionar o crear un usuario tercero";
        }
        
        // Verificar montos
        $monto_total = 0;
        foreach ($montos as $monto) {
            if (!is_numeric($monto) || $monto <= 0) {
                $errores[] = "Los montos deben ser números positivos";
                break;
            }
            $monto_total += floatval($monto);
        }
        
        // Validar archivos (si se subieron)
        $archivos_subidos = [];
        if ($archivos && !empty($archivos['name'][0])) {
            $tipos_permitidos = ['image/jpeg', 'image/png', 'application/pdf'];
            $tamano_maximo = 10 * 1024 * 1024; // 5 MB
            
            foreach ($archivos['name'] as $index => $nombre) {
                if ($archivos['error'][$index] === UPLOAD_ERR_OK) {
                    $tipo = $archivos['type'][$index];
                    $tamano = $archivos['size'][$index];
                    $tmp_name = $archivos['tmp_name'][$index];
                    
                    // Validar tipo de archivo
                    if (!in_array($tipo, $tipos_permitidos)) {
                        $errores[] = "El archivo '$nombre' no es un tipo permitido (solo JPEG, PNG o PDF)";
                        continue;
                    }
                    
                    // Validar tamaño
                    if ($tamano > $tamano_maximo) {
                        $errores[] = "El archivo '$nombre' excede el tamaño máximo de 5 MB";
                        continue;
                    }
                    
                    // Sanitizar nombre del archivo
                    $nombre_sanitizado = preg_replace('/[^A-Za-z0-9\-\_\.]/', '_', $nombre);
                    $archivos_subidos[] = [
                        'nombre' => $nombre_sanitizado,
                        'tmp_name' => $tmp_name,
                        'tipo' => $tipo
                    ];
                } elseif ($archivos['error'][$index] !== UPLOAD_ERR_NO_FILE) {
                    $errores[] = "Error al subir el archivo '$nombre': " . $archivos['error'][$index];
                }
            }
        }
        
        // Si no hay errores, proceder a guardar
        if (empty($errores)) {
            $conn->beginTransaction();
            
            try {
                // Verificar duplicados antes de insertar
                if ($action === 'new') {
                    $stmt = $conn->prepare("
                        SELECT id FROM rendiciones 
                        WHERE user_id = :user_id 
                        AND fecha = :fecha 
                        AND monto_total = :monto_total 
                        AND descripcion = :descripcion 
                        AND fecha_creacion >= DATE_SUB(NOW(), INTERVAL 1 HOUR)
                    ");
                    
                    $stmt->bindParam(':user_id', $_SESSION['user_id']);
                    $stmt->bindParam(':fecha', $fecha);
                    $stmt->bindParam(':monto_total', $monto_total);
                    $stmt->bindParam(':descripcion', $descripcion);
                    $stmt->execute();
                    
                    if ($stmt->rowCount() > 0) {
                        throw new Exception("Ya existe una rendición similar creada en la última hora. Por favor, verifique los datos.");
                    }
                }

                // Si es una edición, verificar permisos y plazo de 24 horas
                if ($action === 'edit' && $id > 0) {
                    // Obtener la rendición actual
                    $stmt = $conn->prepare("SELECT * FROM rendiciones WHERE id = :id");
                    $stmt->bindParam(':id', $id);
                    $stmt->execute();
                    $rendicion = $stmt->fetch();
                    
                    // Verificar que la rendición exista
                    if (!$rendicion) {
                        throw new Exception("La rendición no existe");
                    }
                    
                    // Verificar que sea el propietario o administrador
                    if ($rendicion['user_id'] != $user_id && !$is_admin) {
                        throw new Exception("No tiene permisos para editar esta rendición");
                    }
                    
                    // Verificar estado (solo se puede editar si está en estado Enviada o Rechazada)
                    if ($rendicion['estado_id'] != 1 && $rendicion['estado_id'] != 4) {
                        throw new Exception("No se puede editar una rendición que ya ha sido aprobada o está en revisión");
                    }
                    
                    // Verificar plazo de 24 horas (excepto para administradores)
                    if (!$is_admin && !estaEnTiempoEdicion($rendicion['fecha_creacion'])) {
                        throw new Exception("Solo puede editar rendiciones dentro de las 24 horas posteriores a su creación");
                    }
                    
                    // Actualizar la rendición
                    $sql = "UPDATE rendiciones 
                    SET descripcion = :descripcion, 
                        area_origen = :area_origen,
                        fecha = :fecha, 
                        monto_total = :monto_total, 
                        fondo_a_rendir = :fondo_a_rendir,
                        tercero_id = :tercero_id,
                        ultima_modificacion = NOW()
                    WHERE id = :id";
                    
                    $stmt = $conn->prepare($sql);
                    $stmt->bindParam(':descripcion', $descripcion);
                    $stmt->bindParam(':area_origen', $area_origen);
                    $stmt->bindParam(':fecha', $fecha);
                    $stmt->bindParam(':monto_total', $monto_total);
                    $stmt->bindParam(':fondo_a_rendir', $fondo_a_rendir);
                    $stmt->bindParam(':tercero_id', $tercero_id);
                    $stmt->bindParam(':id', $id);
                    $stmt->execute();
                    
                    // Eliminar líneas anteriores
                    $stmt = $conn->prepare("DELETE FROM rendicion_categoria WHERE rendicion_id = :id");
                    $stmt->bindParam(':id', $id);
                    $stmt->execute();
                    
                    // Registrar actividad
                    registrarActividad($conn, $user_id, 'editar', 'Edición de rendición', $id, 'rendicion');
                    
                    $mensaje = "Rendición actualizada correctamente";
                } else {
                    // Calcular monto total
                    $monto_total = 0;
                    foreach ($montos as $monto) {
                        $monto_total += floatval($monto);
                    }
                    
                    // Crear nueva rendición
                    $sql = "INSERT INTO rendiciones (user_id, descripcion, area_origen, fecha, monto_total, fondo_a_rendir, tercero_id, estado_id, fecha_creacion, ultima_modificacion) 
                            VALUES (:user_id, :descripcion, :area_origen, :fecha, :monto_total, :fondo_a_rendir, :tercero_id, 1, NOW(), NOW())";
                    $stmt = $conn->prepare($sql);
                    $stmt->bindParam(':user_id', $_SESSION['user_id']);
                    $stmt->bindParam(':descripcion', $descripcion);
                    $stmt->bindParam(':area_origen', $area_origen);
                    $stmt->bindParam(':fecha', $fecha);
                    $stmt->bindParam(':monto_total', $monto_total);
                    $stmt->bindParam(':fondo_a_rendir', $fondo_a_rendir);
                    $stmt->bindParam(':tercero_id', $tercero_id);
                    $stmt->execute();
                    
                    $id = $conn->lastInsertId();
                    
                    // Registrar actividad
                    registrarActividad($conn, $user_id, 'crear', 'Creación de rendición', $id, 'rendicion');
                    
                    $mensaje = "Rendición creada correctamente";
                }
                
                // Insertar líneas de detalle
                foreach ($lineas as $index => $categoria_id) {
                    if (empty($categoria_id) || !isset($montos[$index])) continue;
                    
                    $stmt = $conn->prepare("
                    INSERT INTO rendicion_categoria 
                    (rendicion_id, categoria_id, monto, detalle, tipo_documento, numero_documento, 
                     folio_factura, proveedor_factura, rut_proveedor, fecha_documento) 
                    VALUES (:rendicion_id, :categoria_id, :monto, :detalle, :tipo_documento, :numero_documento,
                            :folio_factura, :proveedor_factura, :rut_proveedor, :fecha_documento)
                    ");
                    
                    $monto = floatval($montos[$index]);
                    $detalle = $detalles[$index] ?? '';
                    $tipo_documento = $tipos_documento[$index] ?? 'otro';
                    $numero_documento = $numeros_documento[$index] ?? '';
                    $folio_factura = ($tipo_documento == 'factura') ? ($folios_factura[$index] ?? '') : null;
                    $proveedor_factura = ($tipo_documento == 'factura') ? ($proveedores_factura[$index] ?? '') : null;
                    $rut_proveedor = ($tipo_documento == 'factura') ? ($ruts_proveedor[$index] ?? '') : null;
                    $fecha_documento = ($tipo_documento == 'factura' && !empty($fechas_documento[$index])) ? 
                                     $fechas_documento[$index] : null;
                    
                    $stmt->bindParam(':rendicion_id', $id);
                    $stmt->bindParam(':categoria_id', $categoria_id);
                    $stmt->bindParam(':monto', $monto);
                    $stmt->bindParam(':detalle', $detalle);
                    $stmt->bindParam(':tipo_documento', $tipo_documento);
                    $stmt->bindParam(':numero_documento', $numero_documento);
                    $stmt->bindParam(':folio_factura', $folio_factura);
                    $stmt->bindParam(':proveedor_factura', $proveedor_factura);
                    $stmt->bindParam(':rut_proveedor', $rut_proveedor);
                    $stmt->bindParam(':fecha_documento', $fecha_documento);
                    $stmt->execute();
                }
                
                // Procesar archivos subidos
                if (!empty($archivos_subidos)) {
                    // Crear directorio para la rendición si no existe
                    $carpeta_rendicion = "upload/rendicion_$id/";
                    if (!file_exists($carpeta_rendicion)) {
                        mkdir($carpeta_rendicion, 0777, true);
                    }
                    
                    foreach ($archivos_subidos as $archivo) {
                        $nombre_archivo = time() . '_' . $archivo['nombre'];
                        $ruta_destino = $carpeta_rendicion . $nombre_archivo;
                        
                        // Mover el archivo al directorio
                        if (!move_uploaded_file($archivo['tmp_name'], $ruta_destino)) {
                            throw new Exception("Error al mover el archivo '$nombre_archivo'");
                        }
                        
                        // Registrar el archivo en la base de datos
                        $stmt = $conn->prepare("
                            INSERT INTO rendicion_soportes 
                            (rendicion_id, nombre_archivo, ruta_archivo, fecha_subida) 
                            VALUES (:rendicion_id, :nombre_archivo, :ruta_archivo, NOW())
                        ");
                        $stmt->bindParam(':rendicion_id', $id);
                        $stmt->bindParam(':nombre_archivo', $archivo['nombre']);
                        $stmt->bindParam(':ruta_archivo', $ruta_destino);
                        $stmt->execute();
                    }
                }
                
                $conn->commit();
                
                // Redireccionar a la lista con mensaje de éxito
                $_SESSION['flash_message'] = $mensaje;
                $_SESSION['flash_type'] = 'success';
                header('Location: rendiciones.php');
                exit;
                
            } catch (Exception $e) {
                $conn->rollBack();
                $error_message = $e->getMessage();
            }
        } else {
            $error_message = implode("<br>", $errores);
        }
    }
    
    // Procesar eliminación de un archivo de soporte (solo para usuarios autorizados)
    if ($action === 'delete_file' && isset($_GET['file_id']) && ($is_admin || $rendicion['user_id'] == $user_id)) {
        $file_id = intval($_GET['file_id']);
        $is_ajax = isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest';
        
        try {
            // Obtener información del archivo
            $stmt = $conn->prepare("SELECT * FROM rendicion_soportes WHERE id = :file_id AND rendicion_id = :rendicion_id");
            $stmt->bindParam(':file_id', $file_id);
            $stmt->bindParam(':rendicion_id', $id);
            $stmt->execute();
            $soporte = $stmt->fetch();
            
            if (!$soporte) {
                throw new Exception("Archivo no encontrado");
            }
            
            // Verificar permisos
            $stmt = $conn->prepare("SELECT user_id, estado_id FROM rendiciones WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $rendicion = $stmt->fetch();
            
            if (!$rendicion || ($rendicion['user_id'] != $user_id && !$is_admin)) {
                throw new Exception("No tiene permisos para eliminar este archivo");
            }
            
            // Solo permitir eliminar si la rendición está en estado Enviada o Rechazada
            if ($rendicion['estado_id'] != 1 && $rendicion['estado_id'] != 4) {
                throw new Exception("No se puede eliminar archivos de una rendición que ya ha sido aprobada o está en revisión");
            }
            
            // Eliminar archivo del sistema de archivos
            if (file_exists($soporte['ruta_archivo'])) {
                unlink($soporte['ruta_archivo']);
            }
            
            // Eliminar registro de la base de datos
            $stmt = $conn->prepare("DELETE FROM rendicion_soportes WHERE id = :file_id");
            $stmt->bindParam(':file_id', $file_id);
            $stmt->execute();
            
            // Registrar actividad
            registrarActividad($conn, $user_id, 'eliminar', 'Eliminación de archivo de soporte de rendición', $id, 'rendicion');
            
            if ($is_ajax) {
                header('Content-Type: application/json');
                echo json_encode(['success' => true]);
                exit;
            } else {
            $_SESSION['flash_message'] = "Archivo eliminado correctamente";
            $_SESSION['flash_type'] = 'success';
            header("Location: rendiciones.php?action=view&id=$id");
            exit;
            }
            
        } catch (Exception $e) {
            if ($is_ajax) {
                header('Content-Type: application/json');
                echo json_encode(['success' => false, 'message' => $e->getMessage()]);
                exit;
            } else {
            $error_message = "Error al eliminar el archivo: " . $e->getMessage();
            }
        }
    }
    
    // Procesar aprobación o rechazo
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && $action === 'review' && $is_admin) {
        $decision = $_POST['decision'];
        $comentario = trim($_POST['comentario']);
        
        if ($decision != 'aprobar' && $decision != 'rechazar') {
            $error_message = "Acción no válida";
        } else {
            try {
                // Actualizar estado de la rendición
                $estado_id = ($decision === 'aprobar') ? 3 : 4; // 3=Aprobada, 4=Rechazada
                
                $stmt = $conn->prepare("
                    UPDATE rendiciones 
                    SET estado_id = :estado_id, 
                        revisor_id = :revisor_id, 
                        comentario_revision = :comentario,
                        fecha_revision = NOW()
                    WHERE id = :id
                ");
                
                $stmt->bindParam(':estado_id', $estado_id);
                $stmt->bindParam(':revisor_id', $user_id);
                $stmt->bindParam(':comentario', $comentario);
                $stmt->bindParam(':id', $id);
                $stmt->execute();
                
                // Registrar actividad
                $tipo_accion = ($decision === 'aprobar') ? 'aprobar' : 'rechazar';
                $descripcion = ($decision === 'aprobar') ? 'Aprobación de rendición' : 'Rechazo de rendición';
                registrarActividad($conn, $user_id, $tipo_accion, $descripcion, $id, 'rendicion');
                
                $mensaje = ($decision === 'aprobar') ? 
                    "Rendición aprobada correctamente" : 
                    "Rendición rechazada correctamente";
                
                $_SESSION['flash_message'] = $mensaje;
                $_SESSION['flash_type'] = 'success';
                header('Location: rendiciones.php');
                exit;
            } catch (Exception $e) {
                $error_message = "Error al procesar la solicitud: " . $e->getMessage();
            }
        }
    }
    
    // Acción para eliminar rendición (solo admin)
    if ($action === 'delete' && $is_admin && $id > 0) {
        try {
            // Obtener la rendición para verificar
            $stmt = $conn->prepare("SELECT * FROM rendiciones WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $rendicion = $stmt->fetch();
            
            if (!$rendicion) {
                throw new Exception("La rendición no existe");
            }
            
            // Verificar que no esté aprobada o en revisión
            if ($rendicion['estado_id'] == 2 || $rendicion['estado_id'] == 3 || $rendicion['estado_id'] == 5) {
                throw new Exception("No se puede eliminar una rendición que está en revisión, aprobada o pagada");
            }
            
            // Eliminar líneas de detalle
            $stmt = $conn->prepare("DELETE FROM rendicion_categoria WHERE rendicion_id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            
            // Eliminar rendición (los archivos y registros en rendicion_soportes se eliminan automáticamente por ON DELETE CASCADE)
            $stmt = $conn->prepare("DELETE FROM rendiciones WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            
            // Eliminar carpeta de archivos (por si acaso ON DELETE CASCADE no elimina archivos físicos)
            $carpeta_rendicion = "upload/rendicion_$id/";
            if (file_exists($carpeta_rendicion)) {
                array_map('unlink', glob("$carpeta_rendicion*.*"));
                rmdir($carpeta_rendicion);
            }
            
            // Registrar actividad
            registrarActividad($conn, $user_id, 'eliminar', 'Eliminación de rendición', $id, 'rendicion');
            
            $_SESSION['flash_message'] = "Rendición eliminada correctamente";
            $_SESSION['flash_type'] = 'success';
            header('Location: rendiciones.php');
            exit;
        } catch (Exception $e) {
            $error_message = "Error al eliminar la rendición: " . $e->getMessage();
        }
    }
} catch (PDOException $e) {
    $error_message = 'Error de conexión a la base de datos: ' . $e->getMessage();
}

// Añadir después de la conexión a la base de datos
if (isset($_GET['check_factura'])) {
    try {
        $folio = $_GET['folio'];
        $rendicion_id = isset($_GET['rendicion_id']) ? intval($_GET['rendicion_id']) : null;
        
        // Modificar la consulta para excluir la rendición actual
        $sql = "SELECT rc.*, r.descripcion as rendicion_descripcion, r.fecha as rendicion_fecha 
                FROM rendicion_categoria rc 
                JOIN rendiciones r ON rc.rendicion_id = r.id 
                WHERE rc.tipo_documento = 'factura' 
                AND rc.folio_factura = :folio";
        
        // Solo verificar duplicados en otras rendiciones
        if ($rendicion_id) {
            $sql .= " AND rc.rendicion_id != :rendicion_id";
        }
        
        $stmt = $conn->prepare($sql);
        $stmt->bindParam(':folio', $folio);
        if ($rendicion_id) {
            $stmt->bindParam(':rendicion_id', $rendicion_id);
        }
        
        $stmt->execute();
        $factura = $stmt->fetch(PDO::FETCH_ASSOC);
        
        header('Content-Type: application/json');
        echo json_encode([
            'exists' => $factura ? true : false,
            'data' => $factura,
            'is_editing' => !empty($rendicion_id)
        ]);
        exit;
    } catch (Exception $e) {
        header('Content-Type: application/json');
        echo json_encode([
            'error' => true,
            'message' => $e->getMessage()
        ]);
        exit;
    }
}

// Incluir el header
require_once 'includes/header.php';
?>

<!-- Mensaje de error si existe -->
<?php if (isset($error_message)): ?>
<div class="alert alert-danger">
    <?php echo $error_message; ?>
</div>
<?php endif; ?>

<?php
// Mostrar contenido según la acción
switch ($action) {
    case 'new':
        // Obtener lista de usuarios para el selector de terceros
        $stmt = $conn->prepare("
            SELECT id, nombre, apellido, rut, cargo 
            FROM users 
            WHERE activo = 1 
            ORDER BY nombre, apellido
        ");
        $stmt->execute();
        $usuarios = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Formulario para crear nueva rendición
        ?>
        <div class="card">
            <div class="panel-header">
                <h2 class="panel-title">Nueva Rendición de Gastos</h2>
                <a href="rendiciones.php" class="btn btn-primary btn-sm">
                    <i class="material-icons">arrow_back</i> Volver a la Lista
                </a>
            </div>
            
            <form method="POST" action="rendiciones.php?action=new" id="form-rendicion" enctype="multipart/form-data">
                <style>
                    /* Mejorar aspecto de los inputs */
                    .form-control:focus {
                        border-color: #1e88e5;
                        box-shadow: 0 0 0 0.15rem rgba(30, 136, 229, 0.25);
                    }
                    
                    /* Destacar campos requeridos */
                    .form-control:required {
                        border-left: 3px solid #1e88e5;
                    }
                    
                    /* Reducir padding para inputs de texto en formularios */
                    .input-sm {
                        padding: 6px 8px;
                        height: auto;
                    }

                    /* Layout de dos columnas */
                    .form-row {
                        display: flex;
                        flex-wrap: wrap;
                        margin: 0 -10px;
                    }

                    .form-col {
                        flex: 1;
                        min-width: 300px;
                        padding: 0 10px;
                    }

                    .form-group {
                        margin-bottom: 10px;
                    }

                    .form-group label {
                        margin-bottom: 3px;
                        font-weight: 500;
                    }

                    /* Estilos para las cards */
                    .info-card {
                        background: #f8f9fa;
                        border: 1px solid #dee2e6;
                        border-radius: 4px;
                        padding: 15px;
                        margin-bottom: 15px;
                    }

                    .info-card h4 {
                        margin: 0 0 10px 0;
                        font-size: 1rem;
                        color: #1e88e5;
                    }
                </style>

                <div class="form-row">
                    <div class="form-col">
                        <div class="info-card">
                            <h4>Información General</h4>
                            <div class="form-group">
                                <label for="descripcion">Descripción</label>
                                <input type="text" id="descripcion" name="descripcion" class="form-control input-sm" 
                                       placeholder="Descripción de la rendición" required autocomplete="off"
                                       value="<?php echo isset($rendicion['descripcion']) ? htmlspecialchars($rendicion['descripcion']) : ''; ?>">
                            </div>
                            <div class="form-row">
                                <div class="form-col">
                                    <div class="form-group">
                                        <label for="fecha">Fecha</label>
                                        <input type="date" id="fecha" name="fecha" class="form-control input-sm" required
                                               value="<?php echo isset($rendicion['fecha']) ? $rendicion['fecha'] : date('Y-m-d'); ?>">
                                    </div>
                                </div>
                                <div class="form-col">
                                    <div class="form-group">
                                        <label for="area_origen">Área</label>
                                        <select id="area_origen" name="area_origen" class="form-control input-sm" required>
                                            <option value="logistica" <?php echo isset($rendicion) && $rendicion['area_origen'] == 'logistica' ? 'selected' : ''; ?>>Logística</option>
                                            <option value="servicio_tecnico" <?php echo isset($rendicion) && $rendicion['area_origen'] == 'servicio_tecnico' ? 'selected' : ''; ?>>Servicio Técnico</option>
                                            <option value="marketing" <?php echo isset($rendicion) && $rendicion['area_origen'] == 'marketing' ? 'selected' : ''; ?>>Marketing</option>
                                            <option value="ventas" <?php echo isset($rendicion) && $rendicion['area_origen'] == 'ventas' ? 'selected' : ''; ?>>Ventas</option>
                                            <option value="comercial" <?php echo isset($rendicion) && $rendicion['area_origen'] == 'comercial' ? 'selected' : ''; ?>>Área Comercial</option>
                                            <option value="operaciones" <?php echo isset($rendicion) && $rendicion['area_origen'] == 'operaciones' ? 'selected' : ''; ?>>Operaciones</option>
                                            <option value="finanzas" <?php echo isset($rendicion) && $rendicion['area_origen'] == 'finanzas' ? 'selected' : ''; ?>>Finanzas</option>
                                            <option value="ecommerce" <?php echo isset($rendicion) && $rendicion['area_origen'] == 'ecommerce' ? 'selected' : ''; ?>>Ecommerce</option>
                                            <option value="barista" <?php echo isset($rendicion) && $rendicion['area_origen'] == 'barista' ? 'selected' : ''; ?>>Barista</option>
                                            <option value="otra" <?php echo isset($rendicion) && $rendicion['area_origen'] == 'otra' ? 'selected' : ''; ?>>Otra</option>
                                        </select>
                                    </div>
                                </div>
                            </div>
                            <div class="form-group">
                                <label for="fondo_a_rendir">Fondo a Rendir</label>
                                <input type="number" id="fondo_a_rendir" name="fondo_a_rendir" class="form-control input-sm" 
                                       min="0" step="1" placeholder="0" required autocomplete="off"
                                       value="<?php echo isset($rendicion['fondo_a_rendir']) ? htmlspecialchars($rendicion['fondo_a_rendir']) : '0'; ?>">
                            </div>
                        </div>
                    </div>

                    <div class="form-col">
                        <div class="info-card">
                            <h4>Información del Usuario</h4>
                            <div class="form-group">
                                <label for="tipo_rendicion">Tipo de Rendición</label>
                                <select id="tipo_rendicion" name="tipo_rendicion" class="form-control input-sm" onchange="toggleTerceroFields(this.value)" required>
                                    <option value="propia" <?php echo empty($rendicion['tercero_id']) ? 'selected' : ''; ?>>Esta rendición es mía</option>
                                    <option value="tercero" <?php echo !empty($rendicion['tercero_id']) ? 'selected' : ''; ?>>Esta rendición pertenece a un tercero</option>
                                </select>
                            </div>

                            <div id="tercero_fields" style="display: <?php echo !empty($rendicion['tercero_id']) ? 'block' : 'none'; ?>;">
                                <div class="form-group">
                                    <label for="tercero_select">Usuario</label>
                                    <select id="tercero_select" name="tercero_id" class="form-control input-sm" onchange="toggleNuevoTercero(this.value)">
                                        <option value="">Seleccionar usuario...</option>
                                        <option value="nuevo">+ Crear Nuevo Usuario</option>
                                        <?php foreach ($usuarios as $usuario): ?>
                                            <option value="<?php echo $usuario['id']; ?>" 
                                                    <?php echo (isset($rendicion['tercero_id']) && $rendicion['tercero_id'] == $usuario['id']) ? 'selected' : ''; ?>>
                                                <?php echo htmlspecialchars($usuario['nombre'] . ' ' . $usuario['apellido'] . ' - ' . $usuario['rut']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>

                                <div id="nuevo_tercero_fields" style="display: none;">
                                    <div class="form-row">
                                        <div class="form-col">
                                            <div class="form-group">
                                                <label for="tercero_nombre">Nombre</label>
                                                <input type="text" id="tercero_nombre" name="tercero_nombre" class="form-control input-sm">
                                            </div>
                                        </div>
                                        <div class="form-col">
                                            <div class="form-group">
                                                <label for="tercero_apellido">Apellido</label>
                                                <input type="text" id="tercero_apellido" name="tercero_apellido" class="form-control input-sm">
                                            </div>
                                        </div>
                                    </div>
                                    <div class="form-row">
                                        <div class="form-col">
                                            <div class="form-group">
                                                <label for="tercero_rut">RUT</label>
                                                <input type="text" id="tercero_rut" name="tercero_rut" class="form-control input-sm">
                                            </div>
                                        </div>
                                        <div class="form-col">
                                            <div class="form-group">
                                                <label for="tercero_cargo">Cargo</label>
                                                <input type="text" id="tercero_cargo" name="tercero_cargo" class="form-control input-sm">
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="card" style="margin-top: 20px;">
                    <div class="panel-header">
                        <h3 class="panel-title">Detalle de Gastos</h3>
                        <button type="button" id="btn-agregar-linea" class="btn btn-success btn-sm">
                            <i class="material-icons">add</i> Agregar Línea
                        </button>
                    </div>
                    
                    <div class="table-responsive">
                        <table>
                            <thead>
                                <tr>
                                    <th>Categoría</th>
                                    <th>Monto</th>
                                    <th>Detalle</th>
                                    <th>Tipo Doc.</th>
                                    <th>N° Documento</th>
                                    <th>Acciones</th>
                                </tr>
                            </thead>
                            <tbody id="lineas-container">
                                <!-- Primera línea por defecto -->
                                <tr>
                                    <td>
                                        <select name="categoria[]" class="form-control" required>
                                            <option value="">Seleccione una categoría</option>
                                            <?php foreach ($categorias as $categoria): ?>
                                            <option value="<?php echo $categoria['id']; ?>">
                                                <?php echo htmlspecialchars($categoria['nombre']); ?>
                                            </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </td>
                                    <td>
                                        <input type="number" name="monto[]" class="form-control monto-input input-sm" 
                                               min="1" step="1" placeholder="0" required autocomplete="off">
                                    </td>
                                    <td>
                                        <input type="text" name="detalle[]" class="form-control input-sm" 
                                               placeholder="Descripción específica del gasto" autocomplete="off">
                                    </td>
                                    
                                    <td>
                                        <select name="tipo_documento[]" class="form-control input-sm tipo-documento-select" onchange="toggleFacturaFields(this)">
                                            <option value="boleta">Boleta</option>
                                            <option value="factura">Factura</option>
                                            <option value="recibo">Recibo</option>
                                            <option value="cotizacion">Cotización</option>
                                            <option value="otro">Otro</option>
                                        </select>
                                    </td>
                                    <td>
                                        <div class="documento-fields">
                                            <div class="numero-documento-field">
                                                <input type="text" name="numero_documento[]" class="form-control input-sm" placeholder="N° del documento">
                                            </div>
                                            <!-- Campos adicionales para factura -->
                                            <div class="factura-fields" style="display: none;">
                                                <input type="text" name="folio_factura[]" class="form-control input-sm mt-2" placeholder="Folio factura">
                                                <input type="text" name="proveedor_factura[]" class="form-control input-sm mt-2" placeholder="Nombre proveedor">
                                                <input type="text" name="rut_proveedor[]" class="form-control input-sm mt-2" placeholder="RUT proveedor">
                                                <input type="date" name="fecha_documento[]" class="form-control input-sm mt-2">
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <button type="button" class="btn btn-danger btn-sm" onclick="eliminarLinea(this.parentNode.parentNode)">
                                            <i class="material-icons">delete</i>
                                        </button>
                                    </td>
                                </tr>
                            </tbody>
                            <tfoot>
                                <tr>
                                    <td colspan="6" style="text-align: right; padding: 15px;">
                                        <button type="button" id="btn-agregar-linea-bottom" class="btn btn-success btn-sm">
                                            <i class="material-icons">add</i> Agregar Línea
                                        </button>
                                    </td>
                                </tr>
                                <tr>
                                    <th colspan="1" style="text-align: right;">Total Rendido:</th>
                                    <th id="total-rendicion">$0</th>
                                    <th colspan="4"></th>
                                </tr>
                                <tr>
                                    <th colspan="1" style="text-align: right;">Fondo a Rendir:</th>
                                    <th id="fondo-total">$0</th>
                                    <th colspan="4"></th>
                                </tr>
                                <tr>
                                    <th colspan="1" style="text-align: right;">Saldo Pendiente:</th>
                                    <th id="saldo-pendiente">$0</th>
                                    <th colspan="4"></th>
                                </tr>
                            </tfoot>
                        </table>
                    </div>
                </div>
                
                <!-- Nueva sección para subir archivos -->
                <div class="card" style="margin-top: 20px;">
                    <div class="panel-header">
                        <h3 class="panel-title">Archivos de Soporte</h3>
                    </div>
                    
                    <div class="form-group">
                        <label for="soportes" class="form-label">Subir Archivos (JPEG, PNG, PDF - Máx. 5MB)</label>
                        <input type="file" id="soportes" name="soportes[]" class="form-control" multiple
                               accept="image/jpeg,image/png,application/pdf">
                    </div>
                </div>
                
                <div style="margin-top: 20px; text-align: center;">
                    <button type="submit" class="btn btn-primary">
                        <i class="material-icons">save</i> Guardar Rendición
                    </button>
                </div>
            </form>
        </div>
        
        <!-- Template oculto para nuevas líneas -->
        <template id="linea-template">
            <tr>
                <td>
                    <select name="categoria[]" class="form-control" required>
                        <option value="">Seleccione una categoría</option>
                        <?php foreach ($categorias as $categoria): ?>
                        <option value="<?php echo $categoria['id']; ?>">
                            <?php echo htmlspecialchars($categoria['nombre']); ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </td>
                <td>
                    <input type="number" name="monto[]" class="form-control monto-input" 
                           min="1" step="1" placeholder="0" required>
                </td>
                <td>
                    <input type="text" name="detalle[]" class="form-control" 
                           placeholder="Descripción específica del gasto">
                </td>
                <td>
                    <select name="tipo_documento[]" class="form-control input-sm tipo-documento-select" onchange="toggleFacturaFields(this)">
                        <option value="boleta">Boleta</option>
                        <option value="factura">Factura</option>
                        <option value="recibo">Recibo</option>
                        <option value="cotizacion">Cotización</option>
                        <option value="otro">Otro</option>
                    </select>
                </td>
                <td>
                    <div class="documento-fields">
                        <div class="numero-documento-field">
                            <input type="text" name="numero_documento[]" class="form-control input-sm" placeholder="N° del documento">
                        </div>
                        <!-- Campos adicionales para factura -->
                        <div class="factura-fields" style="display: none;">
                            <input type="text" name="folio_factura[]" class="form-control input-sm mt-2" placeholder="Folio factura">
                            <input type="text" name="proveedor_factura[]" class="form-control input-sm mt-2" placeholder="Nombre proveedor">
                            <input type="text" name="rut_proveedor[]" class="form-control input-sm mt-2" placeholder="RUT proveedor">
                            <input type="date" name="fecha_documento[]" class="form-control input-sm mt-2">
                        </div>
                    </div>
                </td>
                <td>
                    <button type="button" class="btn btn-danger btn-sm" onclick="eliminarLinea(this.parentNode.parentNode)">
                        <i class="material-icons">delete</i>
                    </button>
                </td>
            </tr>
        </template>
        <?php
        break;
        
    case 'edit':
        // Formulario para editar rendición existente
        try {
            // Obtener datos de la rendición
            $stmt = $conn->prepare("
                SELECT r.*, 
                       u.nombre, u.apellido, u.rut, u.cargo,
                       tercero.id as tercero_id, 
                       tercero.nombre as tercero_nombre, 
                       tercero.apellido as tercero_apellido, 
                       tercero.rut as tercero_rut, 
                       tercero.cargo as tercero_cargo,
                       e.nombre as estado_nombre
                FROM rendiciones r
                JOIN users u ON r.user_id = u.id
                LEFT JOIN users tercero ON r.tercero_id = tercero.id
                LEFT JOIN estados e ON r.estado_id = e.id
                WHERE r.id = :id " . (!$is_admin ? "AND r.user_id = :user_id" : "")
            );
            
            $stmt->bindParam(':id', $id);
            if (!$is_admin) {
                $stmt->bindParam(':user_id', $user_id);
            }
            
            $stmt->execute();
            $rendicion = $stmt->fetch();
            
            if (!$rendicion) {
                throw new Exception("Rendición no encontrada o sin permisos para editarla");
            }

            // Verificar estado (solo se puede editar si está en estado Enviada o Rechazada)
            if ($rendicion['estado_id'] != 1 && $rendicion['estado_id'] != 4) {
                throw new Exception("No se puede editar una rendición que ya ha sido aprobada o está en revisión");
            }

            // Obtener lista de usuarios para el selector de terceros
            $stmt = $conn->prepare("
                SELECT id, nombre, apellido, rut, cargo 
                FROM users 
                WHERE activo = 1 
                ORDER BY nombre, apellido
            ");
            $stmt->execute();
            $usuarios = $stmt->fetchAll(PDO::FETCH_ASSOC);

            // Obtener líneas de detalle
            $stmt = $conn->prepare("
                SELECT rc.*, cg.nombre as categoria_nombre
                FROM rendicion_categoria rc
                JOIN categorias_gastos cg ON rc.categoria_id = cg.id
                WHERE rc.rendicion_id = :id
                ORDER BY rc.id
            ");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $lineas = $stmt->fetchAll(PDO::FETCH_ASSOC);

            // Obtener archivos de soporte
            $stmt = $conn->prepare("SELECT * FROM rendicion_soportes WHERE rendicion_id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $soportes = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            ?>
            <div class="card">
                <div class="panel-header">
                    <h2 class="panel-title">Editar Rendición de Gastos</h2>
                    <a href="rendiciones.php" class="btn btn-primary btn-sm">
                        <i class="material-icons">arrow_back</i> Volver a la Lista
                    </a>
                </div>
                
                <form method="POST" action="rendiciones.php?action=edit&id=<?php echo $id; ?>" id="form-rendicion" enctype="multipart/form-data">
                    <input type="hidden" name="id" value="<?php echo $id; ?>">
                    <input type="hidden" name="action" value="edit">
                    <div class="form-row">
                        <div class="form-col">
                            <div class="info-card">
                                <h4>Información General</h4>
                                <div class="form-group">
                                    <label for="descripcion">Descripción</label>
                                    <input type="text" id="descripcion" name="descripcion" class="form-control input-sm" 
                                           placeholder="Descripción de la rendición" required autocomplete="off"
                                           value="<?php echo htmlspecialchars($rendicion['descripcion']); ?>">
                                </div>
                                <div class="form-row">
                                    <div class="form-col">
                                        <div class="form-group">
                                            <label for="fecha">Fecha</label>
                                            <input type="date" id="fecha" name="fecha" class="form-control input-sm" required
                                                   value="<?php echo $rendicion['fecha']; ?>">
                                        </div>
                                    </div>
                                    <div class="form-col">
                                        <div class="form-group">
                                            <label for="area_origen">Área</label>
                                            <select id="area_origen" name="area_origen" class="form-control input-sm" required>
                                                <option value="logistica" <?php echo $rendicion['area_origen'] == 'logistica' ? 'selected' : ''; ?>>Logística</option>
                                                <option value="servicio_tecnico" <?php echo $rendicion['area_origen'] == 'servicio_tecnico' ? 'selected' : ''; ?>>Servicio Técnico</option>
                                                <option value="marketing" <?php echo $rendicion['area_origen'] == 'marketing' ? 'selected' : ''; ?>>Marketing</option>
                                                <option value="ventas" <?php echo $rendicion['area_origen'] == 'ventas' ? 'selected' : ''; ?>>Ventas</option>
                                                <option value="comercial" <?php echo $rendicion['area_origen'] == 'comercial' ? 'selected' : ''; ?>>Área Comercial</option>
                                                <option value="operaciones" <?php echo $rendicion['area_origen'] == 'operaciones' ? 'selected' : ''; ?>>Operaciones</option>
                                                <option value="finanzas" <?php echo $rendicion['area_origen'] == 'finanzas' ? 'selected' : ''; ?>>Finanzas</option>
                                                <option value="ecommerce" <?php echo $rendicion['area_origen'] == 'ecommerce' ? 'selected' : ''; ?>>Ecommerce</option>
                                                <option value="barista" <?php echo $rendicion['area_origen'] == 'barista' ? 'selected' : ''; ?>>Barista</option>
                                                <option value="otra" <?php echo $rendicion['area_origen'] == 'otra' ? 'selected' : ''; ?>>Otra</option>
                                            </select>
                                        </div>
                                    </div>
                                </div>
                                <div class="form-group">
                                    <label for="fondo_a_rendir">Fondo a Rendir</label>
                                    <input type="number" id="fondo_a_rendir" name="fondo_a_rendir" class="form-control input-sm" 
                                           min="0" step="1" placeholder="0" required autocomplete="off"
                                           value="<?php echo htmlspecialchars($rendicion['fondo_a_rendir']); ?>">
                                </div>
                            </div>
                        </div>

                        <div class="form-col">
                            <div class="info-card">
                                <h4>Información del Usuario</h4>
                                <div class="form-group">
                                    <label for="tipo_rendicion">Tipo de Rendición</label>
                                    <select id="tipo_rendicion" name="tipo_rendicion" class="form-control input-sm" onchange="toggleTerceroFields(this.value)" required>
                                        <option value="propia" <?php echo empty($rendicion['tercero_id']) ? 'selected' : ''; ?>>Esta rendición es mía</option>
                                        <option value="tercero" <?php echo !empty($rendicion['tercero_id']) ? 'selected' : ''; ?>>Esta rendición pertenece a un tercero</option>
                                    </select>
                                </div>

                                <div id="tercero_fields" style="display: <?php echo !empty($rendicion['tercero_id']) ? 'block' : 'none'; ?>;">
                                    <div class="form-group">
                                        <label for="tercero_select">Usuario</label>
                                        <select id="tercero_select" name="tercero_id" class="form-control input-sm" onchange="toggleNuevoTercero(this.value)">
                                            <option value="">Seleccionar usuario...</option>
                                            <option value="nuevo">+ Crear Nuevo Usuario</option>
                                            <?php foreach ($usuarios as $usuario): ?>
                                                <option value="<?php echo $usuario['id']; ?>" 
                                                        <?php echo (isset($rendicion['tercero_id']) && $rendicion['tercero_id'] == $usuario['id']) ? 'selected' : ''; ?>>
                                                    <?php echo htmlspecialchars($usuario['nombre'] . ' ' . $usuario['apellido'] . ' - ' . $usuario['rut']); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>

                                    <div id="nuevo_tercero_fields" style="display: none;">
                                        <div class="form-row">
                                            <div class="form-col">
                                                <div class="form-group">
                                                    <label for="tercero_nombre">Nombre</label>
                                                    <input type="text" id="tercero_nombre" name="tercero_nombre" class="form-control input-sm">
                                                </div>
                                            </div>
                                            <div class="form-col">
                                                <div class="form-group">
                                                    <label for="tercero_apellido">Apellido</label>
                                                    <input type="text" id="tercero_apellido" name="tercero_apellido" class="form-control input-sm">
                                                </div>
                                            </div>
                                        </div>
                                        <div class="form-row">
                                            <div class="form-col">
                                                <div class="form-group">
                                                    <label for="tercero_rut">RUT</label>
                                                    <input type="text" id="tercero_rut" name="tercero_rut" class="form-control input-sm">
                                                </div>
                                            </div>
                                            <div class="form-col">
                                                <div class="form-group">
                                                    <label for="tercero_cargo">Cargo</label>
                                                    <input type="text" id="tercero_cargo" name="tercero_cargo" class="form-control input-sm">
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Detalle de gastos -->
                    <div class="card" style="margin-top: 20px;">
                        <div class="panel-header">
                            <h3 class="panel-title">Detalle de Gastos</h3>
                            <button type="button" id="btn-agregar-linea" class="btn btn-success btn-sm">
                                <i class="material-icons">add</i> Agregar Línea
                            </button>
                        </div>
                        
                        <div class="table-responsive">
                            <table>
                                <thead>
                                    <tr>
                                        <th>Categoría</th>
                                        <th>Monto</th>
                                        <th>Detalle</th>
                                        <th>Tipo Doc.</th>
                                        <th>N° Documento</th>
                                        <th>Acciones</th>
                                    </tr>
                                </thead>
                                <tbody id="lineas-container">
                                    <?php foreach ($lineas as $index => $linea): ?>
                                    <tr data-existing="true">
                                        <td>
                                            <select name="categoria[]" class="form-control" required>
                                                <option value="">Seleccione una categoría</option>
                                                <?php foreach ($categorias as $categoria): ?>
                                                <option value="<?php echo $categoria['id']; ?>" <?php echo ($categoria['id'] == $linea['categoria_id']) ? 'selected' : ''; ?>>
                                                    <?php echo htmlspecialchars($categoria['nombre']); ?>
                                                </option>
                                                <?php endforeach; ?>
                                            </select>
                                        </td>
                                        <td>
                                            <input type="number" name="monto[]" class="form-control monto-input" 
                                                   min="1" step="1" value="<?php echo $linea['monto']; ?>" required>
                                        </td>
                                        <td>
                                            <input type="text" name="detalle[]" class="form-control" 
                                                   value="<?php echo htmlspecialchars($linea['detalle'] ?? ''); ?>"
                                                   placeholder="Descripción específica del gasto">
                                        </td>
                                        <td>
                                            <select name="tipo_documento[]" class="form-control tipo-documento-select" onchange="toggleFacturaFields(this)">
                                                <option value="boleta" <?php echo ($linea['tipo_documento'] ?? '') == 'boleta' ? 'selected' : ''; ?>>Boleta</option>
                                                <option value="factura" <?php echo ($linea['tipo_documento'] ?? '') == 'factura' ? 'selected' : ''; ?>>Factura</option>
                                                <option value="recibo" <?php echo ($linea['tipo_documento'] ?? '') == 'recibo' ? 'selected' : ''; ?>>Recibo</option>
                                                <option value="cotizacion" <?php echo ($linea['tipo_documento'] ?? '') == 'cotizacion' ? 'selected' : ''; ?>>Cotización</option>
                                                <option value="otro" <?php echo ($linea['tipo_documento'] ?? '') == 'otro' ? 'selected' : ''; ?>>Otro</option>
                                            </select>
                                        </td>
                                        <td>
                                            <div class="documento-fields">
                                                <div class="numero-documento-field">
                                                    <input type="text" name="numero_documento[]" class="form-control input-sm" 
                                                           value="<?php echo htmlspecialchars($linea['numero_documento'] ?? ''); ?>"
                                                           placeholder="N° del documento">
                                                </div>
                                                <div class="factura-fields" style="display: <?php echo ($linea['tipo_documento'] ?? '') == 'factura' ? 'block' : 'none'; ?>;">
                                                    <input type="text" name="folio_factura[]" class="form-control input-sm mt-2" 
                                                           value="<?php echo htmlspecialchars($linea['folio_factura'] ?? ''); ?>"
                                                           placeholder="Folio factura">
                                                    <input type="text" name="proveedor_factura[]" class="form-control input-sm mt-2" 
                                                           value="<?php echo htmlspecialchars($linea['proveedor_factura'] ?? ''); ?>"
                                                           placeholder="Nombre proveedor">
                                                    <input type="text" name="rut_proveedor[]" class="form-control input-sm mt-2" 
                                                           value="<?php echo htmlspecialchars($linea['rut_proveedor'] ?? ''); ?>"
                                                           placeholder="RUT proveedor">
                                                    <input type="date" name="fecha_documento[]" class="form-control input-sm mt-2" 
                                                           value="<?php echo htmlspecialchars($linea['fecha_documento'] ?? ''); ?>">
                                                </div>
                                            </div>
                                        </td>
                                        <td>
                                            <button type="button" class="btn btn-danger btn-sm" onclick="eliminarLinea(this.parentNode.parentNode)">
                                                <i class="material-icons">delete</i>
                                            </button>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>

                    <div style="margin-top: 20px; text-align: center;">
                        <button type="button" id="btn-agregar-linea-bottom" class="btn btn-success btn-sm">
                            <i class="material-icons">add</i> Agregar Línea
                        </button>
                    </div>

                    <!-- Sección para subir archivos -->
                    <div class="card" style="margin-top: 20px;">
                        <div class="panel-header">
                            <h3 class="panel-title">Archivos de Soporte</h3>
                        </div>
                        
                        <?php if (!empty($soportes)): ?>
                        <div class="file-preview-container" id="existingFiles">
                            <?php foreach ($soportes as $soporte): ?>
                            <div class="file-preview" data-file-id="<?php echo $soporte['id']; ?>">
                                <?php
                                $extension = strtolower(pathinfo($soporte['nombre_archivo'], PATHINFO_EXTENSION));
                                if (in_array($extension, ['jpg', 'jpeg', 'png'])): ?>
                                    <img src="<?php echo $soporte['ruta_archivo']; ?>" alt="<?php echo htmlspecialchars($soporte['nombre_archivo']); ?>">
                                <?php elseif ($extension === 'pdf'): ?>
                                    <i class="material-icons file-icon">picture_as_pdf</i>
                                <?php else: ?>
                                    <i class="material-icons file-icon">insert_drive_file</i>
                                <?php endif; ?>
                                <div class="file-name"><?php echo htmlspecialchars($soporte['nombre_archivo']); ?></div>
                                <div class="file-actions">
                                    <button onclick="window.open('<?php echo $soporte['ruta_archivo']; ?>', '_blank')" title="Ver archivo">
                                        <i class="material-icons">visibility</i>
                                    </button>
                                    <?php if ($rendicion['estado_id'] == 1 || $rendicion['estado_id'] == 4): ?>
                                    <button onclick="deleteExistingFile(<?php echo $soporte['id']; ?>, <?php echo $id; ?>)" title="Eliminar archivo">
                                        <i class="material-icons">delete</i>
                                    </button>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <?php endif; ?>
                        
                        <div class="form-group">
                            <label for="soportes" class="form-label">Subir Archivos (JPEG, PNG, PDF - Máx. 5MB)</label>
                            <input type="file" id="soportes" name="soportes[]" class="form-control" multiple
                                   accept="image/jpeg,image/png,application/pdf">
                        </div>
                        
                        <div id="filePreviewContainer" class="file-preview-container"></div>
                    </div>

                    <div style="margin-top: 20px; text-align: center;">
                        <button type="submit" class="btn btn-primary" id="btnSubmit">
                            <i class="material-icons">save</i> Actualizar Rendición
                        </button>
                    </div>
                </form>
            </div>
            
            <!-- Template oculto para nuevas líneas -->
            <template id="linea-template">
                <tr>
                    <td>
                        <select name="categoria[]" class="form-control" required>
                            <option value="">Seleccione una categoría</option>
                            <?php foreach ($categorias as $categoria): ?>
                            <option value="<?php echo $categoria['id']; ?>">
                                <?php echo htmlspecialchars($categoria['nombre']); ?>
                            </option>
                            <?php endforeach; ?>
                        </select>
                    </td>
                    <td>
                        <input type="number" name="monto[]" class="form-control monto-input" 
                               min="1" step="1" placeholder="0" required>
                    </td>
                    <td>
                        <input type="text" name="detalle[]" class="form-control" 
                               placeholder="Descripción específica del gasto">
                    </td>
                    <td>
                        <select name="tipo_documento[]" class="form-control input-sm tipo-documento-select" onchange="toggleFacturaFields(this)">
                            <option value="boleta">Boleta</option>
                            <option value="factura">Factura</option>
                            <option value="recibo">Recibo</option>
                            <option value="cotizacion">Cotización</option>
                            <option value="otro">Otro</option>
                        </select>
                    </td>
                    <td>
                        <div class="documento-fields">
                            <div class="numero-documento-field">
                                <input type="text" name="numero_documento[]" class="form-control input-sm" placeholder="N° del documento">
                            </div>
                            <!-- Campos adicionales para factura -->
                            <div class="factura-fields" style="display: none;">
                                <input type="text" name="folio_factura[]" class="form-control input-sm mt-2" placeholder="Folio factura">
                                <input type="text" name="proveedor_factura[]" class="form-control input-sm mt-2" placeholder="Nombre proveedor">
                                <input type="text" name="rut_proveedor[]" class="form-control input-sm mt-2" placeholder="RUT proveedor">
                                <input type="date" name="fecha_documento[]" class="form-control input-sm mt-2">
                            </div>
                        </div>
                    </td>
                    <td>
                        <button type="button" class="btn btn-danger btn-sm" onclick="eliminarLinea(this.parentNode.parentNode)">
                            <i class="material-icons">delete</i>
                        </button>
                    </td>
                </tr>
            </template>
            <?php
        } catch (Exception $e) {
            echo '<div class="alert alert-danger">' . $e->getMessage() . '</div>';
            echo '<div style="text-align: center; margin-top: 20px;">
                      <a href="rendiciones.php" class="btn btn-primary">
                          <i class="material-icons">arrow_back</i> Volver a la Lista
                      </a>
                  </div>';
        }
        break;
        
    case 'view':
        // Ver detalles de una rendición
        try {
            // Obtener datos de la rendición
            $stmt = $conn->prepare("
                SELECT r.*, 
                       u.nombre, u.apellido, u.rut, u.cargo,
                       tercero.id as tercero_id, 
                       tercero.nombre as tercero_nombre, 
                       tercero.apellido as tercero_apellido, 
                       tercero.rut as tercero_rut, 
                       tercero.cargo as tercero_cargo,
                       e.nombre as estado_nombre
                FROM rendiciones r
                JOIN users u ON r.user_id = u.id
                LEFT JOIN users tercero ON r.tercero_id = tercero.id
                LEFT JOIN estados e ON r.estado_id = e.id
                WHERE r.id = :id " . (!$is_admin ? "AND r.user_id = :user_id" : "")
            );
            
            $stmt->bindParam(':id', $id);
            if (!$is_admin) {
                $stmt->bindParam(':user_id', $user_id);
            }
            
            $stmt->execute();
            $rendicion = $stmt->fetch();
            
            if (!$rendicion) {
                throw new Exception("Rendición no encontrada o sin permisos para verla");
            }
            
            // Obtener líneas de detalle
            $stmt = $conn->prepare("
                SELECT rc.*, cg.nombre as categoria_nombre 
                FROM rendicion_categoria rc
                JOIN categorias_gastos cg ON rc.categoria_id = cg.id
                WHERE rc.rendicion_id = :id
                ORDER BY rc.id"
            );
            
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $lineas = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Obtener archivos de soporte
            $stmt = $conn->prepare("SELECT * FROM rendicion_soportes WHERE rendicion_id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $soportes = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Generar código
            $codigo = generarCodigo('REN', $id);
            
            ?>
            <div class="card">
                <div class="panel-header">
                    <h2 class="panel-title">Detalle de Rendición: <?php echo $codigo; ?></h2>
                    <div>
                        <a href="rendiciones.php" class="btn btn-primary btn-sm">
                            <i class="material-icons">arrow_back</i> Volver a la Lista
                        </a>
                        
                        <?php if ($rendicion['estado_id'] == 3 || $rendicion['estado_id'] == 5): // Si está aprobada o pagada ?>
                            <a href="generate_pdf.php?tipo=rendicion&id=<?php echo $id; ?>" class="btn btn-success btn-sm" target="_blank">
                                <i class="material-icons">picture_as_pdf</i> Generar PDF
                            </a>
                        <?php else: ?>
                            <a href="generate_pdf.php?tipo=rendicion&id=<?php echo $id; ?>" class="btn btn-warning btn-sm" target="_blank">
                                <i class="material-icons">picture_as_pdf</i> Generar PDF
                            </a>
                        <?php endif; ?>
                        
                        <?php if (($rendicion['estado_id'] == 1 || $rendicion['estado_id'] == 4) && 
                                  ($rendicion['user_id'] == $user_id || $is_admin)): ?>
                            <a href="rendiciones.php?action=edit&id=<?php echo $id; ?>" class="btn btn-warning btn-sm">
                                <i class="material-icons">edit</i> Editar
                            </a>
                        <?php endif; ?>
                        
                        <?php if ($is_admin && ($rendicion['estado_id'] == 1 || $rendicion['estado_id'] == 4)): ?>
                            <a href="rendiciones.php?action=delete&id=<?php echo $id; ?>" 
                               onclick="return confirm('¿Está seguro de eliminar esta rendición?')"
                               class="btn btn-danger btn-sm">
                                <i class="material-icons">delete</i> Eliminar
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <div class="form-row">
                        <div class="form-col">
                            <div class="info-card">
                                <h4>Información General</h4>
                                <table class="table-compact">
                                    <tr>
                                        <th style="width: 40%;">Código:</th>
                                        <td><?php echo $codigo; ?></td>
                                    </tr>
                                    <tr>
                                        <th>Fecha:</th>
                                        <td><?php echo formatoFecha($rendicion['fecha']); ?></td>
                                    </tr>
                                    <tr>
                                        <th>Estado:</th>
                                        <td>
                                            <span class="status-badge badge-estado-<?php echo $rendicion['estado_id']; ?>">
                                                <?php echo $estados_map[$rendicion['estado_id']]['nombre'] ?? 'Desconocido'; ?>
                                            </span>
                                        </td>
                                    </tr>
                                    <tr>
                                        <th>Área:</th>
                                        <td>
                                            <?php
                                            $areas = [
                                                'logistica' => 'Logística',
                                                'servicio_tecnico' => 'Servicio Técnico',
                                                'marketing' => 'Marketing',
                                                'ventas' => 'Ventas',
                                                'comercial' => 'Área Comercial',
                                                'operaciones' => 'Operaciones',
                                                'finanzas' => 'Finanzas',
                                                'ecommerce' => 'Ecommerce',
                                                'barista' => 'Barista',
                                                'otra' => 'Otra'
                                            ];
                                            echo isset($rendicion['area_origen']) && isset($areas[$rendicion['area_origen']]) ? 
                                                $areas[$rendicion['area_origen']] : 'No especificada';
                                            ?>
                                        </td>
                                    </tr>
                                    <tr>
                                        <th>Descripción:</th>
                                        <td><?php echo nl2br(htmlspecialchars($rendicion['descripcion'])); ?></td>
                                    </tr>
                                </table>
                            </div>
                        </div>
                        
                        <div class="form-col">
                            <div class="info-card">
                                <h4>Información Financiera</h4>
                                <table class="table-compact">
                                    <tr>
                                        <th style="width: 40%;">Monto Total:</th>
                                        <td><?php echo formatoMoneda($rendicion['monto_total']); ?></td>
                                    </tr>
                                    <tr>
                                        <th>Fondo a Rendir:</th>
                                        <td><?php echo formatoMoneda($rendicion['fondo_a_rendir']); ?></td>
                                    </tr>
                                    <tr>
                                        <th>Saldo Pendiente:</th>
                                        <td style="color: <?php echo ($rendicion['fondo_a_rendir'] - $rendicion['monto_total']) >= 0 ? '#28a745' : '#dc3545'; ?>">
                                            <?php echo formatoMoneda($rendicion['fondo_a_rendir'] - $rendicion['monto_total']); ?>
                                        </td>
                                    </tr>
                                </table>
                            </div>

                            <div class="info-card">
                                <h4>Información del Usuario</h4>
                                <table class="table-compact">
                                    <?php if (!empty($rendicion['tercero_id'])): ?>
                                        <tr>
                                            <th style="width: 40%;">Rendido por:</th>
                                            <td><?php echo htmlspecialchars($rendicion['tercero_nombre'] . ' ' . $rendicion['tercero_apellido']); ?></td>
                                        </tr>
                                        <tr>
                                            <th>RUT:</th>
                                            <td><?php echo htmlspecialchars($rendicion['tercero_rut']); ?></td>
                                        </tr>
                                        <tr>
                                            <th>Cargo:</th>
                                            <td><?php echo htmlspecialchars($rendicion['tercero_cargo']); ?></td>
                                        </tr>
                                        <tr>
                                            <th>Ingresado por:</th>
                                            <td><?php echo htmlspecialchars($rendicion['nombre'] . ' ' . $rendicion['apellido']); ?></td>
                                        </tr>
                                    <?php else: ?>
                                        <tr>
                                            <th style="width: 40%;">Nombre:</th>
                                            <td><?php echo htmlspecialchars($rendicion['nombre'] . ' ' . $rendicion['apellido']); ?></td>
                                        </tr>
                                        <tr>
                                            <th>RUT:</th>
                                            <td><?php echo htmlspecialchars($rendicion['rut']); ?></td>
                                        </tr>
                                        <tr>
                                            <th>Cargo:</th>
                                            <td><?php echo htmlspecialchars($rendicion['cargo']); ?></td>
                                        </tr>
                                    <?php endif; ?>
                                    <?php if ($rendicion['revisor_id']): ?>
                                    <tr>
                                        <th>Revisado por:</th>
                                        <td><?php echo htmlspecialchars($rendicion['revisor_nombre'] . ' ' . $rendicion['revisor_apellido']); ?></td>
                                    </tr>
                                    <tr>
                                        <th>Fecha Revisión:</th>
                                        <td><?php echo date('d/m/Y H:i', strtotime($rendicion['fecha_revision'])); ?></td>
                                    </tr>
                                    <?php endif; ?>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>

                <style>
                    .table-compact {
                        width: 100%;
                        margin-bottom: 0;
                    }
                    .table-compact th,
                    .table-compact td {
                        padding: 4px 8px;
                        vertical-align: top;
                        border-bottom: 1px solid #dee2e6;
                    }
                    .table-compact tr:last-child th,
                    .table-compact tr:last-child td {
                        border-bottom: none;
                    }
                    .info-card + .info-card {
                        margin-top: 15px;
                    }
                </style>
                
                <div class="card">
                    <div class="panel-header">
                        <h3 class="panel-title">Detalle de Gastos</h3>
                    </div>
                    
                    <div class="table-responsive">
                        <table>
                            <thead>
                                    <tr>
                                        <th>#</th>
                                        <th>Categoría</th>
                                        <th>Monto</th>
                                        <th>Detalle</th>
                                        <th>Tipo Doc.</th>
                                        <th>Datos Documento</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (empty($lineas)): ?>
                                    <tr>
                                        <td colspan="6" style="text-align: center;">No hay líneas de detalle</td>
                                    </tr>
                                    <?php else: ?>
                                        <?php foreach ($lineas as $index => $linea): ?>
                                        <tr data-existing="true">
                                            <td><?php echo $index + 1; ?></td>
                                            <td><?php echo htmlspecialchars($linea['categoria_nombre']); ?></td>
                                            <td><?php echo formatoMoneda($linea['monto']); ?></td>
                                            <td><?php echo htmlspecialchars($linea['detalle'] ?? ''); ?></td>
                                            <td>
                                                <?php 
                                                $tipos_doc = [
                                                    'boleta' => 'Boleta',
                                                    'factura' => 'Factura',
                                                    'recibo' => 'Recibo',
                                                    'cotizacion' => 'Cotización',
                                                    'otro' => 'Otro'
                                                ];
                                                echo isset($linea['tipo_documento']) && isset($tipos_doc[$linea['tipo_documento']]) ? 
                                                    $tipos_doc[$linea['tipo_documento']] : 'No especificado'; 
                                                ?>
                                            </td>
                                            <td>
                                                <?php if ($linea['tipo_documento'] == 'factura'): ?>
                                                    <strong>Folio:</strong> <?php echo htmlspecialchars($linea['folio_factura'] ?? ''); ?><br>
                                                    <strong>Proveedor:</strong> <?php echo htmlspecialchars($linea['proveedor_factura'] ?? ''); ?><br>
                                                    <strong>RUT:</strong> <?php echo htmlspecialchars($linea['rut_proveedor'] ?? ''); ?><br>
                                                    <strong>Fecha:</strong> <?php echo !empty($linea['fecha_documento']) ? date('d/m/Y', strtotime($linea['fecha_documento'])) : '-'; ?>
                                                <?php else: ?>
                                                    <?php echo htmlspecialchars($linea['numero_documento'] ?? '-'); ?>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            <tfoot>
                                <tr>
                                    <th colspan="2" style="text-align: right;">Total:</th>
                                    <th><?php echo formatoMoneda($rendicion['monto_total']); ?></th>
                                    <th></th>
                                </tr>
                            </tfoot>
                        </table>
                    </div>
                </div>
                
                <!-- Nueva sección para mostrar archivos de soporte -->
                <div class="card" style="margin-top: 20px;">
                    <div class="panel-header">
                        <h3 class="panel-title">Archivos de Soporte</h3>
                    </div>
                    
                    <?php if (!empty($soportes)): ?>
                    <div class="file-preview-container" id="existingFiles">
                        <?php foreach ($soportes as $soporte): ?>
                        <div class="file-preview" data-file-id="<?php echo $soporte['id']; ?>">
                            <?php
                            $extension = strtolower(pathinfo($soporte['nombre_archivo'], PATHINFO_EXTENSION));
                            if (in_array($extension, ['jpg', 'jpeg', 'png'])): ?>
                                <img src="<?php echo $soporte['ruta_archivo']; ?>" alt="<?php echo htmlspecialchars($soporte['nombre_archivo']); ?>">
                            <?php elseif ($extension === 'pdf'): ?>
                                <i class="material-icons file-icon">picture_as_pdf</i>
                            <?php else: ?>
                                <i class="material-icons file-icon">insert_drive_file</i>
                            <?php endif; ?>
                            <div class="file-name"><?php echo htmlspecialchars($soporte['nombre_archivo']); ?></div>
                            <div class="file-actions">
                                <button onclick="window.open('<?php echo $soporte['ruta_archivo']; ?>', '_blank')" title="Ver archivo">
                                    <i class="material-icons">visibility</i>
                                </button>
                                <?php if ($rendicion['estado_id'] == 1 || $rendicion['estado_id'] == 4): ?>
                                <button onclick="deleteExistingFile(<?php echo $soporte['id']; ?>, <?php echo $id; ?>)" title="Eliminar archivo">
                                    <i class="material-icons">delete</i>
                                </button>
                                <?php endif; ?>
                            </div>
                        </div>
                                <?php endforeach; ?>
                    </div>
                    <?php endif; ?>
                    
                    <div class="file-upload-container" onclick="document.getElementById('soportes').click()">
                        <i class="material-icons upload-icon">cloud_upload</i>
                        <div class="upload-text">Arrastra y suelta archivos aquí o haz clic para seleccionar</div>
                        <div class="upload-hint">Formatos permitidos: JPEG, PNG, PDF (Máx. 5MB)</div>
                        <input type="file" id="soportes" name="soportes[]" multiple accept="image/jpeg,image/png,application/pdf">
                    </div>
                    
                    <div id="filePreviewContainer" class="file-preview-container"></div>
                </div>
                
                <?php if ($is_admin && $rendicion['estado_id'] == 1): // Si está pendiente y es admin ?>
                <div class="card" style="margin-top: 20px;">
                    <div class="panel-header">
                        <h3 class="panel-title">Revisión de Rendición</h3>
                    </div>
                    
                    <form method="POST" action="rendiciones.php?action=review&id=<?php echo $id; ?>">
                        <div class="form-group">
                            <label for="decision" class="form-label">Decisión</label>
                            <select id="decision" name="decision" class="form-control" required>
                                <option value="">Seleccione una opción</option>
                                <option value="aprobar">Aprobar</option>
                                <option value="rechazar">Rechazar</option>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="comentario" class="form-label">Comentario (opcional)</label>
                            <textarea id="comentario" name="comentario" class="form-control" 
                                      rows="3" placeholder="Ingrese comentarios sobre la revisión"></textarea>
                        </div>
                        
                        <div style="text-align: center;">
                            <button type="submit" class="btn btn-primary">
                                <i class="material-icons">check_circle</i> Enviar Revisión
                            </button>
                        </div>
                    </form>
                </div>
                <?php endif; ?>
            </div>
            <?php
        } catch (Exception $e) {
            echo '<div class="alert alert-danger">' . $e->getMessage() . '</div>';
            echo '<div style="text-align: center; margin-top: 20px;">
                      <a href="rendiciones.php" class="btn btn-primary">
                          <i class="material-icons">arrow_back</i> Volver a la Lista
                      </a>
                  </div>';
        }
        break;
        
    default:
        // Lista de rendiciones
        try {
            // Filtros de búsqueda
            $estado = isset($_GET['estado']) ? intval($_GET['estado']) : 0;
            $fecha_desde = isset($_GET['fecha_desde']) ? $_GET['fecha_desde'] : '';
            $fecha_hasta = isset($_GET['fecha_hasta']) ? $_GET['fecha_hasta'] : '';
            $busqueda = isset($_GET['busqueda']) ? trim($_GET['busqueda']) : '';
            
            // Obtener parámetros de ordenamiento
            $order_by = isset($_GET['order_by']) ? $_GET['order_by'] : 'fecha_creacion';
            $order_dir = (isset($_GET['order_dir']) && strtolower($_GET['order_dir']) === 'asc') ? 'ASC' : 'DESC';

            // Validar columnas permitidas para ordenar
            $allowed_order = [
                'codigo' => 'r.id',
                'fecha' => 'r.fecha',
                'usuario' => 'u.nombre',
                'monto' => 'r.monto_total',
                'fecha_creacion' => 'r.fecha_creacion'
            ];
            $order_sql = isset($allowed_order[$order_by]) ? $allowed_order[$order_by] : 'r.fecha_creacion';
            
            // Construir consulta base
            $sql = "
                SELECT r.*, u.nombre, u.apellido, u.rut,
                       tercero.nombre as tercero_nombre, tercero.apellido as tercero_apellido
                FROM rendiciones r
                JOIN users u ON r.user_id = u.id
                LEFT JOIN users tercero ON r.tercero_id = tercero.id
                WHERE 1=1
            ";
            
            $params = [];
            
            // Si no es admin, solo ver sus propias rendiciones
            if (!$is_admin) {
                $sql .= " AND r.user_id = :user_id";
                $params[':user_id'] = $user_id;
            }
            
            // Aplicar filtros
            if ($estado > 0) {
                $sql .= " AND r.estado_id = :estado";
                $params[':estado'] = $estado;
            }
            
            if (!empty($fecha_desde)) {
                $sql .= " AND r.fecha >= :fecha_desde";
                $params[':fecha_desde'] = $fecha_desde;
            }
            
            if (!empty($fecha_hasta)) {
                $sql .= " AND r.fecha <= :fecha_hasta";
                $params[':fecha_hasta'] = $fecha_hasta;
            }
            
            if (!empty($busqueda)) {
                $sql .= " AND (
                    r.descripcion LIKE :busqueda1 
                    OR u.nombre LIKE :busqueda2 
                    OR u.apellido LIKE :busqueda3 
                    OR u.rut LIKE :busqueda4
                )";
                $params[':busqueda1'] = '%' . $busqueda . '%';
                $params[':busqueda2'] = '%' . $busqueda . '%';
                $params[':busqueda3'] = '%' . $busqueda . '%';
                $params[':busqueda4'] = '%' . $busqueda . '%';
            }
            
            // Ordenar por columna seleccionada
            $sql .= " ORDER BY $order_sql $order_dir";
            
            // Preparar y ejecutar consulta
            $stmt = $conn->prepare($sql);
            foreach ($params as $param => $value) {
                $stmt->bindValue($param, $value);
            }
            
            $stmt->execute();
            $rendiciones = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            ?>
            <div class="card">
                <div class="panel-header">
                    <h2 class="panel-title">Rendiciones de Gastos</h2>
                    <?php if (tienePermiso('crear_rendicion') || $is_admin): ?>
                    <a href="rendiciones.php?action=new" class="btn btn-primary btn-sm">
                        <i class="material-icons">add</i> Nueva Rendición
                    </a>
                    <?php endif; ?>
                </div>
                
                <!-- Filtros de búsqueda -->
                <div class="card" style="margin-bottom: 20px;">
                    <div class="panel-header">
                        <h3 class="panel-title">Filtros de Búsqueda</h3>
                    </div>
                    
                    <form method="GET" action="rendiciones.php" class="form-inline" style="display: flex; flex-wrap: wrap; gap: 10px;">
                        <div style="flex: 1; min-width: 200px;">
                            <label for="estado" class="form-label">Estado</label>
                            <select id="estado" name="estado" class="form-control">
                                <option value="0">Todos</option>
                                <?php foreach ($estados as $est): ?>
                                <option value="<?php echo $est['id']; ?>" <?php echo $estado == $est['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($est['nombre']); ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div style="flex: 1; min-width: 200px;">
                            <label for="fecha_desde" class="form-label">Fecha Desde</label>
                            <input type="date" id="fecha_desde" name="fecha_desde" class="form-control" 
                                   value="<?php echo $fecha_desde; ?>">
                        </div>
                        
                        <div style="flex: 1; min-width: 200px;">
                            <label for="fecha_hasta" class="form-label">Fecha Hasta</label>
                            <input type="date" id="fecha_hasta" name="fecha_hasta" class="form-control" 
                                   value="<?php echo $fecha_hasta; ?>">
                        </div>
                        
                        <div style="flex: 2; min-width: 300px;">
                            <label for="busqueda" class="form-label">Búsqueda</label>
                            <input type="text" id="busqueda" name="busqueda" class="form-control" 
                                   placeholder="Descripción, usuario, RUT..." value="<?php echo htmlspecialchars($busqueda); ?>">
                        </div>
                        
                        <div style="display: flex; align-items: flex-end; gap: 10px;">
                            <button type="submit" class="btn btn-primary">
                                <i class="material-icons">search</i> Buscar
                            </button>
                            
                            <a href="rendiciones.php" class="btn btn-danger">
                                <i class="material-icons">clear</i> Limpiar
                            </a>
                        </div>
                    </form>
                </div>
                
                <style>
                /* Estilos personalizados para la tabla de rendiciones */
                .rendiciones-table td, .rendiciones-table th {
                    padding: 8px 10px; /* Reducimos el padding para filas menos gruesas */
                }
                
                .rendiciones-table tr:hover {
                    background-color: rgba(0,0,0,0.03);
                }
                
                .action-icon {
                    display: inline-flex;
                    align-items: center;
                    justify-content: center;
                    width: 28px;
                    height: 28px;
                    border-radius: 50%;
                    margin-right: 5px;
                    transition: all 0.2s;
                    color: #555 !important; /* Forzamos color oscuro para los iconos */
                    background-color: #f0f0f0;
                }
                
                .action-icon:hover {
                    background-color: #e0e0e0;
                    transform: scale(1.1);
                }
                
                .action-icon.view-icon:hover {
                    color: #1e88e5 !important;
                }
                
                .action-icon.pdf-icon:hover {
                    color: #66bb6a !important;
                }
                
                .action-icon.edit-icon:hover {
                    color: #ffa726 !important;
                }
                
                .action-icon.delete-icon:hover {
                    color: #ef5350 !important;
                }
                
                .action-icon i {
                    font-size: 18px;
                }
                </style>
                
                <div class="table-responsive">
                    <form id="form-masivo" method="POST" action="rendiciones.php?action=masivo">
                    <div style="margin-bottom:10px; display:flex; gap:10px; align-items:center;">
                        <button type="submit" name="accion_masiva" value="aprobar" class="btn btn-success btn-sm" onclick="return confirm('¿Aprobar todas las seleccionadas?')">Aprobar seleccionadas</button>
                        <button type="submit" name="accion_masiva" value="rechazar" class="btn btn-danger btn-sm" onclick="return confirm('¿Rechazar todas las seleccionadas?')">Rechazar seleccionadas</button>
                    </div>
                    <table class="rendiciones-table">
                        <thead>
                            <tr>
                                <th><input type="checkbox" id="check-todas"></th>
                                <th>
                                    <a href="#" onclick="return ordenar('codigo')">Código
                                        <?php if ($order_by == 'codigo'): ?>
                                            <span class="order-arrows active"><?php echo $order_dir == 'ASC' ? '▲' : '▼'; ?></span>
                                        <?php endif; ?>
                                    </a>
                                </th>
                                <th>
                                    <a href="#" onclick="return ordenar('fecha')">Fecha
                                        <?php if ($order_by == 'fecha'): ?>
                                            <span class="order-arrows"><?php echo $order_dir == 'ASC' ? '▲' : '▼'; ?></span>
                                        <?php endif; ?>
                                    </a>
                                </th>
                                <th>Descripción</th>
                                <th>
                                    <a href="#" onclick="return ordenar('monto')">Monto
                                        <?php if ($order_by == 'monto'): ?>
                                            <span class="order-arrows"><?php echo $order_dir == 'ASC' ? '▲' : '▼'; ?></span>
                                        <?php endif; ?>
                                    </a>
                                </th>
                                <th>
                                    <a href="#" onclick="return ordenar('usuario')">Usuario
                                        <?php if ($order_by == 'usuario'): ?>
                                            <span class="order-arrows"><?php echo $order_dir == 'ASC' ? '▲' : '▼'; ?></span>
                                        <?php endif; ?>
                                    </a>
                                </th>
                                <th>Rendido por</th>
                                <th>Estado</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($rendiciones)): ?>
                            <tr>
                                <td colspan="<?php echo $is_admin ? '8' : '7'; ?>" style="text-align: center;">
                                    No se encontraron rendiciones
                                </td>
                            </tr>
                            <?php else: ?>
                                <?php foreach ($rendiciones as $rendicion): ?>
                                <?php $codigo = generarCodigo('REN', $rendicion['id']); ?>
                                <tr>
                                    <td><input type="checkbox" name="ids[]" value="<?php echo $rendicion['id']; ?>" class="check-rendicion"></td>
                                    <td><?php echo $codigo; ?></td>
                                    <td><?php echo formatoFecha($rendicion['fecha']); ?></td>
                                    <td><?php echo htmlspecialchars(substr($rendicion['descripcion'], 0, 50)) . (strlen($rendicion['descripcion']) > 50 ? '...' : ''); ?></td>
                                    <td><?php echo formatoMoneda($rendicion['monto_total']); ?></td>
                                    <td><?php echo htmlspecialchars($rendicion['nombre'] . ' ' . $rendicion['apellido']); ?></td>
                                    <td>
                                        <?php
                                        if (!empty($rendicion['tercero_nombre'])) {
                                            echo htmlspecialchars($rendicion['tercero_nombre'] . ' ' . $rendicion['tercero_apellido']);
                                        } else {
                                            echo htmlspecialchars($rendicion['nombre'] . ' ' . $rendicion['apellido']);
                                        }
                                        ?>
                                    </td>
                                    <td>
                                        <span class="status-badge badge-estado-<?php echo $rendicion['estado_id']; ?>">
                                            <?php echo $estados_map[$rendicion['estado_id']]['nombre'] ?? 'Desconocido'; ?>
                                        </span>
                                    </td>
                                    <td>
                                        <a href="rendiciones.php?action=view&id=<?php echo $rendicion['id']; ?>" 
                                           title="Ver detalles" class="action-icon view-icon">
                                            <i class="material-icons">visibility</i>
                                        </a>
                                        
                                        <?php if ($rendicion['estado_id'] == 3 || $rendicion['estado_id'] == 5): // Si está aprobada o pagada ?>
                                        <a href="generate_pdf.php?tipo=rendicion&id=<?php echo $rendicion['id']; ?>" 
                                           title="Generar PDF" class="action-icon pdf-icon" target="_blank">
                                            <i class="material-icons">picture_as_pdf</i>
                                        </a>
                                        <?php else: ?>
                                        <a href="generate_pdf.php?tipo=rendicion&id=<?php echo $rendicion['id']; ?>" 
                                           title="Generar PDF" class="action-icon pdf-icon-warning" target="_blank">
                                            <i class="material-icons">picture_as_pdf</i>
                                        </a>
                                        <?php endif; ?>
                                        
                                        <?php if (($rendicion['estado_id'] == 1 || $rendicion['estado_id'] == 4) && 
                                                 ($rendicion['user_id'] == $user_id || $is_admin)): ?>
                                        <a href="rendiciones.php?action=edit&id=<?php echo $rendicion['id']; ?>" 
                                           title="Editar" class="action-icon edit-icon">
                                            <i class="material-icons">edit</i>
                                        </a>
                                        <?php endif; ?>
                                        
                                        <?php if ($is_admin && ($rendicion['estado_id'] == 1 || $rendicion['estado_id'] == 4)): ?>
                                        <a href="rendiciones.php?action=delete&id=<?php echo $rendicion['id']; ?>" 
                                           onclick="return confirm('¿Está seguro de eliminar esta rendición?')"
                                           title="Eliminar" class="action-icon delete-icon">
                                            <i class="material-icons">delete</i>
                                        </a>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                    </form>
                </div>
            </div>
            <?php
        } catch (PDOException $e) {
            echo '<div class="alert alert-danger">Error al cargar rendiciones: ' . $e->getMessage() . '</div>';
        }
        break;
}
?>
<script>
document.addEventListener('DOMContentLoaded', function() {
    const fileInput = document.getElementById('soportes');
    const filePreviewContainer = document.getElementById('file-preview-container');
    const existingFilesContainer = document.getElementById('existing-files');
    const previewModal = document.getElementById('previewModal');
    const previewContent = document.getElementById('previewContent');
    const closePreview = document.getElementById('closePreview');

    // Función para mostrar vista previa de archivo
    function showFilePreview(file) {
        const preview = document.createElement('div');
        preview.className = 'file-preview';
        preview.setAttribute('data-file-name', file.name);
        
        const fileType = file.type;
        const isImage = fileType.startsWith('image/');
        const isPDF = fileType === 'application/pdf';
        
        let previewContent = '';
        if (isImage) {
            previewContent = `<img src="${URL.createObjectURL(file)}" alt="${file.name}">`;
        } else if (isPDF) {
            previewContent = `<i class="fas fa-file-pdf"></i>`;
        } else {
            previewContent = `<i class="fas fa-file"></i>`;
        }
        
        preview.innerHTML = `
            <div class="file-info">
                <div class="file-name">${file.name}</div>
                <div class="file-size">${formatFileSize(file.size)}</div>
            </div>
            <div class="file-preview-content">
                ${previewContent}
            </div>
            <div class="file-actions">
                <button type="button" class="btn btn-sm btn-info" onclick="previewFile('${URL.createObjectURL(file)}', '${fileType}')">
                    <i class="fas fa-eye"></i>
                </button>
                <button type="button" class="btn btn-sm btn-danger" onclick="removeFile(this)">
                    <i class="fas fa-trash"></i>
                </button>
            </div>
        `;
        
        filePreviewContainer.appendChild(preview);
    }

    // Función para formatear tamaño de archivo
    function formatFileSize(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }

    // Función para eliminar archivo existente
    window.deleteExistingFile = function(fileId, rendicionId) {
        if (confirm('¿Está seguro de que desea eliminar este archivo?')) {
            fetch('rendiciones.php?action=delete_file&id=' + rendicionId + '&file_id=' + fileId, {
                method: 'POST',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    const fileElement = document.querySelector(`[data-file-id="${fileId}"]`);
                    if (fileElement) {
                        fileElement.remove();
                    }
                } else {
                    alert('Error al eliminar el archivo: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error al eliminar el archivo');
            });
        }
    }

    // Función para eliminar archivo seleccionado
    window.removeFile = function(button) {
        const preview = button.closest('.file-preview');
        if (preview) {
            preview.remove();
        }
    }

    // Función para previsualizar archivo
    window.previewFile = function(url, type) {
        previewContent.innerHTML = '';
        if (type.startsWith('image/')) {
            const img = document.createElement('img');
            img.src = url;
            img.style.maxWidth = '100%';
            img.style.maxHeight = '80vh';
            previewContent.appendChild(img);
        } else if (type === 'application/pdf') {
            const iframe = document.createElement('iframe');
            iframe.src = url;
            iframe.style.width = '100%';
            iframe.style.height = '80vh';
            previewContent.appendChild(iframe);
        } else {
            previewContent.innerHTML = '<p>Vista previa no disponible para este tipo de archivo</p>';
        }
        previewModal.style.display = 'block';
    }

    // Función para agregar una nueva línea
    function agregarLinea() {
        const template = document.getElementById('linea-template');
        const tbody = document.getElementById('lineas-container');
        
        if (!template || !tbody) {
            console.error('No se encontró el template o el contenedor de líneas');
            return;
        }
        
        // Clonar el contenido del template
        const nuevaLinea = template.content.cloneNode(true);
        tbody.appendChild(nuevaLinea);
        
        // Inicializar los nuevos campos
        const ultimaLinea = tbody.lastElementChild;
        const montoInput = ultimaLinea.querySelector('.monto-input');
        const tipoDocumentoSelect = ultimaLinea.querySelector('.tipo-documento-select');
        const facturaFields = ultimaLinea.querySelector('.factura-fields');
        
        if (montoInput) {
            montoInput.addEventListener('input', calcularTotales);
        }
        
        // Inicializar el estado de los campos de factura
        if (tipoDocumentoSelect) {
            toggleFacturaFields(tipoDocumentoSelect);
        }
    }
    
    // Función para eliminar una línea
    window.eliminarLinea = function(tr) {
        if (!tr) return;
        
        if (confirm('¿Está seguro de eliminar esta línea?')) {
            tr.remove();
            calcularTotales();
        }
    }
    
    // Función para calcular totales
    function calcularTotales() {
        let total = 0;
        const montoInputs = document.querySelectorAll('.monto-input');
        montoInputs.forEach(input => {
            total += parseFloat(input.value || 0);
        });
        
        const fondoRendirInput = document.getElementById('fondo_a_rendir');
        const fondoRendir = fondoRendirInput ? parseFloat(fondoRendirInput.value || 0) : 0;
        const saldoPendiente = fondoRendir - total;
        
        const totalRendicion = document.getElementById('total-rendicion');
        const fondoTotal = document.getElementById('fondo-total');
        const saldoPendienteElement = document.getElementById('saldo-pendiente');
        
        if (totalRendicion) totalRendicion.textContent = formatoMoneda(total);
        if (fondoTotal) fondoTotal.textContent = formatoMoneda(fondoRendir);
        if (saldoPendienteElement) saldoPendienteElement.textContent = formatoMoneda(saldoPendiente);
    }
    
    // Agregar event listeners para los botones de agregar línea
    const btnAgregarLinea = document.getElementById('btn-agregar-linea');
    const btnAgregarLineaBottom = document.getElementById('btn-agregar-linea-bottom');
    
    if (btnAgregarLinea) {
        btnAgregarLinea.addEventListener('click', agregarLinea);
    }
    
    if (btnAgregarLineaBottom) {
        btnAgregarLineaBottom.addEventListener('click', agregarLinea);
    }
    
    // Agregar event listeners para los inputs de monto
    const montoInputs = document.querySelectorAll('.monto-input');
    montoInputs.forEach(input => {
        input.addEventListener('input', calcularTotales);
    });
    
    // Evento para cerrar modal
    if (closePreview) {
        closePreview.onclick = function() {
            previewModal.style.display = 'none';
        }
    }

    // Evento para selección de archivos
    if (fileInput) {
        fileInput.addEventListener('change', function(e) {
            const files = e.target.files;
            if (files.length > 0) {
                Array.from(files).forEach(file => {
                    showFilePreview(file);
                });
            }
        });
    }

    // Cerrar modal al hacer clic fuera
    window.onclick = function(event) {
        if (event.target == previewModal) {
            previewModal.style.display = 'none';
        }
    }
    
    // Calcular totales iniciales
    calcularTotales();

    // Función para mostrar/ocultar campos de factura
    window.toggleFacturaFields = function(selectElement) {
        const row = selectElement.closest('tr');
        if (!row) return;
        
        const documentoFields = row.querySelector('.documento-fields');
        if (!documentoFields) return;
        
        const numeroDocumentoField = documentoFields.querySelector('.numero-documento-field');
        const facturaFields = documentoFields.querySelector('.factura-fields');
        
        if (!numeroDocumentoField || !facturaFields) return;
        
        if (selectElement.value === 'factura') {
            numeroDocumentoField.style.display = 'none';
            facturaFields.style.display = 'block';
            
            // Hacer los campos de factura requeridos
            facturaFields.querySelectorAll('input').forEach(input => {
                input.required = true;
            });
            numeroDocumentoField.querySelector('input').required = false;
            numeroDocumentoField.querySelector('input').value = '';
            
            // Inicializar la validación de factura duplicada
            initializeFacturaValidation(facturaFields);
            
        } else {
            numeroDocumentoField.style.display = 'block';
            facturaFields.style.display = 'none';
            
            // Hacer el campo de número de documento requerido
            numeroDocumentoField.querySelector('input').required = true;
            
            // Limpiar y quitar required de los campos de factura
            facturaFields.querySelectorAll('input').forEach(input => {
                input.required = false;
                input.value = '';
                input.setCustomValidity('');
            });
            
            // Remover alerta si existe
            const existingAlert = facturaFields.querySelector('.alert');
            if (existingAlert) {
                existingAlert.remove();
            }
        }
    };

    // Agregar función de inicialización de validación
    function initializeFacturaValidation(facturaFields) {
        const folioInput = facturaFields.querySelector('input[name="folio_factura[]"]');
        if (!folioInput) return; // Si no existe el input, salir de la función
        
        const checkFacturaDuplicada = () => {
            const folio = folioInput.value.trim();
            
            if (folio) {
                const rendicionId = document.querySelector('input[name="id"]')?.value;
                const url = `rendiciones.php?check_factura=1&folio=${encodeURIComponent(folio)}${rendicionId ? '&rendicion_id=' + rendicionId : ''}`;
                
                fetch(url)
                    .then(response => response.json())
                    .then(data => {
                        if (data.exists) {
                            // Verificar si el modal existe, si no, crearlo
                            let modal = document.getElementById('facturaDuplicadaModal');
                            if (!modal) {
                                const modalHTML = `
                                    <div class="modal" id="facturaDuplicadaModal" tabindex="-1">
                                        <div class="modal-dialog">
                                            <div class="modal-content">
                                                <div class="modal-header">
                                                    <h5 class="modal-title">Factura Duplicada</h5>
                                                    <button type="button" class="btn-close" aria-label="Close">&times;</button>
                                                </div>
                                                <div class="modal-body">
                                                    <div id="facturaDuplicadaInfo"></div>
                                                </div>
                                                <div class="modal-footer">
                                                    <button type="button" class="btn btn-primary" id="btnContinuarFactura">Continuar</button>
                                                    <button type="button" class="btn btn-secondary" id="btnCancelarFactura">Cancelar</button>
                                                </div>
                                            </div>
                                        </div>
                                    </div>`;
                                document.body.insertAdjacentHTML('beforeend', modalHTML);
                                modal = document.getElementById('facturaDuplicadaModal');
                            }

                            const infoDiv = document.getElementById('facturaDuplicadaInfo');
                            if (!infoDiv) {
                                console.error('No se encontró el div de información');
                                return;
                            }

                            infoDiv.innerHTML = `
                                <div class="factura-info">
                                    <h6 class="mb-2">Se ha encontrado una factura con el mismo folio en otra rendición:</h6>
                                    <p class="mb-1"><strong>Rendición:</strong> ${data.data.rendicion_descripcion}</p>
                                    <p class="mb-1"><strong>Fecha:</strong> ${new Date(data.data.rendicion_fecha).toLocaleDateString()}</p>
                                    <p class="mb-0"><strong>Monto:</strong> $${parseInt(data.data.monto).toLocaleString()}</p>
                                </div>
                                <p class="mb-0 mt-3">¿Está seguro que desea continuar con el registro de esta factura?</p>
                            `;

                            // Mostrar el modal
                            modal.style.display = 'block';
                            modal.classList.add('show');

                            // Configurar eventos del modal
                            const closeModal = () => {
                                modal.style.display = 'none';
                                modal.classList.remove('show');
                            };

                            const handleContinuar = () => {
                                closeModal();
                                folioInput.setCustomValidity('');
                                folioInput.reportValidity();
                            };

                            const handleCancelar = () => {
                                closeModal();
                                folioInput.value = '';
                                folioInput.focus();
                            };

                            // Remover eventos anteriores si existen
                            const btnContinuar = modal.querySelector('#btnContinuarFactura');
                            const btnCancelar = modal.querySelector('#btnCancelarFactura');
                            const closeBtn = modal.querySelector('.btn-close');

                            if (btnContinuar) {
                                btnContinuar.removeEventListener('click', handleContinuar);
                                btnContinuar.addEventListener('click', handleContinuar);
                            }

                            if (btnCancelar) {
                                btnCancelar.removeEventListener('click', handleCancelar);
                                btnCancelar.addEventListener('click', handleCancelar);
                            }

                            if (closeBtn) {
                                closeBtn.removeEventListener('click', closeModal);
                                closeBtn.addEventListener('click', closeModal);
                            }

                            // Cerrar modal al hacer clic fuera
                            const handleOutsideClick = (event) => {
                                if (event.target === modal) {
                                    handleCancelar();
                                }
                            };
                            window.removeEventListener('click', handleOutsideClick);
                            window.addEventListener('click', handleOutsideClick);
                        } else {
                            folioInput.setCustomValidity('');
                            folioInput.reportValidity();
                        }
                    })
                    .catch(error => {
                        console.error('Error al verificar factura:', error);
                        // En caso de error, permitir continuar
                        folioInput.setCustomValidity('');
                        folioInput.reportValidity();
                    });
            }
        };
        
        // Añadir evento para verificar al cambiar el campo
        folioInput.addEventListener('change', checkFacturaDuplicada);
        
        // Verificar inmediatamente si ya hay valor
        if (folioInput.value.trim()) {
            checkFacturaDuplicada();
        }
    }

    // Inicializar los campos de factura para las líneas existentes
    document.querySelectorAll('.tipo-documento-select').forEach(select => {
        toggleFacturaFields(select);
    });
});

function toggleTerceroFields(value) {
    const terceroFields = document.getElementById('tercero_fields');
    const terceroSelect = document.getElementById('tercero_select');
    const nuevoTerceroFields = document.getElementById('nuevo_tercero_fields');
    
    if (value === 'tercero') {
        terceroFields.style.display = 'block';
        terceroSelect.required = true;
    } else {
        terceroFields.style.display = 'none';
        terceroSelect.required = false;
        terceroSelect.value = '';
        nuevoTerceroFields.style.display = 'none';
        clearTerceroFields();
    }
}

function toggleNuevoTercero(value) {
    const nuevoTerceroFields = document.getElementById('nuevo_tercero_fields');
    const camposNuevoTercero = nuevoTerceroFields.querySelectorAll('input');
    
    if (value === 'nuevo') {
        nuevoTerceroFields.style.display = 'block';
        camposNuevoTercero.forEach(campo => {
            campo.required = true;
            campo.disabled = false;
        });
    } else {
        nuevoTerceroFields.style.display = 'none';
        camposNuevoTercero.forEach(campo => {
            campo.required = false;
            campo.disabled = true;
            campo.value = '';
        });
    }
}

function clearTerceroFields() {
    const campos = ['tercero_nombre', 'tercero_apellido', 'tercero_rut', 'tercero_cargo'];
    campos.forEach(campo => {
        const elemento = document.getElementById(campo);
        if (elemento) {
            elemento.value = '';
            elemento.required = false;
            elemento.disabled = true;
        }
    });
}

// Inicializar los campos cuando se carga la página
document.addEventListener('DOMContentLoaded', function() {
    const tipoRendicion = document.getElementById('tipo_rendicion');
    const terceroSelect = document.getElementById('tercero_select');
    
    if (tipoRendicion) {
        toggleTerceroFields(tipoRendicion.value);
        if (tipoRendicion.value === 'tercero' && terceroSelect) {
            toggleNuevoTercero(terceroSelect.value);
        }
    }
});

// Función para validar RUT chileno
function validarRut(rut) {
    // Limpiar el RUT de puntos y guión
    rut = rut.replace(/\./g, '').replace(/-/g, '');
    
    // Obtener dígito verificador
    const dv = rut.slice(-1).toUpperCase();
    const rutNumero = parseInt(rut.slice(0, -1));
    
    // Calcular dígito verificador esperado
    let suma = 0;
    let multiplicador = 2;
    let rutReverso = String(rutNumero).split('').reverse();
    
    for (let i = 0; i < rutReverso.length; i++) {
        suma += parseInt(rutReverso[i]) * multiplicador;
        multiplicador = multiplicador === 7 ? 2 : multiplicador + 1;
    }
    
    const dvEsperado = 11 - (suma % 11);
    const dvCalculado = dvEsperado === 11 ? '0' : dvEsperado === 10 ? 'K' : String(dvEsperado);
    
    return dvCalculado === dv;
}

// Agregar validación al formulario
document.getElementById('form-rendicion').addEventListener('submit', async function(e) {
    e.preventDefault(); // Prevenir el envío por defecto
    
    // Validar el formulario
    const tipoRendicion = document.getElementById('tipo_rendicion').value;
    if (tipoRendicion === 'tercero') {
        const terceroId = document.getElementById('tercero_select').value;
        if (terceroId === 'nuevo') {
            // Validar campos del nuevo tercero
            const nombre = document.getElementById('tercero_nombre').value.trim();
            const apellido = document.getElementById('tercero_apellido').value.trim();
            const rut = document.getElementById('tercero_rut').value.trim();
            
            if (!nombre || !apellido || !rut) {
                alert('Por favor complete todos los campos del nuevo usuario');
                return;
            }
            
            if (!validarRut(rut)) {
                alert('El RUT ingresado no es válido');
                return;
            }
        } else if (!terceroId) {
            alert('Por favor seleccione un usuario tercero');
            return;
        }
    }
    
    // Validar facturas duplicadas
    const facturaInputs = document.querySelectorAll('input[name="folio_factura[]"]');
    for (const input of facturaInputs) {
        if (input.value.trim()) {
            const isValid = await checkFacturaDuplicada(input);
            if (!isValid) {
                return;
            }
        }
    }
    
    // Si todo está bien, enviar el formulario
    this.submit();
});

// Inicialización específica para el formulario de edición
document.addEventListener('DOMContentLoaded', function() {
    const tipoRendicion = document.getElementById('tipo_rendicion');
    const terceroFields = document.getElementById('tercero_fields');
    const terceroSelect = document.getElementById('tercero_select');
    
    if (tipoRendicion && terceroFields && terceroSelect) {
        // Configurar los event listeners
        tipoRendicion.addEventListener('change', function() {
            toggleTerceroFields(this.value);
        });
        
        terceroSelect.addEventListener('change', function() {
            toggleNuevoTercero(this.value);
        });
        
        // Inicializar el estado actual
        if (tipoRendicion.value === 'tercero') {
            terceroFields.style.display = 'block';
            terceroSelect.required = true;
            
            if (terceroSelect.value === 'nuevo') {
                document.getElementById('nuevo_tercero_fields').style.display = 'block';
                const camposNuevoTercero = document.getElementById('nuevo_tercero_fields').querySelectorAll('input');
                camposNuevoTercero.forEach(campo => campo.required = true);
            }
        }
    }
});

// Función global para verificar facturas duplicadas
window.checkFacturaDuplicada = async function(input) {
    const folio = input.value.trim();
    if (!folio) return true;

    // Verificar si es una factura existente en la rendición actual
    const row = input.closest('tr');
    if (row && row.hasAttribute('data-existing')) {
        return true; // No verificar facturas existentes
    }

    try {
        const rendicionId = document.querySelector('form[action*="action=edit"]') ? 
                          new URLSearchParams(document.querySelector('form[action*="action=edit"]').action).get('id') : 
                          null;
        
        const url = `rendiciones.php?check_factura=1&folio=${encodeURIComponent(folio)}${rendicionId ? '&rendicion_id=' + rendicionId : ''}`;
        
        const response = await fetch(url);
        if (!response.ok) {
            throw new Error('Error en la respuesta del servidor');
        }
        
        const data = await response.json();
        
        if (data.error) {
            console.error('Error del servidor:', data.message);
            return true; // Permitir continuar en caso de error
        }

        if (data.exists) {
            return new Promise((resolve) => {
                // ... resto del código del modal ...
            });
        }
        return true;
    } catch (error) {
        console.error('Error al verificar factura:', error);
        return true; // Permitir continuar en caso de error
    }
};

// Inicializar validación de facturas
document.getElementById('form-rendicion').addEventListener('submit', async function(e) {
    e.preventDefault();
    
    // Validar el formulario
    const tipoRendicion = document.getElementById('tipo_rendicion').value;
    if (tipoRendicion === 'tercero') {
        const terceroId = document.getElementById('tercero_select').value;
        if (terceroId === 'nuevo') {
            // Validar campos del nuevo tercero
            const nombre = document.getElementById('tercero_nombre').value.trim();
            const apellido = document.getElementById('tercero_apellido').value.trim();
            const rut = document.getElementById('tercero_rut').value.trim();
            
            if (!nombre || !apellido || !rut) {
                alert('Por favor complete todos los campos del nuevo usuario');
                return;
            }
            
            if (!validarRut(rut)) {
                alert('El RUT ingresado no es válido');
                return;
            }
        } else if (!terceroId) {
            alert('Por favor seleccione un usuario tercero');
            return;
        }
    }
    
    // Validar facturas duplicadas
    const facturaInputs = document.querySelectorAll('input[name="folio_factura[]"]');
    for (const input of facturaInputs) {
        if (input.value.trim()) {
            const isValid = await window.checkFacturaDuplicada(input);
            if (!isValid) {
                return;
            }
        }
    }
    
    // Si todo está bien, enviar el formulario
    this.submit();
});

function agregarNuevaLinea() {
    const tbody = document.querySelector('#lineas-detalle tbody');
    const tr = document.createElement('tr');
    // Las nuevas líneas no tendrán el atributo data-existing
    tr.innerHTML = templateNuevaLinea;
    tbody.appendChild(tr);
    initializeRow(tr);
}

// Inicializar las filas existentes
document.querySelectorAll('#lineas-detalle tbody tr[data-existing="true"]').forEach(row => {
    initializeRow(row);
});

// Inicializar las filas nuevas
document.querySelectorAll('#lineas-detalle tbody tr:not([data-existing])').forEach(row => {
    initializeRow(row);
});

// Checkbox maestro para seleccionar/deseleccionar todos
const checkTodas = document.getElementById('check-todas');
if (checkTodas) {
    checkTodas.addEventListener('change', function() {
        document.querySelectorAll('.check-rendicion').forEach(cb => {
            cb.checked = checkTodas.checked;
        });
    });
}

function ordenar(col) {
    const url = new URL(window.location.href);
    let dir = url.searchParams.get('order_dir') === 'ASC' ? 'DESC' : 'ASC';
    url.searchParams.set('order_by', col);
    url.searchParams.set('order_dir', dir);
    window.location.href = url.toString();
    return false;
}
</script>

<style>
.file-preview-container {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(120px, 1fr));
    gap: 10px;
    margin: 15px 0;
}

.file-preview {
    position: relative;
    border: 1px solid #ddd;
    border-radius: 4px;
    padding: 8px;
    text-align: center;
    background: #fff;
    transition: all 0.3s ease;
}

.file-preview:hover {
    box-shadow: 0 2px 5px rgba(0,0,0,0.1);
    transform: translateY(-2px);
}

.file-preview img {
    max-width: 100%;
    height: 80px;
    object-fit: cover;
    border-radius: 4px;
}

.file-preview .file-icon {
    font-size: 36px;
    color: #666;
    margin-bottom: 5px;
}

.file-preview .file-name {
    font-size: 11px;
    margin-top: 5px;
    word-break: break-all;
    color: #333;
}

.file-preview .file-actions {
    position: absolute;
    top: 2px;
    right: 2px;
    display: none;
    z-index: 1;
}

.file-preview:hover .file-actions {
    display: block;
}

.file-preview .file-actions button {
    background: rgba(255,255,255,0.9);
    border: none;
    border-radius: 50%;
    width: 20px;
    height: 20px;
    padding: 0;
    cursor: pointer;
    transition: all 0.2s;
    margin-left: 2px;
}

.file-preview .file-actions button:hover {
    background: #fff;
    transform: scale(1.1);
}

.file-preview .file-actions button i {
    font-size: 14px;
    line-height: 1;
}

.mt-2 {
    margin-top: 8px;
}

.documento-fields {
    min-width: 200px;
}

.factura-fields {
    transition: all 0.3s ease;
}

.factura-fields input {
    margin-top: 8px;
}

.order-arrows { font-size: 12px; margin-left: 2px; color: #1e88e5; }
.order-arrows.active { color: #1565c0; font-weight: bold; text-shadow: 0 1px 2px #b3d1f7; }
</style>

<?php
// Incluir el footer
require_once 'includes/footer.php';
?>

<!-- Modal para factura duplicada -->
<div id="facturaDuplicadaModal" class="modal fade" tabindex="-1" role="dialog">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header bg-warning">
                <h5 class="modal-title">¡Factura Duplicada!</h5>
            </div>
            <div class="modal-body">
                <div id="facturaDuplicadaInfo"></div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-warning" id="btnContinuarFactura">
                    Continuar de todos modos
                </button>
                <button type="button" class="btn btn-secondary" id="btnCancelarFactura">
                    Cancelar
                </button>
            </div>
        </div>
    </div>
</div>

<style>
.modal {
    display: none;
    position: fixed;
    z-index: 1050;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0,0,0,0.5);
    padding: 20px;
}

.modal.show {
    display: block;
}

.modal-dialog {
    position: relative;
    width: auto;
    margin: 1.75rem auto;
    max-width: 500px;
}

.modal-content {
    position: relative;
    background-color: #fff;
    border-radius: 0.3rem;
    box-shadow: 0 0.5rem 1rem rgba(0,0,0,0.15);
}

.modal-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 1rem;
    border-bottom: 1px solid #dee2e6;
}

.modal-body {
    padding: 1rem;
}

.modal-footer {
    display: flex;
    justify-content: flex-end;
    padding: 1rem;
    border-top: 1px solid #dee2e6;
    gap: 0.5rem;
}

.btn-close {
    background: transparent;
    border: 0;
    font-size: 1.5rem;
    padding: 0.5rem;
    cursor: pointer;
}
</style>

<script>
// Función para inicializar el modal y sus eventos
function initializeFacturaModal() {
    const modal = document.getElementById('facturaDuplicadaModal');
    const btnContinuar = document.getElementById('btnContinuarFactura');
    const btnCancelar = document.getElementById('btnCancelarFactura');
    
    // Solo inicializar si los elementos existen
    if (modal && btnContinuar && btnCancelar) {
        // Cerrar modal al hacer clic fuera
        modal.addEventListener('click', (e) => {
            if (e.target === modal) {
                modal.style.display = 'none';
                modal.classList.remove('show');
            }
        });
    }
}

function initializeFacturaValidation(facturaFields) {
    const folioInput = facturaFields.querySelector('input[name="folio_factura[]"]');
    if (!folioInput) return;

    let currentValidation = Promise.resolve();

    const checkFacturaDuplicada = async () => {
        const folio = folioInput.value.trim();
        if (!folio) return;

        try {
            const rendicionId = document.querySelector('form[action*="action=edit"]') ? 
                              new URLSearchParams(document.querySelector('form[action*="action=edit"]').action).get('id') : 
                              null;
            
            const url = `rendiciones.php?check_factura=1&folio=${encodeURIComponent(folio)}${rendicionId ? '&rendicion_id=' + rendicionId : ''}`;
            
            const response = await fetch(url);
            const data = await response.json();

            if (data.exists) {
                return new Promise((resolve) => {
                    const modal = document.getElementById('facturaDuplicadaModal');
                    const infoDiv = document.getElementById('facturaDuplicadaInfo');
                    
                    // Verificar que los elementos existan
                    if (!modal || !infoDiv) {
                        console.warn('Elementos del modal no encontrados');
                        resolve(true);
                        return;
                    }

                    infoDiv.innerHTML = `
                        <div class="factura-info">
                            <h6 class="mb-2">Se ha encontrado una factura con el mismo folio en otra rendición:</h6>
                            <p class="mb-1"><strong>Rendición:</strong> ${data.data.rendicion_descripcion}</p>
                            <p class="mb-1"><strong>Fecha:</strong> ${new Date(data.data.rendicion_fecha).toLocaleDateString()}</p>
                            <p class="mb-0"><strong>Monto:</strong> $${parseInt(data.data.monto).toLocaleString()}</p>
                        </div>
                        <p class="mb-0">¿Está seguro que desea continuar con el registro de esta factura?</p>
                    `;

                    modal.style.display = 'block';
                    modal.classList.add('show');

                    const btnContinuar = document.getElementById('btnContinuarFactura');
                    const btnCancelar = document.getElementById('btnCancelarFactura');

                    // Verificar que los botones existan
                    if (!btnContinuar || !btnCancelar) {
                        console.warn('Botones del modal no encontrados');
                        resolve(true);
                        return;
                    }

                    const cleanup = () => {
                        modal.style.display = 'none';
                        modal.classList.remove('show');
                        btnContinuar.removeEventListener('click', handleContinuar);
                        btnCancelar.removeEventListener('click', handleCancelar);
                    };

                    const handleContinuar = () => {
                        cleanup();
                        folioInput.setCustomValidity('');
                        resolve(true);
                    };

                    const handleCancelar = () => {
                        cleanup();
                        folioInput.value = '';
                        folioInput.focus();
                        resolve(false);
                    };

                    btnContinuar.addEventListener('click', handleContinuar);
                    btnCancelar.addEventListener('click', handleCancelar);

                    modal.addEventListener('click', (e) => {
                        if (e.target === modal) {
                            handleCancelar();
                        }
                    });
                });
            }
            return true;
        } catch (error) {
            console.error('Error al verificar factura:', error);
            return true;
        }
    };

    folioInput.addEventListener('change', async () => {
        currentValidation = checkFacturaDuplicada();
        const isValid = await currentValidation;
        if (!isValid) {
            // No establecer mensaje de error aquí, ya que el modal manejará la interacción
            folioInput.setCustomValidity('');
        } else {
            folioInput.setCustomValidity('');
        }
        folioInput.reportValidity();
    });

    folioInput.closest('form')?.addEventListener('submit', async (e) => {
        if (folioInput.value.trim()) {
            const isValid = await currentValidation;
            if (!isValid) {
                e.preventDefault();
                // No establecer mensaje de error aquí tampoco
                folioInput.setCustomValidity('');
                folioInput.reportValidity();
            }
        }
    });
}

// Inicializar la validación y el modal solo cuando sea necesario
document.addEventListener('DOMContentLoaded', () => {
    // Solo inicializar si estamos en una página de creación o edición
    const isFormPage = document.querySelector('form[action*="action=new"]') || 
                      document.querySelector('form[action*="action=edit"]');
    
    if (isFormPage) {
        initializeFacturaModal();
        document.querySelectorAll('.factura-fields').forEach(facturaFields => {
            initializeFacturaValidation(facturaFields);
        });
    }
});
</script>