<?php
/**
 * Página de inicio de sesión
 * Maneja la autenticación de usuarios y la creación de sesiones
 */

// Incluir archivos de configuración y funciones
require_once 'includes/config.php';
require_once 'includes/functions.php';

// Establecer codificación UTF-8
header('Content-Type: text/html; charset=utf-8');

// Iniciar la sesión
session_start();

// Si el usuario ya está autenticado, redirigir al dashboard
if (isset($_SESSION['user_id'])) {
    header('Location: dashboard.php');
    exit;
}

// Inicializar variables
$error = '';
$email = '';

// Procesar el formulario de inicio de sesión
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email = trim($_POST['email']);
    $password = $_POST['password'];

    // Validar campos
    if (empty($email) || empty($password)) {
        $error = 'Por favor, complete todos los campos.';
    } else {
        try {
            // Conexión a la base de datos con manejo mejorado de errores
            $conn = new PDO("mysql:host=$db_host;dbname=$db_name;charset=utf8mb4", $db_user, $db_pass, $db_options);
            
            // Buscar usuario por correo
            $stmt = $conn->prepare("SELECT id, rut, nombre, apellido, email, password, rol, activo FROM users WHERE email = :email");
            $stmt->bindParam(':email', $email);
            $stmt->execute();
            
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($user && $user['activo'] == 1) {
    // Si DEBUG_MODE es verdadero, omitir la verificación de contraseña
    if (DEBUG_MODE || password_verify($password, $user['password'])) {
        // Iniciar sesión
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['user_rut'] = $user['rut'];
        $_SESSION['user_nombre'] = $user['nombre'];
        $_SESSION['user_apellido'] = $user['apellido'];
        $_SESSION['user_email'] = $user['email'];
        $_SESSION['user_rol'] = $user['rol'];
                    
                    // Obtener permisos del usuario
                    $stmt = $conn->prepare("
                        SELECT p.nombre 
                        FROM permisos p 
                        INNER JOIN user_permisos up ON p.id = up.permiso_id 
                        WHERE up.user_id = :user_id
                    ");
                    $stmt->bindParam(':user_id', $user['id']);
                    $stmt->execute();
                    
                    $permisos = [];
                    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                        $permisos[] = $row['nombre'];
                    }
                    
                    $_SESSION['user_permisos'] = $permisos;
                    
                    // Registrar actividad de inicio de sesión
                    $ip = $_SERVER['REMOTE_ADDR'] ?? 'Desconocida';
                    
                    $stmt = $conn->prepare("
                        INSERT INTO historial_cambios (user_id, tipo, descripcion, fecha_hora, ip_address)
                        VALUES (:user_id, 'login', 'Inicio de sesión exitoso', NOW(), :ip_address)
                    ");
                    
                    $stmt->bindParam(':user_id', $user['id']);
                    $stmt->bindParam(':ip_address', $ip);
                    $stmt->execute();
                    
                    // Redirigir al dashboard
                    header('Location: dashboard.php');
                    exit;
                } else {
                    $error = 'Credenciales inválidas.';
                }
            } else {
                $error = 'Credenciales inválidas o cuenta desactivada.';
            }
        } catch (PDOException $e) {
            // Log del error para el administrador, mensaje genérico para el usuario
            error_log('Error en login.php: ' . $e->getMessage());
            $error = 'Error de conexión. Por favor, inténtelo más tarde.';
        }
    }
}

// Determinar si estamos en modo desarrollo (para mostrar credenciales de ejemplo)
$is_dev_mode = defined('DEBUG_MODE') && DEBUG_MODE === true;
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Iniciar Sesión | Sistema de Rendiciones</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/icon?family=Material+Icons" rel="stylesheet">
    <style>
        :root {
            --primary: #1e88e5;
            --primary-light: #6ab7ff;
            --primary-dark: #005cb2;
            --secondary: #26a69a;
            --success: #66bb6a;
            --warning: #ffa726;
            --danger: #ef5350;
            --light: #f5f5f5;
            --dark: #263238;
            --gray: #607d8b;
            --white: #ffffff;
            --shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            --border-radius: 10px;
            --transition: all 0.3s ease;
        }
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        body {
            background-color: var(--light);
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
            padding: 20px;
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
        }
        
        .login-container {
            width: 100%;
            max-width: 400px;
            background: var(--white);
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            overflow: hidden;
            transition: var(--transition);
        }
        
        .login-container:hover {
            box-shadow: 0 10px 20px rgba(0, 0, 0, 0.15);
            transform: translateY(-5px);
        }
        
        .login-header {
            background: linear-gradient(to right, var(--primary-dark), var(--primary));
            color: var(--white);
            padding: 30px 20px;
            text-align: center;
        }
        
        .logo {
            display: flex;
            align-items: center;
            justify-content: center;
            margin-bottom: 20px;
        }
        
        .logo-icon {
            font-size: 40px;
            margin-right: 10px;
            color: rgba(255, 255, 255, 0.9);
        }
        
        .login-header h1 {
            font-size: 24px;
            font-weight: 600;
            margin-bottom: 5px;
        }
        
        .login-header p {
            font-size: 14px;
            opacity: 0.9;
        }
        
        .login-form {
            padding: 30px;
        }
        
        .form-group {
            margin-bottom: 20px;
            position: relative;
        }
        
        .form-label {
            display: block;
            font-size: 14px;
            font-weight: 600;
            margin-bottom: 8px;
            color: var(--dark);
        }
        
        .form-control {
            width: 100%;
            padding: 12px 15px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 15px;
            transition: var(--transition);
        }
        
        .form-control:focus {
            border-color: var(--primary);
            outline: none;
            box-shadow: 0 0 0 3px rgba(30, 136, 229, 0.25);
        }
        
        .btn {
            width: 100%;
            padding: 12px;
            border: none;
            border-radius: 6px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: var(--transition);
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .btn-primary {
            background-color: var(--primary);
            color: var(--white);
        }
        
        .btn-primary:hover {
            background-color: var(--primary-dark);
        }
        
        .btn-icon {
            margin-right: 8px;
            font-size: 18px;
        }
        
        .alert {
            padding: 12px 15px;
            border-radius: var(--border-radius);
            margin-bottom: 20px;
            font-size: 14px;
            display: flex;
            align-items: center;
        }
        
        .alert-danger {
            background-color: rgba(239, 83, 80, 0.2);
            color: var(--danger);
            border: 1px solid rgba(239, 83, 80, 0.3);
        }
        
        .alert-icon {
            margin-right: 10px;
            font-size: 20px;
        }
        
        .footer-text {
            text-align: center;
            margin-top: 20px;
            font-size: 13px;
            color: var(--gray);
            line-height: 1.6;
        }
        
        .dev-credentials {
            margin-top: 15px;
            padding: 10px;
            background-color: rgba(255, 167, 38, 0.2);
            border-radius: 6px;
            color: #e57373;
            font-weight: 500;
        }
        
        .dev-credentials p {
            margin: 5px 0;
            font-size: 12px;
        }
        
        /* Responsive adjustments */
        @media (max-width: 430px) {
            .login-container {
                max-width: 100%;
            }
            
            .login-header, .login-form {
                padding: 20px 15px;
            }
        }
    </style>
</head>
<body>
    <div class="login-container">
        <div class="login-header">
            <div class="logo">
                <i class="material-icons logo-icon">receipt_long</i>
                <span style="font-size: 28px; font-weight: 700;">SisRendiciones</span>
            </div>
            <h1>Iniciar Sesión</h1>
            <p>Ingrese sus credenciales para acceder al sistema</p>
        </div>
        
        <div class="login-form">
            <?php if (!empty($error)): ?>
            <div class="alert alert-danger">
                <i class="material-icons alert-icon">error_outline</i>
                <span><?php echo $error; ?></span>
            </div>
            <?php endif; ?>
            
            <form method="POST" action="" autocomplete="off">
                <div class="form-group">
                    <label for="email" class="form-label">Correo Electrónico</label>
                    <input type="email" id="email" name="email" class="form-control" 
                           placeholder="ejemplo@empresa.cl" required 
                           value="<?php echo htmlspecialchars($email); ?>">
                </div>
                
                <div class="form-group">
                    <label for="password" class="form-label">Contraseña</label>
                    <input type="password" id="password" name="password" class="form-control" 
                           placeholder="••••••••" required>
                </div>
                
                <button type="submit" class="btn btn-primary">
                    <i class="material-icons btn-icon">login</i>
                    Ingresar al Sistema
                </button>
            </form>
            
            <div class="footer-text">
                <p>&copy; <?php echo date('Y'); ?> SisRendiciones - Todos los derechos reservados</p>
                <p>Contacte al administrador para restablecer su contraseña</p>
                
                <?php if ($is_dev_mode): ?>
                <!-- Credenciales de desarrollo - Solo visible en modo desarrollo -->
                <div class="dev-credentials">
                    <p><strong>Credenciales de desarrollo:</strong></p>
                    <p>Admin: admin@sisrendiciones.cl | Contraseña: password</p>
                    <p>Usuario: juan.diaz@empresa.cl | Contraseña: password</p>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</body>
</html>