<?php
/**
 * Generador de PDFs para solicitudes de repuestos
 * Permite exportar solicitudes a formato PDF con estilos mejorados
 * Versión actualizada con soporte para nuevos campos
 */

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Incluir archivos necesarios
require_once 'includes/config.php';
require_once 'includes/functions.php';
require_once 'tfpdf/tfpdf.php';

// Verificar sesión y obtener datos del usuario
verificarSesion();
$user_id = $_SESSION['user_id'];
$user_rol = $_SESSION['user_rol'] ?? '';
$is_admin = ($user_rol === 'admin' || $user_rol === 'super_admin');

// Obtener parámetros de la URL
$id = isset($_GET['id']) ? intval($_GET['id']) : 0;
$save = isset($_GET['save']) ? true : false;
$redirect = isset($_GET['redirect']) ? $_GET['redirect'] : '';

if (empty($id)) {
    die('Parámetros incorrectos');
}

// Crear directorio temporal si no existe
if (!file_exists('temp')) {
    mkdir('temp', 0777, true);
}

// Definir constantes de la empresa
if (!defined('COMPANY_NAME')) {
    define('COMPANY_NAME', 'COMERCIAL FIGARO LIMITADA');
}
define('COMPANY_ADDRESS', 'Puerta del Sol 180 Of.02');
define('COMPANY_RUT', '76.080.987-K');
define('COMPANY_PHONE', '+562 32212511');
define('COMPANY_EMAIL', 'CONTACTO@COMERCIALFIGARO.CL');
define('COMPANY_BUSINESS', 'Comercializadora de Café y otros Productos Gourmet');
define('COMPANY_LOGO', 'assets/img/logo.png');

// Clase personalizada para generar el PDF
class SolicitudPDF extends tFPDF {
    private $logo = null;
    public $titulo;
    public $codigo;
    public $footer_text;
    public $solicitud;

    function __construct($solicitud = [], $titulo = '', $codigo = '', $orientation = 'P', $unit = 'mm', $size = 'A4') {
        setlocale(LC_ALL, 'es_ES.UTF-8');
        parent::__construct($orientation, $unit, $size);
        $this->AddFont('DejaVu', '', 'DejaVuSans.ttf', true);
        $this->AddFont('DejaVu', 'B', 'DejaVuSans-Bold.ttf', true);
        $this->SetFont('DejaVu', '', 10);
        
        if (file_exists(COMPANY_LOGO)) {
            $this->logo = COMPANY_LOGO;
        }
        
        $this->footer_text = 'Documento generado por ' . APP_NAME . ' - ' . date('d/m/Y H:i:s');
        $this->SetMargins(15, 15, 15);
        $this->solicitud = $solicitud;
        $this->titulo = $titulo;
        $this->codigo = $codigo;
        $this->AddPage();
    }

    function Header() {
        $y_initial = $this->GetY();
        if ($this->logo) {
            $this->Image($this->logo, (210 - 55) / 2, 15, 55); // Centrar logo
            $this->SetY($y_initial + 35);
        }

        // Información de la empresa
        $this->SetFont('DejaVu', 'B', 12);
        $this->SetTextColor(0, 51, 102); // Azul corporativo
        $this->Cell(0, 6, COMPANY_NAME, 0, 1, 'C');
        $this->Ln(2);

        $this->SetFont('DejaVu', '', 8);
        $this->SetTextColor(80, 80, 80);
        $this->Cell(0, 4, 'RUT: ' . COMPANY_RUT, 0, 1, 'C');
        $this->Cell(0, 4, COMPANY_ADDRESS, 0, 1, 'C');
        $this->Cell(0, 4, 'Tel: ' . COMPANY_PHONE . ' | Email: ' . COMPANY_EMAIL, 0, 1, 'C');
        $this->Cell(0, 4, 'Giro: ' . COMPANY_BUSINESS, 0, 1, 'C');

        // Cuadro con el código
        $this->SetFont('DejaVu', 'B', 9);
        $this->SetTextColor(255, 255, 255);
        $this->SetDrawColor(80, 80, 80);
        $this->SetFillColor(0, 51, 102); // Azul corporativo
        $code_text = 'COD: ' . ($this->codigo ?? 'No especificado');
        $code_width = min($this->GetStringWidth($code_text) + 10, 50);
        $this->SetX(195 - $code_width);
        $this->Cell($code_width, 8, $code_text, 1, 1, 'C', true);

        $this->Ln(5);

        // Título principal
        $this->SetDrawColor(0, 51, 102);
        $this->Line(15, $this->GetY(), 195, $this->GetY());
        $this->Ln(1);
        $this->SetFont('DejaVu', 'B', 16);
        $this->SetTextColor(0, 51, 102);
        $this->Cell(0, 10, $this->titulo, 0, 1, 'C');
        $this->Ln(1);
        $this->SetDrawColor(0, 51, 102);
        $this->Line(15, $this->GetY(), 195, $this->GetY());
        $this->Ln(5);
        $this->SetTextColor(0, 0, 0);
    }

    function Footer() {
        $this->SetY(-15);
        $this->SetDrawColor(0, 51, 102);
        $this->Line(15, $this->GetY(), 195, $this->GetY());
        $this->Ln(1);
        $this->SetFont('DejaVu', '', 8);
        $this->SetTextColor(120, 120, 120);
        $this->Cell(0, 10, $this->footer_text, 0, 0, 'L');
        $this->Cell(0, 10, 'Página ' . $this->PageNo() . ' de {nb}', 0, 0, 'R');
    }

    function AddSection($title, $border = 0, $fill = false) {
        $this->SetFont('DejaVu', 'B', 11);
        $this->SetFillColor(240, 248, 255); // Azul muy claro
        $this->SetTextColor(0, 51, 102); // Azul corporativo
        $this->Cell(0, 8, $title, $border, 1, 'L', $fill);
        $this->SetFont('DejaVu', '', 10);
        $this->SetTextColor(0, 0, 0);
        $this->Ln(2);
    }

    function FormatoMoneda($monto) {
        return '$' . number_format($monto, 0, ',', '.');
    }

    function GetEstadoColor($estado_id) {
        switch ($estado_id) {
            case 1: return [255, 193, 7]; // Amarillo - Enviada
            case 2: return [52, 152, 219]; // Azul - En revisión
            case 3: return [40, 167, 69]; // Verde - Aprobada
            case 4: return [220, 53, 69]; // Rojo - Rechazada
            case 5: return [108, 117, 125]; // Gris - Pagada
            default: return [0, 0, 0]; // Negro - Default
        }
    }

    function AddSignatureSection() {
        $this->Ln(15);
        $this->SetFont('DejaVu', '', 10);
        $this->Cell(90, 6, '____________________________', 0, 0, 'C');
        $this->Cell(90, 6, '____________________________', 0, 1, 'C');
        $this->Cell(90, 6, 'Firma Solicitante', 0, 0, 'C');
        $this->Cell(90, 6, 'Firma Aprobador', 0, 1, 'C');
    }
    
    function AddInformationTable($leftData, $rightData) {
        $maxWidth = 85;
        $this->SetDrawColor(220, 220, 220);
        $this->SetFillColor(248, 248, 248);
        
        $maxLeftRows = count($leftData);
        $maxRightRows = count($rightData);
        $maxRows = max($maxLeftRows, $maxRightRows);
        
        for ($i = 0; $i < $maxRows; $i++) {
            // Reset Y position for both columns
            $startY = $this->GetY();
            
            // Left column
            if ($i < $maxLeftRows) {
                $this->SetTextColor(80, 80, 80);
                $this->SetFont('DejaVu', 'B', 9);
                $this->Cell(30, 7, $leftData[$i][0] . ':', 1, 0, 'L', true);
                
                $this->SetTextColor(0, 0, 0);
                $this->SetFont('DejaVu', '', 9);
                
                // Special styling for estado (status)
                if ($leftData[$i][0] == 'Estado') {
                    $estado_id = $leftData[$i][2] ?? 1;
                    $colorRGB = $this->GetEstadoColor($estado_id);
                    $this->SetTextColor($colorRGB[0], $colorRGB[1], $colorRGB[2]);
                    $this->SetFont('DejaVu', 'B', 9);
                }
                
                $this->Cell(55, 7, $leftData[$i][1], 1, 0, 'L');
                $this->SetTextColor(0, 0, 0);
            } else {
                $this->Cell(85, 7, '', 0, 0);
            }
            
            $this->Cell(5, 7, '', 0, 0); // Espaciado entre columnas
            
            // Right column
            if ($i < $maxRightRows) {
                $this->SetTextColor(80, 80, 80);
                $this->SetFont('DejaVu', 'B', 9);
                $this->Cell(30, 7, $rightData[$i][0] . ':', 1, 0, 'L', true);
                
                $this->SetTextColor(0, 0, 0);
                $this->SetFont('DejaVu', '', 9);
                $this->Cell(55, 7, $rightData[$i][1], 1, 0, 'L');
            }
            
            $this->Ln();
        }
    }
}

try {
    // Conexión a la base de datos
    $conn = new PDO("mysql:host=$db_host;dbname=$db_name;charset=utf8mb4", $db_user, $db_pass, $db_options);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // Obtener datos de la solicitud con los nuevos campos
    $stmt = $conn->prepare("
        SELECT sr.*, 
               m.codigo as maquina_codigo, m.nombre as maquina_nombre,
               m.marca as maquina_marca, m.modelo as maquina_modelo,
               m.serie_manual, m.numero_serie, s.codigo_serie,
               u.nombre as usuario_nombre, u.apellido as usuario_apellido,
               u.rut as usuario_rut, u.cargo as usuario_cargo,
               e.nombre as estado_nombre, e.color as estado_color,
               rev.nombre as revisor_nombre, rev.apellido as revisor_apellido
        FROM solicitud_repuestos sr
        JOIN maquinas m ON sr.maquina_id = m.id
        JOIN users u ON sr.user_id = u.id
        JOIN estados e ON sr.estado_id = e.id
        LEFT JOIN series_maquinas s ON m.serie_id = s.id
        LEFT JOIN users rev ON sr.revisor_id = rev.id
        WHERE sr.id = :id
    ");
    
    // Para depuración
    if (defined('DEBUG_MODE') && DEBUG_MODE) {
        error_log("Consultando solicitud ID: " . $id);
    }
    $stmt->bindParam(':id', $id);
    $stmt->execute();
    $solicitud = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$solicitud) {
        die('Solicitud no encontrada');
    }
    
    // Verificar permisos
    if (!$is_admin && $solicitud['user_id'] != $user_id) {
        die('No tiene permisos para ver esta solicitud');
    }
    
    // Verificar estado (solo solicitudes aprobadas o para administradores)
    if (!$is_admin && $solicitud['estado_id'] != 3) {
        die('Solo se pueden generar PDFs de solicitudes aprobadas');
    }
    
    // Obtener detalles de la solicitud, incluyendo el código de proveedor
    $stmt = $conn->prepare("
        SELECT srd.*, r.codigo as repuesto_codigo, r.nombre as repuesto_nombre, 
               r.numero_pieza, r.disponible, r.costo,
               cr.nombre as categoria_nombre,
               srd.codigo_proveedor
        FROM solicitud_repuesto_detalle srd
        JOIN repuestos r ON srd.repuesto_id = r.id
        LEFT JOIN categorias_repuestos cr ON r.categoria_id = cr.id
        WHERE srd.solicitud_id = :solicitud_id
        ORDER BY srd.id
    ");
    $stmt->bindParam(':solicitud_id', $id);
    $stmt->execute();
    $detalles = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Generar código único para la solicitud
    $codigo = generarCodigo('SOL', $id);
    
    // Crear instancia del PDF
    $pdf = new SolicitudPDF($solicitud, 'SOLICITUD DE REPUESTOS', $codigo);
    $pdf->AliasNbPages();
    $pdf->SetFont('DejaVu', '', 10);
    
    // Sección de información general con los nuevos campos
    $pdf->AddSection('INFORMACIÓN GENERAL', 0, true);
    
    // Datos para la tabla de información general (izquierda)
    $leftInfoData = [
        ['Fecha', formatoFecha($solicitud['fecha'])],
        ['Estado', htmlspecialchars($solicitud['estado_nombre']), $solicitud['estado_id']],
        ['Área', ucfirst(str_replace('_', ' ', $solicitud['area_origen']))],
        ['Motivo', ucfirst(str_replace('_', ' ', $solicitud['motivo_solicitud']))]
    ];
    
    // Datos para la tabla de información general (derecha)
    $rightInfoData = [
        ['Solicitante', htmlspecialchars($solicitud['usuario_nombre'] . ' ' . $solicitud['usuario_apellido'])],
        ['Cargo', htmlspecialchars($solicitud['usuario_cargo'])],
        ['RUT', htmlspecialchars($solicitud['usuario_rut'])]
    ];
    
    // Añadir revisor si existe
    if ($solicitud['revisor_id']) {
        $rightInfoData[] = ['Aprobado por', htmlspecialchars($solicitud['revisor_nombre'] . ' ' . $solicitud['revisor_apellido'])];
    }
    
    // Generar la tabla de información
    $pdf->AddInformationTable($leftInfoData, $rightInfoData);
    
    // Sección de máquina con información de serie
    $pdf->Ln(5);
    $pdf->AddSection('MÁQUINA', 0, true);
    
    // Datos para la tabla de máquina (izquierda)
    $leftMachineData = [
        ['Código', htmlspecialchars($solicitud['maquina_codigo'])],
        ['Nombre', htmlspecialchars($solicitud['maquina_nombre'])]
    ];
    
    // Datos para la tabla de máquina (derecha)
    $rightMachineData = [
        ['Marca', !empty($solicitud['maquina_marca']) ? htmlspecialchars($solicitud['maquina_marca']) : 'N/A'],
        ['Modelo', !empty($solicitud['maquina_modelo']) ? htmlspecialchars($solicitud['maquina_modelo']) : 'N/A']
    ];
    
    // Añadir información de serie - Lógica mejorada para manejar todos los casos posibles
    $serie = 'N/A';
    if (!empty($solicitud['serie_manual'])) {
        $serie = htmlspecialchars($solicitud['serie_manual']);
    } elseif (!empty($solicitud['numero_serie'])) {
        $serie = htmlspecialchars($solicitud['numero_serie']);
    } elseif (!empty($solicitud['codigo_serie'])) {
        $serie = htmlspecialchars($solicitud['codigo_serie']);
    }
    
    $leftMachineData[] = ['Serie', $serie];
    
    // Para depuración
    if (defined('DEBUG_MODE') && DEBUG_MODE) {
        error_log("Serie manual: " . ($solicitud['serie_manual'] ?? 'NULL'));
        error_log("Número serie: " . ($solicitud['numero_serie'] ?? 'NULL'));
        error_log("Código serie: " . ($solicitud['codigo_serie'] ?? 'NULL'));
        error_log("Serie seleccionada: " . $serie);
    }
    
    // Generar la tabla de información de máquina
    $pdf->AddInformationTable($leftMachineData, $rightMachineData);
    
    // Sección de descripción
    $pdf->Ln(5);
    $pdf->AddSection('DESCRIPCIÓN DE LA SOLICITUD', 0, true);
    $pdf->SetFillColor(255, 255, 255);
    $pdf->SetTextColor(0, 0, 0);
    $pdf->MultiCell(0, 6, $solicitud['descripcion'] ?? 'No especificada', 1, 'L', true);
    
    // Sección de comentarios (si hay)
    if (!empty($solicitud['comentario_revision'])) {
        $pdf->Ln(5);
        $pdf->AddSection('COMENTARIOS DE APROBACIÓN', 0, true);
        $pdf->SetFillColor(255, 255, 255);
        $pdf->SetTextColor(0, 0, 0);
        $pdf->MultiCell(0, 6, $solicitud['comentario_revision'], 1, 'L', true);
    }
    
    // Sección de repuestos con nueva columna para código de proveedor
    $pdf->Ln(5);
    $pdf->AddSection('REPUESTOS SOLICITADOS', 0, true);
    
    $pdf->SetFont('DejaVu', 'B', 9);
    $pdf->SetFillColor(240, 248, 255); // Azul muy claro
    $pdf->SetTextColor(0, 51, 102); // Azul corporativo
    
    if ($is_admin) {
        // Tabla para administradores (incluye costos)
        $pdf->Cell(8, 7, '#', 1, 0, 'C', true);
        $pdf->Cell(18, 7, 'Código', 1, 0, 'C', true);
        $pdf->Cell(50, 7, 'Descripción', 1, 0, 'C', true);
        $pdf->Cell(25, 7, 'Categoría', 1, 0, 'C', true);
        $pdf->Cell(20, 7, 'N° Pieza', 1, 0, 'C', true);
        $pdf->Cell(13, 7, 'Cant.', 1, 0, 'C', true);
        $pdf->Cell(20, 7, 'Cód. Prov.', 1, 0, 'C', true);
        $pdf->Cell(20, 7, 'Costo Unit.', 1, 0, 'C', true);
        $pdf->Cell(20, 7, 'Costo Total', 1, 1, 'C', true);
    } else {
        // Tabla para usuarios normales (sin costos)
        $pdf->Cell(10, 7, '#', 1, 0, 'C', true);
        $pdf->Cell(25, 7, 'Código', 1, 0, 'C', true);
        $pdf->Cell(60, 7, 'Descripción', 1, 0, 'C', true);
        $pdf->Cell(25, 7, 'Categoría', 1, 0, 'C', true);
        $pdf->Cell(25, 7, 'N° Pieza', 1, 0, 'C', true);
        $pdf->Cell(20, 7, 'Cantidad', 1, 0, 'C', true);
        $pdf->Cell(25, 7, 'Cód. Proveedor', 1, 1, 'C', true);
    }
    
    $pdf->SetFont('DejaVu', '', 9);
    $pdf->SetFillColor(255, 255, 255);
    $pdf->SetTextColor(0, 0, 0);
    
    $total = 0;
    $fill = false;
    
    foreach ($detalles as $index => $detalle) {
        $fill = !$fill;
        $costo_total = $detalle['cantidad'] * ($detalle['costo'] ?? 0);
        $total += $costo_total;
        
        if ($is_admin) {
            $pdf->Cell(8, 6, $index + 1, 1, 0, 'C', $fill);
            $pdf->Cell(18, 6, $detalle['repuesto_codigo'], 1, 0, 'C', $fill);
            $pdf->Cell(50, 6, $detalle['repuesto_nombre'], 1, 0, 'L', $fill);
            $pdf->Cell(25, 6, !empty($detalle['categoria_nombre']) ? $detalle['categoria_nombre'] : 'Sin categoría', 1, 0, 'L', $fill);
            $pdf->Cell(20, 6, !empty($detalle['numero_pieza']) ? $detalle['numero_pieza'] : 'N/A', 1, 0, 'C', $fill);
            $pdf->Cell(13, 6, $detalle['cantidad'], 1, 0, 'C', $fill);
            $pdf->Cell(20, 6, !empty($detalle['codigo_proveedor']) ? $detalle['codigo_proveedor'] : 'N/A', 1, 0, 'C', $fill);
            $pdf->Cell(20, 6, !empty($detalle['costo']) ? $pdf->FormatoMoneda($detalle['costo']) : 'No def.', 1, 0, 'R', $fill);
            $pdf->Cell(20, 6, !empty($detalle['costo']) ? $pdf->FormatoMoneda($costo_total) : 'No def.', 1, 1, 'R', $fill);
        } else {
            $pdf->Cell(10, 6, $index + 1, 1, 0, 'C', $fill);
            $pdf->Cell(25, 6, $detalle['repuesto_codigo'], 1, 0, 'C', $fill);
            $pdf->Cell(60, 6, $detalle['repuesto_nombre'], 1, 0, 'L', $fill);
            $pdf->Cell(25, 6, !empty($detalle['categoria_nombre']) ? $detalle['categoria_nombre'] : 'Sin categoría', 1, 0, 'L', $fill);
            $pdf->Cell(25, 6, !empty($detalle['numero_pieza']) ? $detalle['numero_pieza'] : 'N/A', 1, 0, 'C', $fill);
            $pdf->Cell(20, 6, $detalle['cantidad'], 1, 0, 'C', $fill);
            $pdf->Cell(25, 6, !empty($detalle['codigo_proveedor']) ? $detalle['codigo_proveedor'] : 'N/A', 1, 1, 'C', $fill);
        }
    }
    
    if ($is_admin && $total > 0) {
        $pdf->SetFont('DejaVu', 'B', 9);
        $pdf->SetFillColor(240, 248, 255);
        $pdf->SetTextColor(0, 51, 102);
        $pdf->Cell(154, 7, 'TOTAL', 1, 0, 'R', true);
        $pdf->Cell(40, 7, $pdf->FormatoMoneda($total), 1, 1, 'R', true);
    }
    
    // Sección de firmas
    $pdf->AddSignatureSection();
    
    // Registrar actividad
    registrarActividad($conn, $user_id, 'descargar', 'Descarga de PDF de solicitud: ' . $codigo, $id, 'solicitud_repuesto');
    
    // Generar PDF
    $file_name = $codigo . '.pdf';
    if ($save) {
        $file_path = 'temp/' . $file_name;
        $pdf->Output($file_path, 'F');
        if (!empty($redirect)) {
            header('Location: ' . $redirect);
            exit;
        }
        echo "PDF guardado exitosamente en: " . $file_path;
    } else {
        $pdf->Output($file_name, 'I');
    }
    
} catch (PDOException $e) {
    die('Error de conexión a la base de datos: ' . $e->getMessage());
} catch (Exception $e) {
    die('Error general: ' . $e->getMessage());
}