<?php
/**
 * Dashboard principal del sistema
 * Versión optimizada - Muestra estadísticas y acciones rápidas según el rol del usuario
 */

$page_title = 'Dashboard';
require_once 'includes/config.php';
require_once 'includes/functions.php';

verificarSesion();
$user_id = $_SESSION['user_id'];
$user_rol = $_SESSION['user_rol'] ?? '';
$user_nombre = $_SESSION['user_nombre'] ?? '';
$user_apellido = $_SESSION['user_apellido'] ?? '';
$is_admin = ($user_rol === 'admin' || $user_rol === 'super_admin');

try {
    $conn = new PDO("mysql:host=$db_host;dbname=$db_name;charset=utf8mb4", $db_user, $db_pass, $db_options);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // 1. Total de rendiciones y monto en los últimos 30 días
    if ($is_admin) {
        $stmt = $conn->prepare("
            SELECT COUNT(*) as total_documentos, COALESCE(SUM(monto_total), 0) as total_monto,
                   COUNT(CASE WHEN fecha_creacion >= DATE_SUB(CURDATE(), INTERVAL 7 DAY) THEN 1 END) as ultimos_7_dias
            FROM rendiciones 
            WHERE fecha_creacion >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
        ");
    } else {
        $stmt = $conn->prepare("
            SELECT COUNT(*) as total_documentos, COALESCE(SUM(monto_total), 0) as total_monto,
                   COUNT(CASE WHEN fecha_creacion >= DATE_SUB(CURDATE(), INTERVAL 7 DAY) THEN 1 END) as ultimos_7_dias
            FROM rendiciones 
            WHERE user_id = :user_id AND fecha_creacion >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
        ");
        $stmt->bindParam(':user_id', $user_id);
    }
    $stmt->execute();
    $rendiciones_stats = $stmt->fetch(PDO::FETCH_ASSOC);
    $total_rendiciones_count = $rendiciones_stats['total_documentos'] ?? 0;
    $total_rendiciones_monto = $rendiciones_stats['total_monto'] ?? 0;
    $rendiciones_recientes = $rendiciones_stats['ultimos_7_dias'] ?? 0;

    // 2. Total de órdenes de compra y monto en los últimos 30 días
    try {
        if ($is_admin) {
            $stmt = $conn->prepare("
                SELECT COUNT(*) as total_documentos, COALESCE(SUM(monto_total), 0) as total_monto,
                       COUNT(CASE WHEN fecha_creacion >= DATE_SUB(CURDATE(), INTERVAL 7 DAY) THEN 1 END) as ultimos_7_dias
                FROM ordenes_compra 
                WHERE fecha_creacion >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
            ");
        } else {
            $stmt = $conn->prepare("
                SELECT COUNT(*) as total_documentos, COALESCE(SUM(monto_total), 0) as total_monto,
                       COUNT(CASE WHEN fecha_creacion >= DATE_SUB(CURDATE(), INTERVAL 7 DAY) THEN 1 END) as ultimos_7_dias
                FROM ordenes_compra 
                WHERE user_id = :user_id AND fecha_creacion >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
            ");
            $stmt->bindParam(':user_id', $user_id);
        }
        $stmt->execute();
        $ordenes_stats = $stmt->fetch(PDO::FETCH_ASSOC);
        $total_ordenes_count = $ordenes_stats['total_documentos'] ?? 0;
        $total_ordenes_monto = $ordenes_stats['total_monto'] ?? 0;
        $ordenes_recientes = $ordenes_stats['ultimos_7_dias'] ?? 0;
    } catch (PDOException $e) {
        $total_ordenes_count = 0;
        $total_ordenes_monto = 0;
        $ordenes_recientes = 0;
    }

    // 3. Documentos pendientes de aprobación
    if ($is_admin) {
        $stmt = $conn->prepare("
            SELECT 
            (SELECT COUNT(*) FROM rendiciones WHERE estado_id IN (1, 2)) +
            (SELECT COUNT(*) FROM ordenes_compra WHERE estado_id IN (1, 2)) +
            (SELECT COUNT(*) FROM solicitud_repuestos WHERE estado_id IN (1, 2)) AS pendientes
        ");
    } else {
        $stmt = $conn->prepare("
            SELECT 
            (SELECT COUNT(*) FROM rendiciones WHERE user_id = :user_id1 AND estado_id IN (1, 2)) +
            (SELECT COUNT(*) FROM ordenes_compra WHERE user_id = :user_id2 AND estado_id IN (1, 2)) +
            (SELECT COUNT(*) FROM solicitud_repuestos WHERE user_id = :user_id3 AND estado_id IN (1, 2)) AS pendientes
        ");
        $stmt->bindParam(':user_id1', $user_id);
        $stmt->bindParam(':user_id2', $user_id);
        $stmt->bindParam(':user_id3', $user_id);
    }
    $stmt->execute();
    $pendientes = $stmt->fetch(PDO::FETCH_ASSOC)['pendientes'] ?? 0;

    // 4. Documentos aprobados en los últimos 30 días
    if ($is_admin) {
        $stmt = $conn->prepare("
            SELECT 
            (SELECT COUNT(*) FROM rendiciones WHERE estado_id = 3 AND fecha_revision >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)) +
            (SELECT COUNT(*) FROM ordenes_compra WHERE estado_id = 3 AND fecha_revision >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)) +
            (SELECT COUNT(*) FROM solicitud_repuestos WHERE estado_id = 3 AND fecha_revision >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)) AS aprobados
        ");
    } else {
        $stmt = $conn->prepare("
            SELECT 
            (SELECT COUNT(*) FROM rendiciones WHERE user_id = :user_id1 AND estado_id = 3 AND fecha_revision >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)) +
            (SELECT COUNT(*) FROM ordenes_compra WHERE user_id = :user_id2 AND estado_id = 3 AND fecha_revision >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)) +
            (SELECT COUNT(*) FROM solicitud_repuestos WHERE user_id = :user_id3 AND estado_id = 3 AND fecha_revision >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)) AS aprobados
        ");
        $stmt->bindParam(':user_id1', $user_id);
        $stmt->bindParam(':user_id2', $user_id);
        $stmt->bindParam(':user_id3', $user_id);
    }
    $stmt->execute();
    $aprobados = $stmt->fetch(PDO::FETCH_ASSOC)['aprobados'] ?? 0;
    
    // 5. Estadísticas de máquinas y repuestos
    try {
        $stmt = $conn->prepare("SELECT COUNT(*) as total FROM maquinas WHERE activo = 1");
        $stmt->execute();
        $total_maquinas = $stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;
        
        $stmt = $conn->prepare("SELECT COUNT(*) as total FROM repuestos WHERE activo = 1");
        $stmt->execute();
        $total_repuestos = $stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;
        
        $stmt = $conn->prepare("
            SELECT COUNT(*) as total, SUM(CASE WHEN stock = 0 THEN 1 ELSE 0 END) as sin_stock 
            FROM repuestos WHERE activo = 1
        ");
        $stmt->execute();
        $stock_stats = $stmt->fetch(PDO::FETCH_ASSOC);
        $repuestos_sin_stock = $stock_stats['sin_stock'] ?? 0;
    } catch (PDOException $e) {
        $total_maquinas = 0;
        $total_repuestos = 0;
        $repuestos_sin_stock = 0;
    }
    
    // 6. Solicitudes de repuestos recientes
    try {
        if ($is_admin) {
            $stmt = $conn->prepare("
                SELECT sr.id, sr.fecha, sr.descripcion, sr.estado_id, sr.fecha_creacion, 
                       u.nombre, u.apellido, m.nombre as maquina_nombre, sr.motivo_solicitud
                FROM solicitud_repuestos sr
                JOIN users u ON sr.user_id = u.id
                JOIN maquinas m ON sr.maquina_id = m.id
                ORDER BY sr.fecha_creacion DESC
                LIMIT 5
            ");
        } else {
            $stmt = $conn->prepare("
                SELECT sr.id, sr.fecha, sr.descripcion, sr.estado_id, sr.fecha_creacion, 
                       u.nombre, u.apellido, m.nombre as maquina_nombre, sr.motivo_solicitud
                FROM solicitud_repuestos sr
                JOIN users u ON sr.user_id = u.id
                JOIN maquinas m ON sr.maquina_id = m.id
                WHERE sr.user_id = :user_id
                ORDER BY sr.fecha_creacion DESC
                LIMIT 5
            ");
            $stmt->bindParam(':user_id', $user_id);
        }
        $stmt->execute();
        $solicitudes_recientes = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        $solicitudes_recientes = [];
    }

    // 7. Documentos recientes combinados
    if ($is_admin) {
        $stmt = $conn->prepare("
            (SELECT 'Rendición' as tipo, r.id, r.fecha_creacion as fecha, r.fecha as fecha_documento, r.monto_total as monto, r.estado_id, 
                   CONCAT('REN-', YEAR(r.fecha_creacion), '-', LPAD(r.id, 3, '0')) as codigo,
                   u.nombre, u.apellido
            FROM rendiciones r
            JOIN users u ON r.user_id = u.id
            ORDER BY r.fecha_creacion DESC
            LIMIT 3)
            UNION ALL
            (SELECT 'Orden de Compra' as tipo, oc.id, oc.fecha_creacion as fecha, oc.fecha as fecha_documento, oc.monto_total as monto, oc.estado_id,
                   CONCAT('OC-', YEAR(oc.fecha_creacion), '-', LPAD(oc.id, 3, '0')) as codigo,
                   u.nombre, u.apellido
            FROM ordenes_compra oc
            JOIN users u ON oc.user_id = u.id
            ORDER BY oc.fecha_creacion DESC
            LIMIT 3)
            UNION ALL
            (SELECT 'Solicitud de Repuestos' as tipo, sr.id, sr.fecha_creacion as fecha, sr.fecha as fecha_documento, 0 as monto, sr.estado_id,
                   CONCAT('SOL-', YEAR(sr.fecha_creacion), '-', LPAD(sr.id, 3, '0')) as codigo,
                   u.nombre, u.apellido
            FROM solicitud_repuestos sr
            JOIN users u ON sr.user_id = u.id
            ORDER BY sr.fecha_creacion DESC
            LIMIT 3)
            ORDER BY fecha DESC
            LIMIT 5
        ");
    } else {
        $stmt = $conn->prepare("
            (SELECT 'Rendición' as tipo, r.id, r.fecha_creacion as fecha, r.fecha as fecha_documento, r.monto_total as monto, r.estado_id, 
                   CONCAT('REN-', YEAR(r.fecha_creacion), '-', LPAD(r.id, 3, '0')) as codigo,
                   u.nombre, u.apellido
            FROM rendiciones r
            JOIN users u ON r.user_id = u.id
            WHERE r.user_id = :user_id1
            ORDER BY r.fecha_creacion DESC
            LIMIT 3)
            UNION ALL
            (SELECT 'Orden de Compra' as tipo, oc.id, oc.fecha_creacion as fecha, oc.fecha as fecha_documento, oc.monto_total as monto, oc.estado_id,
                   CONCAT('OC-', YEAR(oc.fecha_creacion), '-', LPAD(oc.id, 3, '0')) as codigo,
                   u.nombre, u.apellido
            FROM ordenes_compra oc
            JOIN users u ON oc.user_id = u.id
            WHERE oc.user_id = :user_id2
            ORDER BY oc.fecha_creacion DESC
            LIMIT 3)
            UNION ALL
            (SELECT 'Solicitud de Repuestos' as tipo, sr.id, sr.fecha_creacion as fecha, sr.fecha as fecha_documento, 0 as monto, sr.estado_id,
                   CONCAT('SOL-', YEAR(sr.fecha_creacion), '-', LPAD(sr.id, 3, '0')) as codigo,
                   u.nombre, u.apellido
            FROM solicitud_repuestos sr
            JOIN users u ON sr.user_id = u.id
            WHERE sr.user_id = :user_id3
            ORDER BY sr.fecha_creacion DESC
            LIMIT 3)
            ORDER BY fecha DESC
            LIMIT 5
        ");
        $stmt->bindParam(':user_id1', $user_id);
        $stmt->bindParam(':user_id2', $user_id);
        $stmt->bindParam(':user_id3', $user_id);
    }
    $stmt->execute();
    $documentos_recientes = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // 8. Estados para mostrar badges
    $estados = obtenerEstados($conn);
    $estados_map = [];
    foreach ($estados as $estado) {
        $estados_map[$estado['id']] = [
            'nombre' => $estado['nombre'],
            'color' => $estado['color']
        ];
    }

    // 9. Datos para el gráfico de categorías
    $stmt = $conn->prepare("
        SELECT c.nombre, COALESCE(SUM(rc.monto), 0) as total
        FROM categorias_gastos c
        LEFT JOIN rendicion_categoria rc ON c.id = rc.categoria_id
        LEFT JOIN rendiciones r ON rc.rendicion_id = r.id
        WHERE c.activo = 1 " . ($is_admin ? "" : " AND r.user_id = :user_id") . "
        AND r.fecha_creacion >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
        GROUP BY c.id
        ORDER BY total DESC
        LIMIT 5
    ");
    if (!$is_admin) {
        $stmt->bindParam(':user_id', $user_id);
    }
    $stmt->execute();
    $categorias_data = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $categorias = [];
    $valores_categorias = [];
    foreach ($categorias_data as $cat) {
        if ($cat['total'] > 0) {
            $categorias[] = $cat['nombre'];
            $valores_categorias[] = floatval($cat['total']);
        }
    }

    // 10. Datos para el gráfico de solicitudes por motivo
    try {
        $stmt = $conn->prepare("
            SELECT motivo_solicitud, COUNT(*) as total
            FROM solicitud_repuestos
            WHERE fecha_creacion >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
            " . ($is_admin ? "" : " AND user_id = :user_id") . "
            GROUP BY motivo_solicitud
            ORDER BY total DESC
        ");
        if (!$is_admin) {
            $stmt->bindParam(':user_id', $user_id);
        }
        $stmt->execute();
        $motivos_data = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $motivos = [];
        $valores_motivos = [];
        foreach ($motivos_data as $motivo) {
            if ($motivo['total'] > 0) {
                $nombre_motivo = ucfirst(str_replace('_', ' ', $motivo['motivo_solicitud']));
                $motivos[] = $nombre_motivo;
                $valores_motivos[] = intval($motivo['total']);
            }
        }
    } catch (PDOException $e) {
        $motivos = [];
        $valores_motivos = [];
    }

    // 11. Indicadores para tendencias
    $tendencia_rendiciones = 0;
    $tendencia_ordenes = 0;

    if ($is_admin) {
        $stmt = $conn->prepare("
            SELECT 
                COALESCE(SUM(CASE WHEN fecha_creacion >= DATE_SUB(CURDATE(), INTERVAL 30 DAY) THEN monto_total ELSE 0 END), 0) as mes_actual,
                COALESCE(SUM(CASE WHEN fecha_creacion BETWEEN DATE_SUB(CURDATE(), INTERVAL 60 DAY) AND DATE_SUB(CURDATE(), INTERVAL 30 DAY) THEN monto_total ELSE 0 END), 0) as mes_anterior
            FROM rendiciones
        ");
    } else {
        $stmt = $conn->prepare("
            SELECT 
                COALESCE(SUM(CASE WHEN fecha_creacion >= DATE_SUB(CURDATE(), INTERVAL 30 DAY) THEN monto_total ELSE 0 END), 0) as mes_actual,
                COALESCE(SUM(CASE WHEN fecha_creacion BETWEEN DATE_SUB(CURDATE(), INTERVAL 60 DAY) AND DATE_SUB(CURDATE(), INTERVAL 30 DAY) THEN monto_total ELSE 0 END), 0) as mes_anterior
            FROM rendiciones
            WHERE user_id = :user_id
        ");
        $stmt->bindParam(':user_id', $user_id);
    }
    $stmt->execute();
    $tendencia_datos = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($tendencia_datos['mes_anterior'] > 0) {
        $tendencia_rendiciones = (($tendencia_datos['mes_actual'] - $tendencia_datos['mes_anterior']) / $tendencia_datos['mes_anterior']) * 100;
    }

    try {
        if ($is_admin) {
            $stmt = $conn->prepare("
                SELECT 
                    COALESCE(SUM(CASE WHEN fecha_creacion >= DATE_SUB(CURDATE(), INTERVAL 30 DAY) THEN monto_total ELSE 0 END), 0) as mes_actual,
                    COALESCE(SUM(CASE WHEN fecha_creacion BETWEEN DATE_SUB(CURDATE(), INTERVAL 60 DAY) AND DATE_SUB(CURDATE(), INTERVAL 30 DAY) THEN monto_total ELSE 0 END), 0) as mes_anterior
                FROM ordenes_compra
            ");
        } else {
            $stmt = $conn->prepare("
                SELECT 
                    COALESCE(SUM(CASE WHEN fecha_creacion >= DATE_SUB(CURDATE(), INTERVAL 30 DAY) THEN monto_total ELSE 0 END), 0) as mes_actual,
                    COALESCE(SUM(CASE WHEN fecha_creacion BETWEEN DATE_SUB(CURDATE(), INTERVAL 60 DAY) AND DATE_SUB(CURDATE(), INTERVAL 30 DAY) THEN monto_total ELSE 0 END), 0) as mes_anterior
                FROM ordenes_compra
                WHERE user_id = :user_id
            ");
            $stmt->bindParam(':user_id', $user_id);
        }
        $stmt->execute();
        $tendencia_datos = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($tendencia_datos['mes_anterior'] > 0) {
            $tendencia_ordenes = (($tendencia_datos['mes_actual'] - $tendencia_datos['mes_anterior']) / $tendencia_datos['mes_anterior']) * 100;
        }
    } catch (PDOException $e) {
        $tendencia_ordenes = 0;
    }

    // 12. Datos para el gráfico de repuestos más solicitados
    try {
        $stmt = $conn->prepare("
            SELECT r.nombre, COUNT(srd.repuesto_id) as solicitudes
            FROM repuestos r
            JOIN solicitud_repuesto_detalle srd ON r.id = srd.repuesto_id
            JOIN solicitud_repuestos sr ON srd.solicitud_id = sr.id
            WHERE sr.fecha_creacion >= DATE_SUB(CURDATE(), INTERVAL 90 DAY)
            GROUP BY r.id
            ORDER BY solicitudes DESC
            LIMIT 5
        ");
        $stmt->execute();
        $repuestos_datos = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $repuestos_nombres = [];
        $repuestos_solicitudes = [];
        foreach ($repuestos_datos as $rep) {
            $repuestos_nombres[] = $rep['nombre'];
            $repuestos_solicitudes[] = intval($rep['solicitudes']);
        }
    } catch (PDOException $e) {
        $repuestos_nombres = [];
        $repuestos_solicitudes = [];
    }

    // 13. Top usuarios por gastos (solo admins)
    if ($is_admin) {
        $stmt = $conn->prepare("
            SELECT u.id, u.nombre, u.apellido, COALESCE(SUM(r.monto_total), 0) as total_gastos
            FROM users u
            LEFT JOIN rendiciones r ON u.id = r.user_id AND r.fecha_creacion >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
            WHERE u.activo = 1
            GROUP BY u.id
            ORDER BY total_gastos DESC
            LIMIT 5
        ");
        $stmt->execute();
        $top_usuarios_gastos = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // 14. Actividad reciente (historial_cambios)
    if ($is_admin) {
        $stmt = $conn->prepare("
            SELECT hc.*, u.nombre, u.apellido
            FROM historial_cambios hc
            JOIN users u ON hc.user_id = u.id
            ORDER BY hc.fecha_hora DESC
            LIMIT 8
        ");
    } else {
        $stmt = $conn->prepare("
            SELECT hc.*, u.nombre, u.apellido
            FROM historial_cambios hc
            JOIN users u ON hc.user_id = u.id
            WHERE hc.user_id = :user_id
            ORDER BY hc.fecha_hora DESC
            LIMIT 8
        ");
        $stmt->bindParam(':user_id', $user_id);
    }
    $stmt->execute();
    $actividad_reciente = $stmt->fetchAll(PDO::FETCH_ASSOC);

} catch (PDOException $e) {
    $error_message = 'Error de conexión a la base de datos: ' . $e->getMessage();
}

require_once 'includes/header.php';
?>

<!-- Mensaje de error si existe -->
<?php if (isset($error_message)): ?>
<div class="alert alert-danger">
    <?php echo $error_message; ?>
</div>
<?php endif; ?>

<!-- Bienvenida -->
<div class="welcome-section">
    <div class="welcome-content">
        <h2>Bienvenido, <?php echo htmlspecialchars($user_nombre); ?></h2>
        <p class="text-muted">
            <i class="material-icons" style="vertical-align: middle; font-size: 18px;">date_range</i>
            <?php echo date('d \d\e F Y'); ?>
        </p>
    </div>
    <div class="welcome-actions">
        <span class="periodo-label">Periodo: Últimos 30 días</span>
    </div>
</div>

<!-- Tarjetas de resumen principal -->
<div class="summary-cards">
    <?php if (tienePermiso('ver_rendiciones')): ?>
    <div class="summary-card card-primary">
        <div class="summary-icon">
            <i class="material-icons">receipt</i>
        </div>
        <div class="summary-details">
            <h3>Rendiciones</h3>
            <div class="summary-value"><?php echo formatoMoneda($total_rendiciones_monto); ?></div>
            <div class="summary-info">
                <span class="summary-count"><?php echo $total_rendiciones_count; ?> documentos</span>
                <?php if ($tendencia_rendiciones != 0): ?>
                <div class="summary-trend <?php echo $tendencia_rendiciones >= 0 ? 'trend-up' : 'trend-down'; ?>">
                    <i class="material-icons"><?php echo $tendencia_rendiciones >= 0 ? 'trending_up' : 'trending_down'; ?></i>
                    <span><?php echo abs(round($tendencia_rendiciones)); ?>%</span>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <?php if (tienePermiso('ver_ordenes_compra')): ?>
    <div class="summary-card card-secondary">
        <div class="summary-icon">
            <i class="material-icons">shopping_cart</i>
        </div>
        <div class="summary-details">
            <h3>Órdenes de Compra</h3>
            <div class="summary-value"><?php echo formatoMoneda($total_ordenes_monto); ?></div>
            <div class="summary-info">
                <span class="summary-count"><?php echo $total_ordenes_count; ?> documentos</span>
                <?php if ($tendencia_ordenes != 0): ?>
                <div class="summary-trend <?php echo $tendencia_ordenes >= 0 ? 'trend-up' : 'trend-down'; ?>">
                    <i class="material-icons"><?php echo $tendencia_ordenes >= 0 ? 'trending_up' : 'trending_down'; ?></i>
                    <span><?php echo abs(round($tendencia_ordenes)); ?>%</span>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <?php if (tienePermiso('ver_rendiciones') || tienePermiso('ver_ordenes_compra') || tienePermiso('ver_solicitudes_repuesto')): ?>
    <div class="summary-card card-warning">
        <div class="summary-icon">
            <i class="material-icons">pending</i>
        </div>
        <div class="summary-details">
            <h3>Pendientes</h3>
            <div class="summary-value"><?php echo $pendientes; ?></div>
            <div class="summary-info">
                <span class="summary-count">Documentos en espera</span>
            </div>
        </div>
    </div>

    <div class="summary-card card-success">
        <div class="summary-icon">
            <i class="material-icons">check_circle</i>
        </div>
        <div class="summary-details">
            <h3>Aprobados</h3>
            <div class="summary-value"><?php echo $aprobados; ?></div>
            <div class="summary-info">
                <span class="summary-count">Documentos procesados</span>
            </div>
        </div>
    </div>
    <?php endif; ?>
</div>

<!-- Nuevas tarjetas de resumen de mantenimiento -->
<div class="summary-cards">
    <?php if (tienePermiso('ver_maquinas')): ?>
    <div class="summary-card card-info">
        <div class="summary-icon">
            <i class="material-icons">precision_manufacturing</i>
        </div>
        <div class="summary-details">
            <h3>Máquinas</h3>
            <div class="summary-value"><?php echo $total_maquinas; ?></div>
            <div class="summary-info">
                <span class="summary-count">Equipos registrados</span>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <?php if (tienePermiso('ver_repuestos')): ?>
    <div class="summary-card card-secondary">
        <div class="summary-icon">
            <i class="material-icons">build</i>
        </div>
        <div class="summary-details">
            <h3>Repuestos</h3>
            <div class="summary-value"><?php echo $total_repuestos; ?></div>
            <div class="summary-info">
                <span class="summary-count"><?php echo $repuestos_sin_stock; ?> sin stock</span>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <?php if (tienePermiso('ver_solicitudes_repuesto')): ?>
    <div class="summary-card card-purple">
        <div class="summary-icon">
            <i class="material-icons">assignment</i>
        </div>
        <div class="summary-details">
            <h3>Solicitudes</h3>
            <div class="summary-value"><?php echo count($solicitudes_recientes); ?></div>
            <div class="summary-info">
                <span class="summary-count">Recientes</span>
                <a href="solicitud_repuestos.php" class="view-all-link">Ver todas</a>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <?php if ($is_admin): ?>
    <div class="summary-card card-teal">
        <div class="summary-icon">
            <i class="material-icons">business</i>
        </div>
        <div class="summary-details">
            <h3>Proveedores</h3>
            <div class="summary-value">
                <?php 
                $stmt = $conn->prepare("SELECT COUNT(*) as total FROM proveedores WHERE activo = 1");
                $stmt->execute();
                echo $stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;
                ?>
            </div>
            <div class="summary-info">
                <span class="summary-count">Activos</span>
                <a href="proveedores.php" class="view-all-link">Gestionar</a>
            </div>
        </div>
    </div>
    <?php endif; ?>
</div>

<!-- Contenido principal en 3 columnas -->
<div class="dashboard-main">
    <div class="dashboard-column">
        <div class="dashboard-section">
            <div class="section-header">
                <h3><i class="material-icons">flash_on</i> Acciones Rápidas</h3>
            </div>
            <div class="quick-actions">
                <?php if (tienePermiso('crear_rendicion')): ?>
                <a href="rendiciones.php?action=new" class="action-card">
                    <div class="action-icon primary-bg">
                        <i class="material-icons">receipt</i>
                    </div>
                    <h3 class="action-title">Nueva Rendición</h3>
                    <p class="action-description">Registrar gastos para reembolso</p>
                </a>
                <?php endif; ?>
                
                <?php if (tienePermiso('crear_orden_compra')): ?>
                <a href="ordenes_compra.php?action=new" class="action-card">
                    <div class="action-icon secondary-bg">
                        <i class="material-icons">shopping_cart</i>
                    </div>
                    <h3 class="action-title">Nueva Orden</h3>
                    <p class="action-description">Solicitar compra de bienes</p>
                </a>
                <?php endif; ?>
                
                <?php if (tienePermiso('crear_solicitud_repuesto')): ?>
                <a href="solicitud_repuestos.php?action=new" class="action-card">
                    <div class="action-icon warning-bg">
                        <i class="material-icons">build</i>
                    </div>
                    <h3 class="action-title">Solicitar Repuestos</h3>
                    <p class="action-description">Pedir repuestos para máquinas</p>
                </a>
                <?php endif; ?>
                
                <?php if (tienePermiso('ver_reportes')): ?>
                <a href="reportes.php" class="action-card">
                    <div class="action-icon info-bg">
                        <i class="material-icons">analytics</i>
                    </div>
                    <h3 class="action-title">Reportes</h3>
                    <p class="action-description">Ver estadísticas detalladas</p>
                </a>
                <?php endif; ?>
            </div>
        </div>
        
        <?php if (tienePermiso('ver_rendiciones') && !empty($categorias)): ?>
        <div class="dashboard-section">
            <div class="section-header">
                <h3><i class="material-icons">pie_chart</i> Gastos por Categoría</h3>
                <?php if (tienePermiso('ver_reportes')): ?>
                <a href="reportes.php?tipo_reporte=rendiciones_categoria" class="section-link">Ver más</a>
                <?php endif; ?>
            </div>
            <div class="chart-container">
                <canvas id="categoryChart"></canvas>
            </div>
        </div>
        <?php endif; ?>
        
        <?php if (tienePermiso('ver_repuestos') && !empty($repuestos_nombres)): ?>
        <div class="dashboard-section">
            <div class="section-header">
                <h3><i class="material-icons">bar_chart</i> Repuestos más Solicitados</h3>
                <?php if (tienePermiso('ver_reportes')): ?>
                <a href="reportes.php?tipo_reporte=solicitudes_repuestos" class="section-link">Ver más</a>
                <?php endif; ?>
            </div>
            <div class="chart-container">
                <canvas id="repuestosChart"></canvas>
            </div>
        </div>
        <?php endif; ?>
    </div>
    
    <div class="dashboard-column">
        <?php if (tienePermiso('ver_rendiciones') || tienePermiso('ver_ordenes_compra') || tienePermiso('ver_solicitudes_repuesto')): ?>
        <div class="dashboard-section">
            <div class="section-header">
                <h3><i class="material-icons">history</i> Documentos Recientes</h3>
                <a href="historial.php" class="section-link">Ver todo</a>
            </div>
            <div class="activity-list">
                <?php if (empty($documentos_recientes)): ?>
                    <div class="empty-state">
                        <i class="material-icons">receipt</i>
                        <p>No hay documentos recientes</p>
                    </div>
                <?php else: ?>
                    <?php foreach ($documentos_recientes as $doc): ?>
                    <?php 
                    $mostrar_doc = false;
                    if ($doc['tipo'] == 'Rendición' && tienePermiso('ver_rendiciones')) $mostrar_doc = true;
                    if ($doc['tipo'] == 'Orden de Compra' && tienePermiso('ver_ordenes_compra')) $mostrar_doc = true;
                    if ($doc['tipo'] == 'Solicitud de Repuestos' && tienePermiso('ver_solicitudes_repuesto')) $mostrar_doc = true;
                    if ($mostrar_doc):
                    ?>
                    <div class="activity-item">
                        <div class="activity-icon 
                            <?php 
                            if ($doc['tipo'] == 'Rendición') echo 'primary-bg';
                            elseif ($doc['tipo'] == 'Orden de Compra') echo 'secondary-bg';
                            else echo 'warning-bg';
                            ?>">
                            <i class="material-icons">
                                <?php 
                                if ($doc['tipo'] == 'Rendición') echo 'receipt';
                                elseif ($doc['tipo'] == 'Orden de Compra') echo 'shopping_cart';
                                else echo 'build';
                                ?>
                            </i>
                        </div>
                        <div class="activity-content">
                            <div class="activity-header">
                                <span class="activity-title"><?php echo htmlspecialchars($doc['codigo']); ?></span>
                                <span class="activity-date"><?php echo date('d/m/Y', strtotime($doc['fecha_documento'])); ?></span>
                            </div>
                            <div class="activity-details">
                                <?php if ($doc['tipo'] != 'Solicitud de Repuestos'): ?>
                                <span class="activity-amount"><?php echo formatoMoneda($doc['monto']); ?></span>
                                <?php endif; ?>
                                <span class="status-badge badge-estado-<?php echo $doc['estado_id']; ?>">
                                    <?php echo $estados_map[$doc['estado_id']]['nombre'] ?? 'Desconocido'; ?>
                                </span>
                            </div>
                            <div class="activity-actions">
                                <?php 
                                if ($doc['tipo'] == 'Rendición') {
                                    $url_prefix = 'rendiciones.php';
                                } elseif ($doc['tipo'] == 'Orden de Compra') {
                                    $url_prefix = 'ordenes_compra.php';
                                } else {
                                    $url_prefix = 'solicitud_repuestos.php';
                                }
                                ?>
                                <a href="<?php echo $url_prefix; ?>?action=view&id=<?php echo $doc['id']; ?>" class="action-link">
                                    <i class="material-icons">visibility</i> Ver
                                </a>
                                <?php if ($doc['estado_id'] == 3): ?>
                                <a href="generate_pdf.php?tipo=<?php 
                                    if ($doc['tipo'] == 'Rendición') echo 'rendicion';
                                    elseif ($doc['tipo'] == 'Orden de Compra') echo 'orden_compra';
                                    else echo 'solicitud_repuesto';
                                ?>&id=<?php echo $doc['id']; ?>" class="action-link">
                                    <i class="material-icons">picture_as_pdf</i> PDF
                                </a>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    <?php endif; ?>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
        <?php endif; ?>
        
        <?php if (tienePermiso('ver_solicitudes_repuesto') && !empty($motivos)): ?>
        <div class="dashboard-section">
            <div class="section-header">
                <h3><i class="material-icons">donut_large</i> Solicitudes por Motivo</h3>
            </div>
            <div class="chart-container">
                <canvas id="motivosChart"></canvas>
            </div>
        </div>
        <?php endif; ?>
        
        <?php if ($is_admin && !empty($top_usuarios_gastos)): ?>
        <div class="dashboard-section">
            <div class="section-header">
                <h3><i class="material-icons">group</i> Top Usuarios</h3>
                <a href="reportes.php?tipo_reporte=por_usuario" class="section-link">Ver más</a>
            </div>
            <div class="ranking-list">
                <?php foreach ($top_usuarios_gastos as $index => $usuario): ?>
                    <?php if ($usuario['total_gastos'] > 0): ?>
                    <div class="ranking-item">
                        <div class="ranking-position"><?php echo $index + 1; ?></div>
                        <div class="ranking-info">
                            <div class="ranking-name"><?php echo htmlspecialchars($usuario['nombre'] . ' ' . $usuario['apellido']); ?></div>
                        </div>
                        <div class="ranking-value"><?php echo formatoMoneda($usuario['total_gastos']); ?></div>
                    </div>
                    <?php endif; ?>
                <?php endforeach; ?>
            </div>
        </div>
        <?php endif; ?>
    </div>
    
    <div class="dashboard-column">
        <?php if (tienePermiso('ver_solicitudes_repuesto')): ?>
        <div class="dashboard-section">
            <div class="section-header">
                <h3><i class="material-icons">build</i> Solicitudes Recientes</h3>
                <a href="solicitud_repuestos.php" class="section-link">Ver todas</a>
            </div>
            <?php if (empty($solicitudes_recientes)): ?>
                <div class="empty-state">
                    <i class="material-icons">build</i>
                    <p>No hay solicitudes recientes</p>
                </div>
            <?php else: ?>
                <div class="solicitudes-list">
                    <?php foreach ($solicitudes_recientes as $solicitud): ?>
                    <div class="solicitud-item">
                        <div class="solicitud-header">
                            <span class="solicitud-title">Solicitud para <?php echo htmlspecialchars($solicitud['maquina_nombre']); ?></span>
                            <span class="status-badge badge-estado-<?php echo $solicitud['estado_id']; ?>">
                                <?php echo $estados_map[$solicitud['estado_id']]['nombre'] ?? 'Desconocido'; ?>
                            </span>
                        </div>
                        <div class="solicitud-content">
                            <p><?php echo htmlspecialchars(substr($solicitud['descripcion'], 0, 100)) . (strlen($solicitud['descripcion']) > 100 ? '...' : ''); ?></p>
                        </div>
                        <div class="solicitud-footer">
                            <span class="solicitud-date">
                                <i class="material-icons">calendar_today</i> 
                                <?php echo date('d/m/Y', strtotime($solicitud['fecha'])); ?>
                            </span>
                            <span class="solicitud-motivo">
                                <?php echo ucfirst(str_replace('_', ' ', $solicitud['motivo_solicitud'])); ?>
                            </span>
                            <a href="solicitud_repuestos.php?action=view&id=<?php echo $solicitud['id']; ?>" class="view-link">
                                <i class="material-icons">visibility</i> Ver detalles
                            </a>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
        <?php endif; ?>
        
        <div class="dashboard-section">
            <div class="section-header">
                <h3><i class="material-icons">update</i> Actividad Reciente</h3>
                <a href="historial.php" class="section-link">Ver todo</a>
            </div>
            <?php if (empty($actividad_reciente)): ?>
                <div class="empty-state">
                    <i class="material-icons">history</i>
                    <p>No hay actividad reciente</p>
                </div>
            <?php else: ?>
                <div class="historial-list">
                    <?php foreach ($actividad_reciente as $actividad): ?>
                    <div class="historial-item">
                        <div class="historial-icon 
                            <?php 
                            $tipo_icon = 'info-bg';
                            if (in_array($actividad['tipo'], ['crear', 'aprobar'])) $tipo_icon = 'success-bg';
                            elseif (in_array($actividad['tipo'], ['eliminar', 'rechazar'])) $tipo_icon = 'danger-bg';
                            elseif (in_array($actividad['tipo'], ['editar'])) $tipo_icon = 'warning-bg';
                            echo $tipo_icon;
                            ?>">
                            <i class="material-icons">
                                <?php 
                                switch($actividad['tipo']) {
                                    case 'crear': echo 'add_circle'; break;
                                    case 'editar': echo 'edit'; break;
                                    case 'eliminar': echo 'delete'; break;
                                    case 'aprobar': echo 'check_circle'; break;
                                    case 'rechazar': echo 'cancel'; break;
                                    case 'login': echo 'login'; break;
                                    case 'logout': echo 'logout'; break;
                                    case 'descargar': echo 'file_download'; break;
                                    default: echo 'info';
                                }
                                ?>
                            </i>
                        </div>
                        <div class="historial-content">
                            <div class="historial-header">
                                <span class="historial-user"><?php echo htmlspecialchars($actividad['nombre'].' '.$actividad['apellido']); ?></span>
                                <span class="historial-date"><?php echo date('d/m/Y H:i', strtotime($actividad['fecha_hora'])); ?></span>
                            </div>
                            <div class="historial-description">
                                <?php echo htmlspecialchars($actividad['descripcion']); ?>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<style>
    .welcome-section {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 25px;
        padding: 15px 20px;
        background-color: white;
        border-radius: 8px;
        box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
    }
    .welcome-section h2 {
        margin: 0;
        font-size: 24px;
        font-weight: 600;
        color: #333;
    }
    .text-muted {
        color: #6c757d;
        margin-top: 5px;
        display: flex;
        align-items: center;
    }
    .periodo-label {
        background-color: #f8f9fa;
        padding: 6px 12px;
        border-radius: 20px;
        font-size: 14px;
        color: #495057;
    }
    .summary-cards {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
        gap: 20px;
        margin-bottom: 30px;
    }
    .summary-card {
        display: flex;
        background-color: white;
        border-radius: 12px;
        padding: 20px;
        box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
        transition: transform 0.3s ease, box-shadow 0.3s ease;
        overflow: hidden;
        position: relative;
    }
    .summary-card:hover {
        transform: translateY(-5px);
        box-shadow: 0 8px 16px rgba(0, 0, 0, 0.1);
    }
    .summary-icon {
        display: flex;
        align-items: center;
        justify-content: center;
        width: 50px;
        height: 50px;
        border-radius: 10px;
        margin-right: 15px;
    }
    .summary-icon i {
        font-size: 24px;
        color: white;
    }
    .summary-details {
        flex: 1;
    }
    .summary-details h3 {
        margin: 0 0 8px 0;
        font-size: 16px;
        font-weight: 600;
        color: #495057;
    }
    .summary-value {
        font-size: 24px;
        font-weight: 700;
        margin-bottom: 5px;
        color: #333;
    }
    .summary-info {
        display: flex;
        justify-content: space-between;
        align-items: center;
    }
    .summary-count {
        font-size: 14px;
        color: #6c757d;
    }
    .summary-trend {
        display: flex;
        align-items: center;
        font-size: 14px;
        font-weight: 500;
    }
    .summary-trend i {
        font-size: 16px;
        margin-right: 2px;
    }
    .trend-up { color: #66bb6a; }
    .trend-down { color: #ef5350; }
    .card-primary .summary-icon { background-color: #1e88e5; }
    .card-secondary .summary-icon { background-color: #26a69a; }
    .card-warning .summary-icon { background-color: #ffa726; }
    .card-success .summary-icon { background-color: #66bb6a; }
    .card-info .summary-icon { background-color: #03a9f4; }
    .card-purple .summary-icon { background-color: #7e57c2; }
    .card-teal .summary-icon { background-color: #009688; }
    .card-primary { border-left: 4px solid #1e88e5; }
    .card-secondary { border-left: 4px solid #26a69a; }
    .card-warning { border-left: 4px solid #ffa726; }
    .card-success { border-left: 4px solid #66bb6a; }
    .card-info { border-left: 4px solid #03a9f4; }
    .card-purple { border-left: 4px solid #7e57c2; }
    .card-teal { border-left: 4px solid #009688; }
    .dashboard-main {
        display: grid;
        grid-template-columns: 1fr 1fr 1fr;
        gap: 20px;
    }
    .dashboard-column {
        display: flex;
        flex-direction: column;
        gap: 20px;
    }
    .dashboard-section {
        background-color: white;
        border-radius: 12px;
        box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
        overflow: hidden;
    }
    .section-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding: 16px 20px;
        border-bottom: 1px solid #f1f1f1;
    }
    .section-header h3 {
        margin: 0;
        font-size: 18px;
        font-weight: 600;
        color: #333;
        display: flex;
        align-items: center;
    }
    .section-header h3 i {
        font-size: 20px;
        margin-right: 6px;
        color: #555;
    }
    .section-link {
        color: #1e88e5;
        font-size: 14px;
        font-weight: 500;
        transition: all 0.2s;
    }
    .section-link:hover { text-decoration: underline; }
    .quick-actions {
        display: grid;
        grid-template-columns: repeat(auto-fill, minmax(calc(50% - 10px), 1fr));
        gap: 15px;
        padding: 20px;
    }
    .action-card {
        display: flex;
        flex-direction: column;
        align-items: center;
        padding: 15px;
        border-radius: 10px;
        background-color: #f8f9fa;
        transition: all 0.3s;
        text-decoration: none;
        color: inherit;
    }
    .action-card:hover {
        background-color: #fff;
        transform: translateY(-5px);
        box-shadow: 0 8px 15px rgba(0, 0, 0, 0.1);
    }
    .action-icon {
        width: 50px;
        height: 50px;
        border-radius: 10px;
        display: flex;
        align-items: center;
        justify-content: center;
        margin-bottom: 12px;
    }
    .action-icon i {
        font-size: 24px;
        color: white;
    }
    .action-title {
        font-size: 16px;
        font-weight: 600;
        margin: 0 0 5px 0;
        color: #333;
    }
    .action-description {
        font-size: 13px;
        color: #6c757d;
        margin: 0;
        text-align: center;
    }
    .primary-bg { background-color: #1e88e5; }
    .secondary-bg { background-color: #26a69a; }
    .warning-bg { background-color: #ffa726; }
    .danger-bg { background-color: #ef5350; }
    .info-bg { background-color: #03a9f4; }
    .success-bg { background-color: #66bb6a; }
    .purple-bg { background-color: #7e57c2; }
    .teal-bg { background-color: #009688; }
    .chart-container {
        padding: 20px;
        height: 260px;
        position: relative;
    }
    .activity-list { padding: 10px 0; }
    .activity-item {
        display: flex;
        padding: 15px 20px;
        border-bottom: 1px solid #f1f1f1;
        transition: background-color 0.2s;
    }
    .activity-item:last-child { border-bottom: none; }
    .activity-item:hover { background-color: #f8f9fa; }
    .activity-icon {
        width: 40px;
        height: 40px;
        border-radius: 8px;
        display: flex;
        align-items: center;
        justify-content: center;
        margin-right: 15px;
        flex-shrink: 0;
    }
    .activity-icon i {
        font-size: 20px;
        color: white;
    }
    .activity-content { flex: 1; }
    .activity-header {
        display: flex;
        justify-content: space-between;
        margin-bottom: 5px;
    }
    .activity-title { font-weight: 600; color: #333; }
    .activity-date { font-size: 12px; color: #6c757d; }
    .activity-details {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 5px;
    }
    .activity-amount { font-weight: 600; color: #555; }
    .activity-actions { margin-top: 5px; }
    .action-link {
        display: inline-flex;
        align-items: center;
        font-size: 13px;
        color: #1e88e5;
        margin-right: 12px;
    }
    .action-link i { font-size: 16px; margin-right: 3px; }
    .action-link:hover { text-decoration: underline; }
    .ranking-list { padding: 10px 0; }
    .ranking-item {
        display: flex;
        align-items: center;
        padding: 12px 20px;
        border-bottom: 1px solid #f1f1f1;
    }
    .ranking-item:last-child { border-bottom: none; }
    .ranking-position {
        width: 30px;
        height: 30px;
        background-color: #f1f1f1;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        font-weight: 600;
        margin-right: 12px;
    }
    .ranking-info { flex: 1; }
    .ranking-name { font-weight: 500; }
    .ranking-value { font-weight: 600; color: #333; }
    .ranking-item:nth-child(1) .ranking-position { background-color: #FFD700; color: #333; }
    .ranking-item:nth-child(2) .ranking-position { background-color: #C0C0C0; color: #333; }
    .ranking-item:nth-child(3) .ranking-position { background-color: #CD7F32; color: #333; }
    .empty-state {
        display: flex;
        flex-direction: column;
        align-items: center;
        justify-content: center;
        padding: 40px 20px;
        color: #adb5bd;
    }
    .empty-state i { font-size: 48px; margin-bottom: 10px; }
    .empty-state p { font-size: 16px; margin: 0; }
    
    /* Nuevos estilos para solicitudes y historial */
    .solicitudes-list { padding: 10px 0; }
    .solicitud-item {
        padding: 15px 20px;
        border-bottom: 1px solid #f1f1f1;
        transition: background-color 0.2s;
    }
    .solicitud-item:last-child { border-bottom: none; }
    .solicitud-item:hover { background-color: #f8f9fa; }
    .solicitud-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 8px;
    }
    .solicitud-title { font-weight: 600; color: #333; }
    .solicitud-content { margin-bottom: 10px; }
    .solicitud-content p {
        margin: 0;
        color: #555;
        font-size: 14px;
    }
    .solicitud-footer {
        display: flex;
        align-items: center;
        justify-content: space-between;
        flex-wrap: wrap;
        font-size: 13px;
        color: #6c757d;
    }
    .solicitud-date {
        display: flex;
        align-items: center;
    }
    .solicitud-date i {
        font-size: 14px;
        margin-right: 3px;
    }
    .solicitud-motivo {
        background-color: #f1f1f1;
        padding: 2px 8px;
        border-radius: 20px;
        font-size: 12px;
    }
    .view-link {
        display: flex;
        align-items: center;
        color: #1e88e5;
        margin-left: auto;
    }
    .view-link i {
        font-size: 14px;
        margin-right: 3px;
    }
    .view-link:hover { text-decoration: underline; }
    
    .historial-list { padding: 5px 0; }
    .historial-item {
        display: flex;
        padding: 12px 20px;
        border-bottom: 1px solid #f1f1f1;
        transition: background-color 0.2s;
    }
    .historial-item:last-child { border-bottom: none; }
    .historial-item:hover { background-color: #f8f9fa; }
    .historial-icon {
        width: 32px;
        height: 32px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        margin-right: 12px;
        flex-shrink: 0;
    }
    .historial-icon i {
        font-size: 16px;
        color: white;
    }
    .historial-content { flex: 1; }
    .historial-header {
        display: flex;
        justify-content: space-between;
        margin-bottom: 2px;
    }
    .historial-user { font-weight: 500; color: #333; font-size: 14px; }
    .historial-date { font-size: 12px; color: #6c757d; }
    .historial-description {
        font-size: 13px;
        color: #555;
    }
    
    .view-all-link {
        font-size: 12px;
        color: #1e88e5;
        margin-left: 10px;
    }
    .view-all-link:hover { text-decoration: underline; }
    
    @media (max-width: 1200px) {
        .dashboard-main { grid-template-columns: 1fr 1fr; }
        .dashboard-column:last-child { grid-column: span 2; }
    }
    @media (max-width: 992px) {
        .dashboard-main { grid-template-columns: 1fr; }
        .dashboard-column:last-child { grid-column: auto; }
        .content {
            padding: 15px;
        }
        .header {
            flex-direction: column;
            align-items: flex-start;
        }
        .header > div {
            width: 100%;
        }
    }
    @media (max-width: 768px) {
        .summary-cards { grid-template-columns: 1fr 1fr; }
        .sidebar {
            transform: translateX(-100%);
            box-shadow: none;
        }
        .content {
            margin-left: 0;
        }
        .sidebar.active {
            transform: translateX(0);
            box-shadow: 0 0 15px rgba(0, 0, 0, 0.2);
        }
        .toggle-sidebar {
            display: flex;
            background-color: var(--primary);
        }
        .header {
            margin-top: 30px;
        }
        .user-profile {
            width: auto;
        }
    }
    @media (max-width: 576px) {
        .summary-cards { grid-template-columns: 1fr; }
        .welcome-section { flex-direction: column; align-items: flex-start; }
        .welcome-actions { margin-top: 10px; }
        .summary-card { padding: 15px; }
        .summary-value { font-size: 22px; }
        .welcome-section { padding: 10px 15px; }
        .welcome-section h2 { font-size: 20px; }
        .periodo-label { font-size: 12px; }
        .quick-actions { grid-template-columns: 1fr; }
        .action-card { padding: 12px; }
        .action-icon { width: 40px; height: 40px; }
        .activity-item { padding: 10px 15px; }
        .activity-icon { width: 35px; height: 35px; }
        .activity-title { font-size: 14px; }
        .activity-amount { font-size: 14px; }
        .ranking-item { padding: 10px 15px; }
        .ranking-position { width: 25px; height: 25px; }
        .ranking-name { font-size: 14px; }
    }
</style>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        const categoryChartEl = document.getElementById('categoryChart');
        if (categoryChartEl) {
            const categoryData = {
                labels: <?php echo json_encode($categorias); ?>,
                datasets: [{
                    label: 'Gastos por categoría',
                    data: <?php echo json_encode($valores_categorias); ?>,
                    backgroundColor: [
                        'rgba(54, 162, 235, 0.7)',
                        'rgba(75, 192, 192, 0.7)',
                        'rgba(255, 205, 86, 0.7)',
                        'rgba(255, 99, 132, 0.7)',
                        'rgba(153, 102, 255, 0.7)'
                    ],
                    borderWidth: 1
                }]
            };
            const categoryChart = new Chart(categoryChartEl, {
                type: 'doughnut',
                data: categoryData,
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: { position: 'right', labels: { boxWidth: 15, padding: 15 } },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    const label = context.label || '';
                                    const value = context.parsed;
                                    const total = context.dataset.data.reduce((a, b) => a + b, 0);
                                    const percentage = total > 0 ? Math.round((value / total) * 100) : 0;
                                    return `${label}: ${formatoMoneda(value)} (${percentage}%)`;
                                }
                            }
                        }
                    }
                }
            });
        }
        
        const motivosChartEl = document.getElementById('motivosChart');
        if (motivosChartEl) {
            const motivosData = {
                labels: <?php echo json_encode($motivos); ?>,
                datasets: [{
                    label: 'Solicitudes por motivo',
                    data: <?php echo json_encode($valores_motivos); ?>,
                    backgroundColor: [
                        'rgba(255, 99, 132, 0.7)',
                        'rgba(54, 162, 235, 0.7)',
                        'rgba(255, 205, 86, 0.7)',
                        'rgba(75, 192, 192, 0.7)',
                        'rgba(153, 102, 255, 0.7)'
                    ],
                    borderWidth: 1
                }]
            };
            const motivosChart = new Chart(motivosChartEl, {
                type: 'pie',
                data: motivosData,
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: { position: 'bottom', labels: { boxWidth: 12, padding: 15 } },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    const label = context.label || '';
                                    const value = context.parsed;
                                    const total = context.dataset.data.reduce((a, b) => a + b, 0);
                                    const percentage = total > 0 ? Math.round((value / total) * 100) : 0;
                                    return `${label}: ${value} (${percentage}%)`;
                                }
                            }
                        }
                    }
                }
            });
        }
        
        const repuestosChartEl = document.getElementById('repuestosChart');
        if (repuestosChartEl) {
            const repuestosData = {
                labels: <?php echo json_encode($repuestos_nombres); ?>,
                datasets: [{
                    label: 'Número de solicitudes',
                    data: <?php echo json_encode($repuestos_solicitudes); ?>,
                    backgroundColor: 'rgba(54, 162, 235, 0.7)',
                    borderColor: 'rgba(54, 162, 235, 1)',
                    borderWidth: 1
                }]
            };
            const repuestosChart = new Chart(repuestosChartEl, {
                type: 'bar',
                data: repuestosData,
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    indexAxis: 'y',
                    plugins: {
                        legend: { display: false },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    return `${context.parsed.x} solicitudes`;
                                }
                            }
                        }
                    },
                    scales: {
                        x: {
                            beginAtZero: true,
                            ticks: {
                                precision: 0
                            }
                        }
                    }
                }
            });
        }
    });

    function formatoMoneda(valor) {
        return '$' + parseInt(valor).toLocaleString('es-CL');
    }
</script>

<?php require_once 'includes/footer.php'; ?>