<?php
/**
 * Módulo de Configuraciones del Sistema
 */

$page_title = 'Configuraciones del Sistema';

require_once 'includes/config.php';
require_once 'includes/functions.php';
require_once 'includes/permisos.php';

// Verificar sesión y permisos
verificarSesion();
$user_id = $_SESSION['user_id'];
$user_rol = $_SESSION['user_rol'] ?? '';

// Solo permitir acceso a super_admin
if ($user_rol !== 'super_admin') {
    redirigirConMensaje('dashboard.php', "Acceso denegado. Solo administradores del sistema pueden acceder a esta sección.", 'danger');
    exit;
}

// Verificar permiso específico
if (!tienePermiso('configuracion_sistema')) {
    redirigirConMensaje('dashboard.php', "No tiene los permisos necesarios para acceder a esta sección.", 'danger');
    exit;
}

$conn = getDatabaseConnection();
$mensaje = '';
$tipo_mensaje = '';

// Procesar formulario de actualización
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $conn->beginTransaction();
        
        foreach ($_POST['config'] as $clave => $valor) {
            // Sanitizar la clave
            $clave = filter_var($clave, FILTER_SANITIZE_STRING);
            
            // Procesar archivos si es una configuración de tipo imagen
            if (isset($_FILES['config_file']) && isset($_FILES['config_file']['name'][$clave]) && !empty($_FILES['config_file']['name'][$clave])) {
                $archivo = $_FILES['config_file']['tmp_name'][$clave];
                $nombre = $_FILES['config_file']['name'][$clave];
                $extension = pathinfo($nombre, PATHINFO_EXTENSION);
                
                // Validar extensión
                $extensiones_permitidas = ['jpg', 'jpeg', 'png', 'gif', 'ico'];
                if (!in_array(strtolower($extension), $extensiones_permitidas)) {
                    throw new Exception("Tipo de archivo no permitido para $clave");
                }
                
                // Generar nombre único
                $nombre_archivo = uniqid() . "." . $extension;
                $ruta_destino = "uploads/sistema/" . $nombre_archivo;
                
                // Crear directorio si no existe
                if (!is_dir("uploads/sistema/")) {
                    mkdir("uploads/sistema/", 0755, true);
                }
                
                // Mover archivo
                if (move_uploaded_file($archivo, $ruta_destino)) {
                    $valor = $nombre_archivo;
                } else {
                    throw new Exception("Error al subir el archivo para $clave");
                }
            }
            
            // Actualizar valor en la base de datos
            $stmt = $conn->prepare("
                UPDATE configuraciones 
                SET valor = :valor, 
                    updated_at = CURRENT_TIMESTAMP 
                WHERE clave = :clave
            ");
            
            $stmt->execute([
                ':valor' => $valor,
                ':clave' => $clave
            ]);
        }
        
        $conn->commit();
        registrarActividad($conn, $user_id, 'configuracion', 'Actualización de configuraciones del sistema');
        $mensaje = "Configuraciones actualizadas correctamente.";
        $tipo_mensaje = "success";
        
    } catch (Exception $e) {
        $conn->rollBack();
        $mensaje = "Error: " . $e->getMessage();
        $tipo_mensaje = "danger";
    }
}

// Obtener configuraciones agrupadas por categoría
$configuraciones = [];
$stmt = $conn->query("
    SELECT * 
    FROM configuraciones 
    ORDER BY categoria, clave
");

while ($config = $stmt->fetch(PDO::FETCH_ASSOC)) {
    $configuraciones[$config['categoria']][] = $config;
}

// Incluir el header
require_once 'includes/header.php';
?>

<style>
.config-section {
    background: white;
    border-radius: 8px;
    padding: 20px;
    margin-bottom: 20px;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.config-section h3 {
    margin-bottom: 20px;
    color: #333;
    border-bottom: 2px solid #eee;
    padding-bottom: 10px;
}

.config-item {
    margin-bottom: 15px;
}

.config-item label {
    font-weight: 500;
    margin-bottom: 5px;
    display: block;
}

.config-item small {
    color: #666;
    display: block;
    margin-top: 5px;
}

.color-preview {
    width: 30px;
    height: 30px;
    border-radius: 4px;
    display: inline-block;
    vertical-align: middle;
    margin-left: 10px;
    border: 1px solid #ddd;
}

.image-preview {
    max-width: 200px;
    max-height: 100px;
    margin-top: 10px;
    border-radius: 4px;
    border: 1px solid #ddd;
}

.theme-preview {
    padding: 15px;
    border-radius: 8px;
    margin-top: 15px;
    transition: all 0.3s ease;
}
</style>

<?php if ($mensaje): ?>
<div class="alert alert-<?php echo $tipo_mensaje; ?> alert-dismissible fade show">
    <?php echo $mensaje; ?>
    <button type="button" class="close" data-dismiss="alert">
        <span>&times;</span>
    </button>
</div>
<?php endif; ?>

<div class="card">
    <div class="card-header">
        <h2 class="card-title">Configuraciones del Sistema</h2>
    </div>
    
    <div class="card-body">
        <form method="POST" enctype="multipart/form-data">
            <?php foreach ($configuraciones as $categoria => $configs): ?>
            <div class="config-section">
                <h3>
                    <i class="material-icons"><?php echo getCategoryIcon($categoria); ?></i>
                    <?php echo ucfirst($categoria); ?>
                </h3>
                
                <?php foreach ($configs as $config): ?>
                <div class="config-item">
                    <label for="config_<?php echo $config['clave']; ?>">
                        <?php echo $config['descripcion']; ?>
                    </label>
                    
                    <?php switch ($config['tipo']):
                        case 'texto': ?>
                            <input type="text" 
                                   class="form-control"
                                   name="config[<?php echo $config['clave']; ?>]"
                                   id="config_<?php echo $config['clave']; ?>"
                                   value="<?php echo htmlspecialchars($config['valor']); ?>"
                                   <?php echo $config['modificable'] ? '' : 'readonly'; ?>>
                            <?php break;
                            
                        case 'numero': ?>
                            <input type="number" 
                                   class="form-control"
                                   name="config[<?php echo $config['clave']; ?>]"
                                   id="config_<?php echo $config['clave']; ?>"
                                   value="<?php echo $config['valor']; ?>"
                                   <?php echo $config['modificable'] ? '' : 'readonly'; ?>>
                            <?php break;
                            
                        case 'booleano': ?>
                            <select class="form-control"
                                    name="config[<?php echo $config['clave']; ?>]"
                                    id="config_<?php echo $config['clave']; ?>"
                                    <?php echo $config['modificable'] ? '' : 'disabled'; ?>>
                                <option value="1" <?php echo $config['valor'] == '1' ? 'selected' : ''; ?>>Sí</option>
                                <option value="0" <?php echo $config['valor'] == '0' ? 'selected' : ''; ?>>No</option>
                            </select>
                            <?php break;
                            
                        case 'color': ?>
                            <div class="input-group">
                                <input type="color" 
                                       class="form-control"
                                       name="config[<?php echo $config['clave']; ?>]"
                                       id="config_<?php echo $config['clave']; ?>"
                                       value="<?php echo $config['valor']; ?>"
                                       <?php echo $config['modificable'] ? '' : 'disabled'; ?>>
                                <div class="color-preview" style="background-color: <?php echo $config['valor']; ?>"></div>
                            </div>
                            <?php break;
                            
                        case 'imagen': ?>
                            <div class="custom-file">
                                <input type="file" 
                                       class="custom-file-input"
                                       name="config_file[<?php echo $config['clave']; ?>]"
                                       id="config_file_<?php echo $config['clave']; ?>"
                                       accept="image/*"
                                       <?php echo $config['modificable'] ? '' : 'disabled'; ?>>
                                <label class="custom-file-label" for="config_file_<?php echo $config['clave']; ?>">
                                    Seleccionar archivo
                                </label>
                            </div>
                            <input type="hidden" 
                                   name="config[<?php echo $config['clave']; ?>]"
                                   value="<?php echo $config['valor']; ?>">
                            <?php if ($config['valor']): ?>
                                <img src="uploads/sistema/<?php echo $config['valor']; ?>" 
                                     class="image-preview" 
                                     alt="<?php echo $config['descripcion']; ?>">
                            <?php endif; ?>
                            <?php break;
                            
                        default: ?>
                            <input type="text" 
                                   class="form-control"
                                   name="config[<?php echo $config['clave']; ?>]"
                                   id="config_<?php echo $config['clave']; ?>"
                                   value="<?php echo htmlspecialchars($config['valor']); ?>"
                                   <?php echo $config['modificable'] ? '' : 'readonly'; ?>>
                    <?php endswitch; ?>
                    
                    <?php if ($categoria === 'apariencia'): ?>
                        <div class="theme-preview">
                            Vista previa del tema
                        </div>
                    <?php endif; ?>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endforeach; ?>
            
            <div class="form-group text-center mt-4">
                <button type="submit" class="btn btn-primary">
                    <i class="material-icons">save</i>
                    Guardar Configuraciones
                </button>
                <a href="admin_sistema.php" class="btn btn-secondary">
                    <i class="material-icons">arrow_back</i>
                    Volver
                </a>
            </div>
        </form>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Actualizar label de archivos seleccionados
    document.querySelectorAll('.custom-file-input').forEach(input => {
        input.addEventListener('change', function(e) {
            const fileName = e.target.files[0].name;
            const label = e.target.nextElementSibling;
            label.textContent = fileName;
        });
    });
    
    // Preview de colores
    document.querySelectorAll('input[type="color"]').forEach(input => {
        input.addEventListener('input', function(e) {
            const preview = e.target.parentElement.querySelector('.color-preview');
            if (preview) {
                preview.style.backgroundColor = e.target.value;
            }
            
            // Actualizar vista previa del tema
            updateThemePreview();
        });
    });
    
    // Actualizar vista previa del tema cuando cambie el modo oscuro
    document.querySelector('#config_modo_oscuro')?.addEventListener('change', updateThemePreview);
    
    function updateThemePreview() {
        const colorPrimario = document.querySelector('#config_tema_color_primario')?.value || '#1e88e5';
        const colorSecundario = document.querySelector('#config_tema_color_secundario')?.value || '#ff4081';
        const modoOscuro = document.querySelector('#config_modo_oscuro')?.value === '1';
        
        document.querySelectorAll('.theme-preview').forEach(preview => {
            preview.style.backgroundColor = modoOscuro ? '#333' : '#fff';
            preview.style.color = modoOscuro ? '#fff' : '#333';
            preview.style.borderLeft = `4px solid ${colorPrimario}`;
            preview.style.boxShadow = `0 2px 4px ${colorSecundario}40`;
        });
    }
    
    // Inicializar vista previa del tema
    updateThemePreview();
});
</script>

<?php
/**
 * Obtiene el icono correspondiente a cada categoría
 */
function getCategoryIcon($categoria) {
    $icons = [
        'general' => 'settings',
        'apariencia' => 'palette',
        'correo' => 'email',
        'sistema' => 'computer',
        'documentos' => 'description',
        'notificaciones' => 'notifications'
    ];
    
    return $icons[$categoria] ?? 'settings';
}

// Incluir el footer
require_once 'includes/footer.php';
?> 