<?php
/**
 * API AJAX para operaciones relacionadas con máquinas y repuestos
 * Maneja solicitudes asíncronas para el módulo de máquinas
 */

header('Content-Type: application/json; charset=utf-8');

require_once 'includes/config.php';
require_once 'includes/functions.php';

// Verificar sesión
verificarSesion();
$user_id = $_SESSION['user_id'];
$user_rol = $_SESSION['user_rol'] ?? '';
$is_admin = ($user_rol === 'admin' || $user_rol === 'super_admin');

$action = $_GET['action'] ?? '';
$response = ['success' => false, 'message' => 'Acción no válida'];

try {
    $conn = new PDO("mysql:host=$db_host;dbname=$db_name;charset=utf8mb4", $db_user, $db_pass, $db_options);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    switch ($action) {
        case 'get_images':
            // Obtener imágenes de una máquina
            $maquina_id = isset($_GET['maquina_id']) ? intval($_GET['maquina_id']) : 0;
            
            if ($maquina_id <= 0) {
                $response = ['success' => false, 'message' => 'ID de máquina no válido'];
                break;
            }
            
            $stmt = $conn->prepare("
                SELECT * FROM maquina_imagenes
                WHERE maquina_id = :maquina_id
                ORDER BY es_principal DESC, fecha_subida DESC
            ");
            $stmt->bindParam(':maquina_id', $maquina_id);
            $stmt->execute();
            $imagenes = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $response = [
                'success' => true,
                'imagenes' => $imagenes,
                'count' => count($imagenes)
            ];
            break;
            
        case 'get_repuestos':
            // Obtener repuestos asociados a una máquina
            $maquina_id = isset($_GET['maquina_id']) ? intval($_GET['maquina_id']) : 0;
            
            if ($maquina_id <= 0) {
                $response = ['success' => false, 'message' => 'ID de máquina no válido'];
                break;
            }
            
            $stmt = $conn->prepare("
                SELECT r.id, r.codigo, r.nombre, r.numero_pieza, r.disponible, r.stock, r.costo,
                       cr.nombre as categoria_nombre
                FROM repuestos r
                INNER JOIN maquina_repuesto mr ON r.id = mr.repuesto_id
                LEFT JOIN categorias_repuestos cr ON r.categoria_id = cr.id
                WHERE mr.maquina_id = :maquina_id AND r.activo = 1
                ORDER BY r.codigo
            ");
            $stmt->bindParam(':maquina_id', $maquina_id);
            $stmt->execute();
            $repuestos = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $response = [
                'success' => true,
                'repuestos' => $repuestos,
                'count' => count($repuestos)
            ];
            break;
            
        case 'search_repuestos':
            // Buscar repuestos por código o nombre
            $busqueda = isset($_GET['query']) ? trim($_GET['query']) : '';
            
            if (empty($busqueda)) {
                $response = ['success' => false, 'message' => 'Búsqueda vacía'];
                break;
            }
            
            $stmt = $conn->prepare("
                SELECT r.id, r.codigo, r.nombre, r.numero_pieza, r.disponible, r.stock, r.costo,
                       cr.nombre as categoria_nombre
                FROM repuestos r
                LEFT JOIN categorias_repuestos cr ON r.categoria_id = cr.id
                WHERE (r.codigo LIKE :busqueda OR r.nombre LIKE :busqueda) AND r.activo = 1
                ORDER BY r.codigo
                LIMIT 50
            ");
            $stmt->bindValue(':busqueda', '%' . $busqueda . '%');
            $stmt->execute();
            $repuestos = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $response = [
                'success' => true,
                'repuestos' => $repuestos,
                'count' => count($repuestos)
            ];
            break;
            
        case 'get_estados':
            // Obtener lista de estados predefinidos
            $stmt = $conn->prepare("
                SELECT id, nombre, descripcion
                FROM estados_maquina
                WHERE activo = 1
                ORDER BY nombre
            ");
            $stmt->execute();
            $estados = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $response = [
                'success' => true,
                'estados' => $estados
            ];
            break;
            
        case 'get_historial_estados':
            // Obtener historial de estados de una serie
            $serie_id = isset($_GET['serie_id']) ? intval($_GET['serie_id']) : 0;
            
            if ($serie_id <= 0) {
                $response = ['success' => false, 'message' => 'ID de serie no válido'];
                break;
            }
            
            $stmt = $conn->prepare("
                SELECT he.*, 
                       u.nombre as usuario_nombre,
                       u.apellido as usuario_apellido,
                       c.nombre as cliente_nombre,
                       c.rut as cliente_rut
                FROM historial_estados_series he
                LEFT JOIN users u ON he.usuario_id = u.id
                LEFT JOIN clientes c ON he.cliente_id = c.id
                WHERE he.serie_id = :serie_id
                ORDER BY he.fecha_cambio DESC
            ");
            $stmt->bindParam(':serie_id', $serie_id);
            $stmt->execute();
            $historial = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $response = [
                'success' => true,
                'historial' => $historial
            ];
            break;
            
        case 'get_serie_info':
            // Obtener información detallada de una serie
            $serie_id = isset($_GET['serie_id']) ? intval($_GET['serie_id']) : 0;
            
            if ($serie_id <= 0) {
                $response = ['success' => false, 'message' => 'ID de serie no válido'];
                break;
            }
            
            $stmt = $conn->prepare("
                SELECT sm.*,
                       m.nombre as maquina_nombre,
                       m.codigo as maquina_codigo,
                       p.nombre as proveedor_nombre,
                       p.rut as proveedor_rut,
                       c.nombre as cliente_nombre,
                       c.rut as cliente_rut
                FROM series_maquinas sm
                LEFT JOIN maquinas m ON sm.maquina_id = m.id
                LEFT JOIN proveedores p ON sm.proveedor_id = p.id
                LEFT JOIN clientes c ON sm.cliente_id = c.id
                WHERE sm.id = :serie_id
            ");
            $stmt->bindParam(':serie_id', $serie_id);
            $stmt->execute();
            $serie = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($serie) {
                $serie['historial_estados'] = json_decode($serie['historial_estados'], true);
                $response = [
                    'success' => true,
                    'serie' => $serie
                ];
            } else {
                $response = [
                    'success' => false,
                    'message' => 'Serie no encontrada'
                ];
            }
            break;
            
        case 'get_maquina':
            // Obtener detalles de una máquina
            $maquina_id = isset($_GET['maquina_id']) ? intval($_GET['maquina_id']) : 0;
            
            if ($maquina_id <= 0) {
                $response = ['success' => false, 'message' => 'ID de máquina no válido'];
                break;
            }
            
            $stmt = $conn->prepare("
                SELECT * FROM maquinas
                WHERE id = :maquina_id
            ");
            $stmt->bindParam(':maquina_id', $maquina_id);
            $stmt->execute();
            $maquina = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($maquina) {
                $response = [
                    'success' => true,
                    'maquina' => $maquina
                ];
            } else {
                $response = ['success' => false, 'message' => 'Máquina no encontrada'];
            }
            break;
            
        default:
            $response = ['success' => false, 'message' => 'Acción no soportada'];
            break;
    }
    
} catch (PDOException $e) {
    $response = [
        'success' => false,
        'message' => 'Error de base de datos: ' . $e->getMessage()
    ];
} catch (Exception $e) {
    $response = [
        'success' => false,
        'message' => 'Error: ' . $e->getMessage()
    ];
}

// Devolver respuesta JSON
echo json_encode($response);
exit;