<?php
/**
 * Módulo de Administración del Sistema
 * Solo accesible para super administradores
 */

$page_title = 'Administración del Sistema';

require_once 'includes/config.php';
require_once 'includes/functions.php';
require_once 'includes/permisos.php';

// Verificar sesión y permisos
verificarSesion();
$user_id = $_SESSION['user_id'];
$user_rol = $_SESSION['user_rol'] ?? '';

// Solo permitir acceso a super_admin
if ($user_rol !== 'super_admin') {
    redirigirConMensaje('dashboard.php', "Acceso denegado. Solo administradores del sistema pueden acceder a esta sección.", 'danger');
    exit;
}

// Verificar permiso específico
if (!tienePermiso('acceso_admin_sistema')) {
    redirigirConMensaje('dashboard.php', "No tiene los permisos necesarios para acceder a esta sección.", 'danger');
    exit;
}

// Determinar la acción a realizar
$action = $_GET['action'] ?? '';
$mensaje = '';
$tipo_mensaje = '';

try {
    $conn = getDatabaseConnection();

    switch ($action) {
        case 'limpiar_cache':
            if (tienePermiso('limpiar_cache')) {
                // Limpiar directorio de caché
                $cache_dir = __DIR__ . '/cache/';
                if (is_dir($cache_dir)) {
                    $files = glob($cache_dir . '*');
                    foreach ($files as $file) {
                        if (is_file($file)) {
                            unlink($file);
                        }
                    }
                }
                registrarActividad($conn, $user_id, 'limpiar_cache', 'Limpieza de caché del sistema');
                $mensaje = "Caché del sistema limpiado correctamente.";
                $tipo_mensaje = "success";
            }
            break;

        case 'eliminar_documento':
            if (tienePermiso('eliminar_documentos')) {
                $doc_id = $_POST['documento_id'] ?? 0;
                $tipo_doc = $_POST['tipo_documento'] ?? '';
                
                if ($doc_id && $tipo_doc) {
                    // Iniciar transacción
                    $conn->beginTransaction();
                    
                    try {
                        // Eliminar archivos asociados
                        $stmt = $conn->prepare("SELECT ruta FROM archivos WHERE {$tipo_doc}_id = :doc_id");
                        $stmt->bindParam(':doc_id', $doc_id);
                        $stmt->execute();
                        
                        while ($archivo = $stmt->fetch(PDO::FETCH_ASSOC)) {
                            if (file_exists($archivo['ruta'])) {
                                unlink($archivo['ruta']);
                            }
                        }
                        
                        // Eliminar registros de la base de datos
                        $stmt = $conn->prepare("DELETE FROM archivos WHERE {$tipo_doc}_id = :doc_id");
                        $stmt->bindParam(':doc_id', $doc_id);
                        $stmt->execute();
                        
                        $stmt = $conn->prepare("DELETE FROM {$tipo_doc} WHERE id = :doc_id");
                        $stmt->bindParam(':doc_id', $doc_id);
                        $stmt->execute();
                        
                        $conn->commit();
                        registrarActividad($conn, $user_id, 'eliminar_documento', 
                            "Eliminación de documento: {$tipo_doc} #{$doc_id}");
                        
                        $mensaje = "Documento eliminado correctamente.";
                        $tipo_mensaje = "success";
                    } catch (Exception $e) {
                        $conn->rollBack();
                        throw $e;
                    }
                }
            }
            break;

        case 'limpiar_bd':
            if (tienePermiso('limpiar_base_datos')) {
                $dias = $_POST['dias'] ?? 90;
                
                // Iniciar transacción
                $conn->beginTransaction();
                
                try {
                    // Eliminar registros antiguos del historial
                    $stmt = $conn->prepare("
                        DELETE FROM historial_cambios 
                        WHERE fecha_hora < DATE_SUB(NOW(), INTERVAL :dias DAY)
                    ");
                    $stmt->bindParam(':dias', $dias);
                    $stmt->execute();
                    
                    // Eliminar archivos temporales
                    $stmt = $conn->prepare("
                        DELETE FROM archivos 
                        WHERE temporal = 1 
                        AND fecha_subida < DATE_SUB(NOW(), INTERVAL 1 DAY)
                    ");
                    $stmt->execute();
                    
                    $conn->commit();
                    registrarActividad($conn, $user_id, 'limpiar_bd', 
                        "Limpieza de base de datos (registros > {$dias} días)");
                    
                    $mensaje = "Base de datos limpiada correctamente.";
                    $tipo_mensaje = "success";
                } catch (Exception $e) {
                    $conn->rollBack();
                    throw $e;
                }
            }
            break;

        case 'backup':
            if (tienePermiso('gestionar_respaldos')) {
                // Generar nombre del archivo de respaldo
                $backup_file = 'backup_' . date('Y-m-d_H-i-s') . '.sql';
                $backup_path = __DIR__ . '/backups/' . $backup_file;
                
                // Crear directorio si no existe
                if (!is_dir(__DIR__ . '/backups/')) {
                    mkdir(__DIR__ . '/backups/', 0755, true);
                }
                
                try {
                    // Obtener todas las tablas
                    $tables = [];
                    $result = $conn->query("SHOW TABLES");
                    while ($row = $result->fetch(PDO::FETCH_NUM)) {
                        $tables[] = $row[0];
                    }
                    
                    $output = "-- Respaldo de base de datos\n";
                    $output .= "-- Fecha: " . date('Y-m-d H:i:s') . "\n";
                    $output .= "-- Base de datos: " . $db_name . "\n\n";
                    $output .= "SET FOREIGN_KEY_CHECKS=0;\n\n";
                    
                    // Procesar cada tabla
                    foreach ($tables as $table) {
                        // Obtener estructura de la tabla
                        $result = $conn->query("SHOW CREATE TABLE `$table`");
                        $row = $result->fetch(PDO::FETCH_NUM);
                        $output .= "\n\n" . $row[1] . ";\n\n";
                        
                        // Obtener datos de la tabla
                        $result = $conn->query("SELECT * FROM `$table`");
                        while ($row = $result->fetch(PDO::FETCH_NUM)) {
                            $output .= "INSERT INTO `$table` VALUES (";
                            for ($i = 0; $i < count($row); $i++) {
                                if ($row[$i] === null) {
                                    $output .= 'NULL';
                                } else {
                                    $row[$i] = addslashes($row[$i]);
                                    $row[$i] = str_replace("\n", "\\n", $row[$i]);
                                    $output .= '"' . $row[$i] . '"';
                                }
                                if ($i < (count($row) - 1)) {
                                    $output .= ',';
                                }
                            }
                            $output .= ");\n";
                        }
                    }
                    
                    $output .= "\n\nSET FOREIGN_KEY_CHECKS=1;";
                    
                    // Guardar el archivo
                    if (file_put_contents($backup_path, $output)) {
                        registrarActividad($conn, $user_id, 'backup', 
                            "Generación de respaldo de base de datos: {$backup_file}");
                        $mensaje = "Respaldo generado correctamente.";
                        $tipo_mensaje = "success";
                    } else {
                        throw new Exception("Error al escribir el archivo de respaldo.");
                    }
                } catch (Exception $e) {
                    throw new Exception("Error al generar el respaldo: " . $e->getMessage());
                }
            }
            break;
    }
} catch (Exception $e) {
    $mensaje = "Error: " . $e->getMessage();
    $tipo_mensaje = "danger";
}

// Obtener estadísticas del sistema
$estadisticas = [
    'usuarios' => [
        'total' => $conn->query("SELECT COUNT(*) FROM users")->fetchColumn(),
        'activos' => $conn->query("SELECT COUNT(*) FROM users WHERE activo = 1")->fetchColumn(),
        'inactivos' => $conn->query("SELECT COUNT(*) FROM users WHERE activo = 0")->fetchColumn(),
        'admin' => $conn->query("SELECT COUNT(*) FROM users WHERE rol IN ('admin', 'super_admin')")->fetchColumn()
    ],
    'documentos' => [
        'rendiciones' => [
            'total' => $conn->query("SELECT COUNT(*) FROM rendiciones")->fetchColumn(),
            'pendientes' => $conn->query("SELECT COUNT(*) FROM rendiciones WHERE estado_id = 1")->fetchColumn(),
            'aprobadas' => $conn->query("SELECT COUNT(*) FROM rendiciones WHERE estado_id = 3")->fetchColumn()
        ],
        'ordenes_compra' => [
            'total' => $conn->query("SELECT COUNT(*) FROM ordenes_compra")->fetchColumn(),
            'pendientes' => $conn->query("SELECT COUNT(*) FROM ordenes_compra WHERE estado_id = 1")->fetchColumn(),
            'aprobadas' => $conn->query("SELECT COUNT(*) FROM ordenes_compra WHERE estado_id = 3")->fetchColumn()
        ]
    ],
    'sistema' => [
        'espacio_total' => disk_total_space(UPLOAD_DIR),
        'espacio_libre' => disk_free_space(UPLOAD_DIR),
        'espacio_usado' => array_sum(array_map('filesize', glob(UPLOAD_DIR . '*'))),
        'archivos_total' => count(glob(UPLOAD_DIR . '*')),
        'version_php' => PHP_VERSION,
        'version_mysql' => $conn->query('SELECT VERSION()')->fetchColumn()
    ]
];

// Incluir el header
require_once 'includes/header.php';
?>

<style>
    .admin-dashboard {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
        gap: 20px;
        margin-bottom: 30px;
    }
    
    .stat-card {
        background: #fff;
        border-radius: 8px;
        padding: 20px;
        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        transition: transform 0.2s;
    }
    
    .stat-card:hover {
        transform: translateY(-5px);
        box-shadow: 0 4px 8px rgba(0,0,0,0.2);
    }
    
    .stat-card h3 {
        margin: 0 0 15px;
        color: #333;
        font-size: 18px;
        display: flex;
        align-items: center;
        gap: 8px;
    }
    
    .stat-card h3 i {
        color: #1e88e5;
    }
    
    .stat-value {
        font-size: 24px;
        font-weight: bold;
        color: #1e88e5;
        margin-bottom: 10px;
    }
    
    .stat-detail {
        font-size: 14px;
        color: #666;
        display: flex;
        justify-content: space-between;
        margin-top: 5px;
    }
    
    .progress-bar {
        height: 6px;
        background: #e0e0e0;
        border-radius: 3px;
        margin-top: 10px;
    }
    
    .progress-bar .fill {
        height: 100%;
        background: #1e88e5;
        border-radius: 3px;
        transition: width 0.3s ease;
    }
    
    .admin-tabs {
        display: flex;
        gap: 10px;
        margin-bottom: 20px;
        border-bottom: 2px solid #e0e0e0;
        padding-bottom: 10px;
    }
    
    .admin-tab {
        padding: 8px 16px;
        border-radius: 4px;
        cursor: pointer;
        transition: all 0.2s;
    }
    
    .admin-tab.active {
        background: #1e88e5;
        color: white;
    }
    
    .admin-tab:hover:not(.active) {
        background: #f5f5f5;
    }
</style>

<?php if ($mensaje): ?>
<div class="alert alert-<?php echo $tipo_mensaje; ?>">
    <?php echo $mensaje; ?>
</div>
<?php endif; ?>

<div class="card">
    <div class="panel-header">
        <h2 class="panel-title">Panel de Administración del Sistema</h2>
        <div class="admin-tabs">
            <div class="admin-tab active" data-tab="dashboard">Dashboard</div>
            <div class="admin-tab" data-tab="users">Usuarios</div>
            <div class="admin-tab" data-tab="documents">Documentos</div>
            <div class="admin-tab" data-tab="system">Sistema</div>
            <div class="admin-tab" data-tab="logs">Logs</div>
        </div>
    </div>
    
    <!-- Dashboard de estadísticas -->
    <div class="admin-dashboard" id="tab-dashboard">
        <div class="stat-card">
            <h3><i class="material-icons">people</i> Usuarios</h3>
            <div class="stat-value"><?php echo number_format($estadisticas['usuarios']['total']); ?></div>
            <div class="stat-detail">
                <span>Activos: <?php echo number_format($estadisticas['usuarios']['activos']); ?></span>
                <span>Admins: <?php echo number_format($estadisticas['usuarios']['admin']); ?></span>
            </div>
            <div class="progress-bar">
                <div class="fill" style="width: <?php echo ($estadisticas['usuarios']['activos'] / max(1, $estadisticas['usuarios']['total'])) * 100; ?>%"></div>
            </div>
        </div>
        
        <div class="stat-card">
            <h3><i class="material-icons">description</i> Rendiciones</h3>
            <div class="stat-value"><?php echo number_format($estadisticas['documentos']['rendiciones']['total']); ?></div>
            <div class="stat-detail">
                <span>Pendientes: <?php echo number_format($estadisticas['documentos']['rendiciones']['pendientes']); ?></span>
                <span>Aprobadas: <?php echo number_format($estadisticas['documentos']['rendiciones']['aprobadas']); ?></span>
            </div>
            <div class="progress-bar">
                <div class="fill" style="width: <?php echo ($estadisticas['documentos']['rendiciones']['aprobadas'] / max(1, $estadisticas['documentos']['rendiciones']['total'])) * 100; ?>%"></div>
            </div>
        </div>
        
        <div class="stat-card">
            <h3><i class="material-icons">shopping_cart</i> Órdenes de Compra</h3>
            <div class="stat-value"><?php echo number_format($estadisticas['documentos']['ordenes_compra']['total']); ?></div>
            <div class="stat-detail">
                <span>Pendientes: <?php echo number_format($estadisticas['documentos']['ordenes_compra']['pendientes']); ?></span>
                <span>Aprobadas: <?php echo number_format($estadisticas['documentos']['ordenes_compra']['aprobadas']); ?></span>
            </div>
            <div class="progress-bar">
                <div class="fill" style="width: <?php echo ($estadisticas['documentos']['ordenes_compra']['aprobadas'] / max(1, $estadisticas['documentos']['ordenes_compra']['total'])) * 100; ?>%"></div>
            </div>
        </div>
        
        <div class="stat-card">
            <h3><i class="material-icons">storage</i> Almacenamiento</h3>
            <div class="stat-value"><?php echo formatBytes($estadisticas['sistema']['espacio_usado']); ?></div>
            <div class="stat-detail">
                <span>Total: <?php echo formatBytes($estadisticas['sistema']['espacio_total']); ?></span>
                <span>Libre: <?php echo formatBytes($estadisticas['sistema']['espacio_libre']); ?></span>
            </div>
            <div class="progress-bar">
                <div class="fill" style="width: <?php echo ($estadisticas['sistema']['espacio_usado'] / max(1, $estadisticas['sistema']['espacio_total'])) * 100; ?>%"></div>
            </div>
        </div>
    </div>
    
    <!-- Sección de Usuarios -->
    <div class="admin-section" id="tab-users" style="display: none;">
        <div class="admin-actions">
            <div class="action-card">
                <h4><i class="material-icons">person_add</i> Gestión de Usuarios</h4>
                <div class="action-buttons">
                    <a href="usuarios.php?action=new" class="btn btn-primary">
                        <i class="material-icons">add</i> Nuevo Usuario
                    </a>
                    <a href="usuarios.php" class="btn btn-info">
                        <i class="material-icons">list</i> Ver Usuarios
                    </a>
                </div>
            </div>
            
            <div class="action-card">
                <h4><i class="material-icons">admin_panel_settings</i> Roles y Permisos</h4>
                <div class="action-buttons">
                    <a href="roles.php" class="btn btn-primary">
                        <i class="material-icons">security</i> Gestionar Roles
                    </a>
                    <a href="permisos.php" class="btn btn-info">
                        <i class="material-icons">policy</i> Gestionar Permisos
                    </a>
                </div>
            </div>
            
            <?php if (tienePermiso('bloquear_usuarios')): ?>
            <div class="action-card">
                <h4><i class="material-icons">block</i> Bloqueo de Usuarios</h4>
                <form method="POST" action="?action=bloquear_usuario" class="action-form">
                    <div class="form-group">
                        <select name="user_id" class="form-control" required>
                            <option value="">Seleccionar usuario...</option>
                            <?php
                            $stmt = $conn->query("SELECT id, nombre, apellido, rut FROM users WHERE rol != 'super_admin' ORDER BY nombre");
                            while ($user = $stmt->fetch()) {
                                echo "<option value='{$user['id']}'>{$user['nombre']} {$user['apellido']} ({$user['rut']})</option>";
                            }
                            ?>
                        </select>
                    </div>
                    <div class="form-group">
                        <select name="accion" class="form-control" required>
                            <option value="bloquear">Bloquear</option>
                            <option value="desbloquear">Desbloquear</option>
                        </select>
                    </div>
                    <button type="submit" class="btn btn-warning">
                        <i class="material-icons">security</i> Aplicar
                    </button>
                </form>
            </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Sección de Documentos -->
    <div class="admin-section" id="tab-documents" style="display: none;">
        <div class="admin-actions">
            <?php if (tienePermiso('eliminar_documentos')): ?>
            <div class="action-card">
                <h4><i class="material-icons">delete_sweep</i> Eliminar Documentos</h4>
                <form method="POST" action="?action=eliminar_documento" class="action-form">
                    <div class="form-group">
                        <select name="tipo_documento" class="form-control" required>
                            <option value="">Seleccione tipo de documento</option>
                            <option value="rendicion">Rendición</option>
                            <option value="orden_compra">Orden de Compra</option>
                            <option value="devolucion">Devolución</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <input type="number" name="documento_id" class="form-control" 
                               placeholder="ID del documento" required>
                    </div>
                    <button type="submit" class="btn btn-danger" 
                            onclick="return confirm('¿Está seguro de eliminar este documento? Esta acción no se puede deshacer.')">
                        <i class="material-icons">delete_forever</i> Eliminar
                    </button>
                </form>
            </div>
            <?php endif; ?>

            <div class="action-card">
                <h4><i class="material-icons">find_in_page</i> Búsqueda Avanzada</h4>
                <form method="GET" action="buscar_documentos.php" class="action-form">
                    <div class="form-group">
                        <select name="tipo" class="form-control" required>
                            <option value="todos">Todos los documentos</option>
                            <option value="rendicion">Rendiciones</option>
                            <option value="orden_compra">Órdenes de Compra</option>
                            <option value="devolucion">Devoluciones</option>
                        </select>
                    </div>
                    <div class="form-row">
                        <div class="form-group col-md-6">
                            <input type="date" name="fecha_desde" class="form-control" placeholder="Fecha desde">
                        </div>
                        <div class="form-group col-md-6">
                            <input type="date" name="fecha_hasta" class="form-control" placeholder="Fecha hasta">
                        </div>
                    </div>
                    <button type="submit" class="btn btn-primary">
                        <i class="material-icons">search</i> Buscar
                    </button>
                </form>
            </div>

            <div class="action-card">
                <h4><i class="material-icons">assessment</i> Reportes</h4>
                <div class="action-buttons">
                    <a href="reportes.php?tipo=mensual" class="btn btn-info">
                        <i class="material-icons">bar_chart</i> Reporte Mensual
                    </a>
                    <a href="reportes.php?tipo=anual" class="btn btn-info">
                        <i class="material-icons">timeline</i> Reporte Anual
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- Sección de Sistema -->
    <div class="admin-section" id="tab-system" style="display: none;">
        <div class="admin-actions">
            <?php if (tienePermiso('gestionar_respaldos')): ?>
            <div class="action-card">
                <h4><i class="material-icons">backup</i> Respaldos del Sistema</h4>
                <div class="action-buttons">
                    <a href="?action=backup" class="btn btn-primary">
                        <i class="material-icons">save</i> Generar Respaldo
                    </a>
                    <a href="respaldos.php" class="btn btn-info">
                        <i class="material-icons">folder</i> Ver Respaldos
                    </a>
                </div>
                <div class="backup-schedule">
                    <h5>Próximo respaldo automático:</h5>
                    <p><?php echo date('d/m/Y H:i', strtotime('+1 day')); ?></p>
                </div>
            </div>
            <?php endif; ?>

            <?php if (tienePermiso('configuracion_sistema')): ?>
            <div class="action-card">
                <h4><i class="material-icons">tune</i> Configuraciones Avanzadas</h4>
                <div class="action-buttons">
                    <a href="configuraciones.php" class="btn btn-primary">
                        <i class="material-icons">settings</i> Configuraciones del Sistema
                    </a>
                    <a href="categorias.php" class="btn btn-info">
                        <i class="material-icons">category</i> Gestionar Categorías
                    </a>
                </div>
                <p class="mt-2">
                    <small>Personalice la apariencia, comportamiento y configuraciones generales del sistema.</small>
                </p>
            </div>
            <?php endif; ?>

            <?php if (tienePermiso('limpiar_cache')): ?>
            <div class="action-card">
                <h4><i class="material-icons">cleaning_services</i> Mantenimiento</h4>
                <div class="action-buttons">
                    <a href="?action=limpiar_cache" class="btn btn-warning" 
                       onclick="return confirm('¿Está seguro de limpiar el caché del sistema?')">
                        <i class="material-icons">delete_sweep</i> Limpiar Caché
                    </a>
                    <a href="?action=optimizar" class="btn btn-info">
                        <i class="material-icons">speed</i> Optimizar BD
                    </a>
                </div>
            </div>
            <?php endif; ?>

            <div class="action-card">
                <h4><i class="material-icons">info</i> Información del Sistema</h4>
                <div class="system-info">
                    <p><strong>Versión PHP:</strong> <?php echo $estadisticas['sistema']['version_php']; ?></p>
                    <p><strong>Versión MySQL:</strong> <?php echo $estadisticas['sistema']['version_mysql']; ?></p>
                    <p><strong>Espacio en disco:</strong> <?php echo formatBytes($estadisticas['sistema']['espacio_libre']); ?> libre de <?php echo formatBytes($estadisticas['sistema']['espacio_total']); ?></p>
                    <p><strong>Archivos totales:</strong> <?php echo number_format($estadisticas['sistema']['archivos_total']); ?></p>
                </div>
            </div>
        </div>
    </div>

    <!-- Sección de Logs -->
    <div class="admin-section" id="tab-logs" style="display: none;">
        <?php if (tienePermiso('ver_logs_sistema')): ?>
        <div class="logs-container">
            <div class="logs-header">
                <h4><i class="material-icons">receipt_long</i> Registro de Actividades</h4>
                <div class="logs-filter">
                    <select id="log-type-filter" class="form-control">
                        <option value="">Todos los tipos</option>
                        <option value="login">Inicios de sesión</option>
                        <option value="documento">Documentos</option>
                        <option value="sistema">Sistema</option>
                    </select>
                    <input type="date" id="log-date-filter" class="form-control">
                </div>
            </div>
            
            <div class="table-responsive">
                <table class="log-table">
                    <thead>
                        <tr>
                            <th>Fecha</th>
                            <th>Usuario</th>
                            <th>Acción</th>
                            <th>Descripción</th>
                            <th>IP</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php
                        $stmt = $conn->query("
                            SELECT h.*, u.nombre, u.apellido 
                            FROM historial_cambios h
                            LEFT JOIN users u ON h.user_id = u.id
                            ORDER BY h.fecha_hora DESC
                            LIMIT 50
                        ");
                        while ($log = $stmt->fetch()): ?>
                        <tr>
                            <td><?php echo date('d/m/Y H:i', strtotime($log['fecha_hora'])); ?></td>
                            <td><?php echo htmlspecialchars($log['nombre'] . ' ' . $log['apellido']); ?></td>
                            <td><?php echo htmlspecialchars($log['tipo']); ?></td>
                            <td><?php echo htmlspecialchars($log['descripcion']); ?></td>
                            <td><?php echo htmlspecialchars($log['ip'] ?? '-'); ?></td>
                        </tr>
                        <?php endwhile; ?>
                    </tbody>
                </table>
            </div>
            
            <div class="logs-footer">
                <a href="logs_export.php" class="btn btn-primary">
                    <i class="material-icons">download</i> Exportar Logs
                </a>
            </div>
        </div>
        <?php endif; ?>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Manejo de tabs
    const tabs = document.querySelectorAll('.admin-tab');
    const sections = document.querySelectorAll('.admin-section');
    
    tabs.forEach(tab => {
        tab.addEventListener('click', () => {
            const targetId = 'tab-' + tab.dataset.tab;
            
            // Actualizar tabs
            tabs.forEach(t => t.classList.remove('active'));
            tab.classList.add('active');
            
            // Mostrar sección correspondiente
            sections.forEach(section => {
                section.style.display = section.id === targetId ? 'block' : 'none';
            });
        });
    });
    
    // Filtros de logs
    const logTypeFilter = document.getElementById('log-type-filter');
    const logDateFilter = document.getElementById('log-date-filter');
    
    if (logTypeFilter && logDateFilter) {
        const filterLogs = () => {
            const type = logTypeFilter.value;
            const date = logDateFilter.value;
            const rows = document.querySelectorAll('.log-table tbody tr');
            
            rows.forEach(row => {
                let show = true;
                
                if (type && !row.querySelector('td:nth-child(3)').textContent.includes(type)) {
                    show = false;
                }
                
                if (date) {
                    const rowDate = row.querySelector('td:first-child').textContent.split(' ')[0];
                    if (rowDate !== date) {
                        show = false;
                    }
                }
                
                row.style.display = show ? '' : 'none';
            });
        };
        
        logTypeFilter.addEventListener('change', filterLogs);
        logDateFilter.addEventListener('change', filterLogs);
    }
});
</script>

<style>
/* Estilos adicionales */
.action-buttons {
    display: flex;
    gap: 10px;
    margin-top: 15px;
}

.system-info {
    background: #f8f9fa;
    padding: 15px;
    border-radius: 4px;
    margin-top: 10px;
}

.system-info p {
    margin: 5px 0;
}

.logs-container {
    background: white;
    border-radius: 8px;
    padding: 20px;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.logs-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
}

.logs-filter {
    display: flex;
    gap: 10px;
}

.logs-footer {
    margin-top: 20px;
    text-align: right;
}

.backup-schedule {
    margin-top: 15px;
    padding-top: 15px;
    border-top: 1px solid #eee;
}

.backup-schedule h5 {
    color: #666;
    margin-bottom: 5px;
}
</style>

<?php
/**
 * Formatea bytes a una unidad legible
 */
function formatBytes($bytes, $precision = 2) {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    
    $bytes /= pow(1024, $pow);
    
    return round($bytes, $precision) . ' ' . $units[$pow];
}

// Incluir el footer
require_once 'includes/footer.php';
?> 