<?php

/**
 * Verifica si el usuario tiene una sesión activa
 * Si no tiene sesión, redirecciona al login
 * 
 * @return void
 */
function verificarSesion() {
    // Verificar si ya hay una sesión activa antes de iniciarla
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    
    // Verificar si el usuario está autenticado
    if (!isset($_SESSION['user_id'])) {
        header('Location: login.php');
        exit();
    }

    // Cargar permisos del usuario si no están en sesión
    if (!isset($_SESSION['user_permisos'])) {
        try {
            $conn = getDatabaseConnection();
            $stmt = $conn->prepare("
                SELECT p.nombre 
                FROM user_permisos up 
                JOIN permisos p ON up.permiso_id = p.id 
                WHERE up.user_id = :user_id
            ");
            $stmt->bindParam(':user_id', $_SESSION['user_id']);
            $stmt->execute();
            $_SESSION['user_permisos'] = $stmt->fetchAll(PDO::FETCH_COLUMN);
        } catch (PDOException $e) {
            error_log('Error cargando permisos: ' . $e->getMessage());
            $_SESSION['user_permisos'] = [];
        }
    }

    // Verificar rol del usuario
    if (!isset($_SESSION['user_rol'])) {
        try {
            $conn = getDatabaseConnection();
            $stmt = $conn->prepare("SELECT rol FROM users WHERE id = :user_id");
            $stmt->bindParam(':user_id', $_SESSION['user_id']);
            $stmt->execute();
            $_SESSION['user_rol'] = $stmt->fetchColumn();
        } catch (PDOException $e) {
            error_log('Error cargando rol: ' . $e->getMessage());
            $_SESSION['user_rol'] = 'usuario';
        }
    }
}

/**
 * Verifica si el usuario tiene un permiso específico
 * 
 * @param string $permiso Nombre del permiso a verificar
 * @return bool True si tiene el permiso, false en caso contrario
 */
function tienePermiso($permiso) {
    // El super_admin tiene todos los permisos
    if (isset($_SESSION['user_rol']) && $_SESSION['user_rol'] === 'super_admin') {
        return true;
    }
    
    // Verificar permiso específico
    if (isset($_SESSION['user_permisos']) && in_array($permiso, $_SESSION['user_permisos'])) {
        return true;
    }
    
    return false;
}

/**
 * Verifica los permisos necesarios para acceder a un módulo o realizar una acción
 * Si no tiene los permisos, redirige según corresponda
 * 
 * @param string|array $permisos Permiso o array de permisos requeridos
 * @param bool $redirigir Si es true, redirige al usuario. Si es false, solo retorna el resultado
 * @return bool True si tiene los permisos necesarios, false en caso contrario
 */
function verificarPermisosModulo($permisos, $redirigir = true) {
    // Verificar sesión primero
    if (!isset($_SESSION['user_id'])) {
        if ($redirigir) {
            redirigirConMensaje('login.php', 'Debe iniciar sesión para acceder a esta sección', 'warning');
        }
        return false;
    }
    
    // Convertir permiso único a array
    if (!is_array($permisos)) {
        $permisos = [$permisos];
    }
    
    // Super admin tiene acceso a todo
    if (isset($_SESSION['user_rol']) && $_SESSION['user_rol'] === 'super_admin') {
        return true;
    }
    
    // Verificar cada permiso requerido
    $tiene_permiso = false;
    foreach ($permisos as $permiso) {
        if (tienePermiso($permiso)) {
            $tiene_permiso = true;
            break;
        }
    }
    
    if (!$tiene_permiso && $redirigir) {
        // Si está logueado pero no tiene permisos, redirigir al dashboard
        redirigirConMensaje('dashboard.php', 'No tiene permisos para acceder a esta sección', 'danger');
    }
    
    return $tiene_permiso;
}

/**
 * Verifica los permisos para realizar acciones específicas (CRUD)
 * 
 * @param string $modulo Nombre del módulo (ej: 'rendiciones', 'ordenes_compra', 'devoluciones')
 * @param string $accion Acción a realizar ('crear', 'ver', 'editar', 'eliminar', 'aprobar')
 * @param bool $redirigir Si es true, redirige al usuario. Si es false, solo retorna el resultado
 * @return bool True si tiene los permisos necesarios, false en caso contrario
 */
function verificarPermisosAccion($modulo, $accion, $redirigir = true) {
    $permiso = '';
    
    switch ($accion) {
        case 'crear':
            $permiso = "crear_{$modulo}";
            break;
        case 'ver':
            $permiso = "ver_{$modulo}";
            break;
        case 'editar':
            $permiso = "editar_{$modulo}";
            break;
        case 'eliminar':
            $permiso = "eliminar_{$modulo}";
            break;
        case 'aprobar':
            $permiso = "aprobar_{$modulo}";
            break;
        default:
            if ($redirigir) {
                redirigirConMensaje('dashboard.php', 'Acción no válida', 'danger');
            }
            return false;
    }
    
    return verificarPermisosModulo($permiso, $redirigir);
}

/**
 * Verifica si el usuario puede acceder a una URL específica basado en sus permisos
 * 
 * @param string $url URL a verificar
 * @return bool True si tiene acceso, false en caso contrario
 */
function puedeAccederURL($url) {
    // Mapeo de URLs a permisos requeridos
    $permisos_url = [
        'rendiciones.php' => ['ver_rendiciones'],
        'ordenes_compra.php' => ['ver_ordenes_compra'],
        'usuarios.php' => ['gestionar_usuarios'],
        'maquinas.php' => ['ver_maquinas'],
        'repuestos.php' => ['ver_repuestos']
    ];
    
    // Obtener el nombre del archivo de la URL
    $archivo = basename(parse_url($url, PHP_URL_PATH));
    
    // Si la URL no requiere permisos específicos, permitir acceso
    if (!isset($permisos_url[$archivo])) {
        return true;
    }
    
    // Verificar los permisos requeridos
    return verificarPermisosModulo($permisos_url[$archivo], false);
}

/**
 * Registra una actividad en el historial del sistema
 * Versión mejorada para garantizar codificación UTF-8 y depuración
 * 
 * @param PDO $conn Conexión a la base de datos
 * @param int $user_id ID del usuario
 * @param string $tipo Tipo de actividad (login, crear, editar, eliminar, etc.)
 * @param string $descripcion Descripción de la actividad
 * @param string|null $documento_id ID del documento relacionado (opcional)
 * @param string|null $documento_tipo Tipo de documento relacionado (opcional)
 * @return bool True si se registró correctamente, false en caso contrario
 */
function registrarActividad($conn, $user_id, $tipo, $descripcion, $documento_id = null, $documento_tipo = null) {
    try {
        // Paso 1: Sanitizar y normalizar la descripción
        $descripcion = trim($descripcion);
        
        // Paso 2: Depuración de codificación inicial
        if (defined('DEBUG_MODE') && DEBUG_MODE) {
            $encoding = mb_detect_encoding($descripcion, 'UTF-8, ISO-8859-1, WINDOWS-1252', true);
            error_log("Codificación inicial de descripción: $encoding, texto: $descripcion");
        }
        
        // Paso 3: Forzar a UTF-8 si la cadena no está ya en UTF-8
        if (!mb_check_encoding($descripcion, 'UTF-8')) {
            $descripcion = mb_convert_encoding($descripcion, 'UTF-8', 'auto');
            if (defined('DEBUG_MODE') && DEBUG_MODE) {
                error_log("Descripción convertida a UTF-8: $descripcion");
            }
        }
        
        // Paso 4: Manejar casos específicos según el tipo
        if ($tipo == 'login') {
            $descripcion = 'Acceso a la cuenta';
        } elseif ($tipo == 'logout') {
            $descripcion = 'Salir de la cuenta';
        }
        
        // Paso 5: Obtener IP
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'Desconocida';
        
        // Paso 6: Preparar y ejecutar la consulta
        $stmt = $conn->prepare("
            INSERT INTO historial_cambios 
            (user_id, tipo, descripcion, documento_id, documento_tipo, fecha_hora, ip_address) 
            VALUES (:user_id, :tipo, :descripcion, :documento_id, :documento_tipo, NOW(), :ip_address)
        ");
        
        $stmt->bindValue(':user_id', $user_id, PDO::PARAM_INT);
        $stmt->bindValue(':tipo', $tipo, PDO::PARAM_STR);
        $stmt->bindValue(':descripcion', $descripcion, PDO::PARAM_STR);
        $stmt->bindValue(':documento_id', $documento_id, PDO::PARAM_INT);
        $stmt->bindValue(':documento_tipo', $documento_tipo, PDO::PARAM_STR);
        $stmt->bindValue(':ip_address', $ip, PDO::PARAM_STR);
        
        $stmt->execute();
        
        if (defined('DEBUG_MODE') && DEBUG_MODE) {
            error_log("Actividad registrada: user_id=$user_id, tipo=$tipo, descripcion=$descripcion");
        }
        
        return true;
    } catch (PDOException $e) {
        if (defined('DEBUG_MODE') && DEBUG_MODE) {
            error_log('Error al registrar actividad: ' . $e->getMessage());
        }
        return false;
    }
}

/**
 * Formatea un número como moneda en pesos chilenos
 * 
 * @param float $monto Monto a formatear
 * @return string Monto formateado
 */
function formatoMoneda($monto) {
    return '$' . number_format($monto, 0, ',', '.');
}

/**
 * Formatea una fecha en formato legible
 * 
 * @param string $fecha Fecha en formato Y-m-d
 * @return string Fecha formateada como d/m/Y
 */
function formatoFecha($fecha) {
    if (empty($fecha)) return '';
    $date = new DateTime($fecha);
    return $date->format('d/m/Y');
}

/**
 * Genera un código único para documentos
 * 
 * @param string $tipo Tipo de documento (REN, NV, etc.)
 * @param int $id ID del documento
 * @return string Código formateado (ej: REN-2025-001)
 */
function generarCodigo($tipo, $id) {
    $year = date('Y');
    $id_formatted = str_pad($id, 3, '0', STR_PAD_LEFT);
    return "{$tipo}-{$year}-{$id_formatted}";
}

/**
 * Sanitiza un string para evitar inyecciones HTML
 * 
 * @param string $string String a sanitizar
 * @return string String sanitizado
 */
function sanitizarHTML($string) {
    return htmlspecialchars($string, ENT_QUOTES, 'UTF-8');
}

/**
 * Verifica si un RUT chileno es válido
 * 
 * @param string $rut RUT a validar (con o sin puntos y guión)
 * @return bool True si es válido, false en caso contrario
 */
function validarRut($rut) {
    // Eliminar puntos y guión
    $rut = str_replace(['.', '-'], '', $rut);
    
    // Verificar largo mínimo
    if (strlen($rut) < 2) return false;
    
    // Separar cuerpo y dígito verificador
    $dv = substr($rut, -1);
    $cuerpo = substr($rut, 0, -1);
    
    // Calcular dígito verificador
    $suma = 0;
    $multiplo = 2;
    
    for ($i = strlen($cuerpo) - 1; $i >= 0; $i--) {
        $suma += $cuerpo[$i] * $multiplo;
        $multiplo = $multiplo < 7 ? $multiplo + 1 : 2;
    }
    
    $dvEsperado = 11 - ($suma % 11);
    
    if ($dvEsperado == 11) {
        $dvEsperado = '0';
    } elseif ($dvEsperado == 10) {
        $dvEsperado = 'K';
    } else {
        $dvEsperado = (string)$dvEsperado;
    }
    
    return strtoupper($dv) == strtoupper($dvEsperado);
}

/**
 * Formatea un RUT chileno (XX.XXX.XXX-X)
 * 
 * @param string $rut RUT a formatear
 * @return string RUT formateado
 */
function formatearRut($rut) {
    $rut = str_replace(['.', '-'], '', $rut);
    $dv = substr($rut, -1);
    $cuerpo = substr($rut, 0, -1);
    $rutFormateado = number_format($cuerpo, 0, '', '.') . '-' . $dv;
    return $rutFormateado;
}

/**
 * Envía una notificación por correo electrónico
 * 
 * @param string $destinatario Email del destinatario
 * @param string $asunto Asunto del correo
 * @param string $mensaje Contenido HTML del correo
 * @return bool True si se envió correctamente, false en caso contrario
 */
function enviarCorreo($destinatario, $asunto, $mensaje) {
    // Implementación básica usando mail() de PHP
    // En producción es recomendable usar PHPMailer u otra biblioteca
    
    $cabeceras = "MIME-Version: 1.0" . "\r\n";
    $cabeceras .= "Content-type:text/html;charset=UTF-8" . "\r\n";
    $cabeceras .= "From: " . MAIL_FROM_NAME . " <" . MAIL_USERNAME . ">" . "\r\n";
    
    return mail($destinatario, $asunto, $mensaje, $cabeceras);
}

/**
 * Genera un nombre de archivo único para subidas
 * 
 * @param string $originalName Nombre original del archivo
 * @return string Nombre único generado
 */
function generarNombreArchivo($originalName) {
    $extension = pathinfo($originalName, PATHINFO_EXTENSION);
    $timestamp = time();
    $randomStr = bin2hex(random_bytes(8));
    return $timestamp . '_' . $randomStr . '.' . $extension;
}

/**
 * Verifica si un archivo es del tipo permitido
 * 
 * @param string $mimetype Tipo MIME del archivo
 * @return bool True si es permitido, false en caso contrario
 */
function esArchivoPermitido($mimetype) {
    global $allowed_mimetypes;
    return in_array($mimetype, $allowed_mimetypes);
}

/**
 * Obtiene el nombre de un estado por su ID
 * 
 * @param PDO $conn Conexión a la base de datos
 * @param int $estado_id ID del estado
 * @return string Nombre del estado
 */
function obtenerNombreEstado($conn, $estado_id) {
    try {
        $stmt = $conn->prepare("SELECT nombre FROM estados WHERE id = :id");
        $stmt->bindParam(':id', $estado_id);
        $stmt->execute();
        
        $estado = $stmt->fetch(PDO::FETCH_ASSOC);
        $nombre = $estado ? $estado['nombre'] : 'Desconocido';
        
        // Asegurar UTF-8
        if (!mb_check_encoding($nombre, 'UTF-8')) {
            $nombre = mb_convert_encoding($nombre, 'UTF-8', 'auto');
        }
        
        return $nombre;
    } catch (PDOException $e) {
        if (defined('DEBUG_MODE') && DEBUG_MODE) {
            error_log('Error al obtener nombre del estado: ' . $e->getMessage());
        }
        return 'Error';
    }
}

/**
 * Obtiene todos los estados disponibles
 * 
 * @param PDO $conn Conexión a la base de datos
 * @return array Array de estados
 */
function obtenerEstados($conn) {
    try {
        $stmt = $conn->prepare("SELECT id, nombre, descripcion, color FROM estados");
        $stmt->execute();
        $estados = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Asegurar UTF-8 en los datos devueltos
        foreach ($estados as &$estado) {
            foreach ($estado as $key => &$value) {
                if (is_string($value) && !mb_check_encoding($value, 'UTF-8')) {
                    $value = mb_convert_encoding($value, 'UTF-8', 'auto');
                }
            }
        }
        
        return $estados;
    } catch (PDOException $e) {
        if (defined('DEBUG_MODE') && DEBUG_MODE) {
            error_log('Error al obtener estados: ' . $e->getMessage());
        }
        return [];
    }
}

/**
 * Obtiene todas las categorías de gastos activas
 * 
 * @param PDO $conn Conexión a la base de datos
 * @return array Array de categorías
 */
function obtenerCategorias($conn) {
    try {
        $stmt = $conn->prepare("SELECT id, nombre FROM categorias_gastos WHERE activo = 1");
        $stmt->execute();
        $categorias = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Asegurar UTF-8 en los datos devueltos
        foreach ($categorias as &$categoria) {
            if (is_string($categoria['nombre']) && !mb_check_encoding($categoria['nombre'], 'UTF-8')) {
                $categoria['nombre'] = mb_convert_encoding($categoria['nombre'], 'UTF-8', 'auto');
            }
        }
        
        return $categorias;
    } catch (PDOException $e) {
        if (defined('DEBUG_MODE') && DEBUG_MODE) {
            error_log('Error al obtener categorías: ' . $e->getMessage());
        }
        return [];
    }
}

/**
 * Obtiene el nombre completo de un usuario por su ID
 * 
 * @param PDO $conn Conexión a la base de datos
 * @param int $user_id ID del usuario
 * @return string Nombre completo del usuario
 */
function obtenerNombreUsuario($conn, $user_id) {
    try {
        $stmt = $conn->prepare("SELECT nombre, apellido FROM users WHERE id = :id");
        $stmt->bindParam(':id', $user_id);
        $stmt->execute();
        
        $usuario = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($usuario) {
            $nombre = $usuario['nombre'] . ' ' . $usuario['apellido'];
            // Asegurar UTF-8
            if (!mb_check_encoding($nombre, 'UTF-8')) {
                $nombre = mb_convert_encoding($nombre, 'UTF-8', 'auto');
            }
            return $nombre;
        }
        return 'Usuario Desconocido';
    } catch (PDOException $e) {
        if (defined('DEBUG_MODE') && DEBUG_MODE) {
            error_log('Error al obtener nombre de usuario: ' . $e->getMessage());
        }
        return 'Error';
    }
}

/**
 * Determina si una rendición está en tiempo para ser editada (24 horas)
 * 
 * @param string $fecha_creacion Fecha de creación de la rendición
 * @return bool True si está en tiempo, false en caso contrario
 */
function estaEnTiempoEdicion($fecha_creacion) {
    // Permitir edición sin restricción de tiempo para debugging
    if (defined('DEBUG_MODE') && DEBUG_MODE) {
        return true;
    }
    
    $fecha = new DateTime($fecha_creacion);
    $ahora = new DateTime();
    $diferencia = $ahora->diff($fecha);
    
    // Convertir a horas
    $horas = $diferencia->h + ($diferencia->days * 24);
    
    return $horas <= 24;
}

/**
 * Redirecciona con un mensaje flash
 * 
 * @param string $url URL de destino
 * @param string $mensaje Mensaje a mostrar
 * @param string $tipo Tipo de mensaje (success, danger, warning, info)
 * @return void
 */
function redirigirConMensaje($url, $mensaje, $tipo = 'info') {
    $_SESSION['flash_message'] = $mensaje;
    $_SESSION['flash_type'] = $tipo;
    
    // Si estamos en modo debug y hay un error, enviar al log
    if (defined('DEBUG_MODE') && DEBUG_MODE && $tipo === 'danger') {
        error_log("Redirigiendo a: $url con mensaje de error: $mensaje");
    }
    
    // Usar location absoluta
    if (strpos($url, 'http') !== 0 && strpos($url, '/') !== 0) {
        // Si es una URL relativa, construir URL completa
        $url = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . 
               "://" . $_SERVER['HTTP_HOST'] . 
               dirname($_SERVER['PHP_SELF']) . "/" . $url;
    }
    
    header("Location: $url");
    exit();
}

/**
 * Muestra un mensaje flash si existe
 * 
 * @return string|null Código HTML del mensaje o null si no hay mensaje
 */
function mostrarMensajeFlash() {
    if (isset($_SESSION['flash_message'])) {
        $mensaje = $_SESSION['flash_message'];
        $tipo = $_SESSION['flash_type'] ?? 'info';
        
        unset($_SESSION['flash_message']);
        unset($_SESSION['flash_type']);
        
        $alertClass = 'alert-info';
        if ($tipo === 'success') $alertClass = 'alert-success';
        if ($tipo === 'danger') $alertClass = 'alert-danger';
        if ($tipo === 'warning') $alertClass = 'alert-warning';
        
        return '<div class="alert ' . $alertClass . '">' . $mensaje . '</div>';
    }
    
    return null;
}

/**
 * Envía un correo electrónico con los documentos adjuntos
 * 
 * @param string $destinatario Email del destinatario
 * @param string $asunto Asunto del correo
 * @param string $mensaje Contenido HTML del correo
 * @param array $archivos_adjuntos Array de rutas de archivos a adjuntar
 * @return bool True si se envió correctamente, false en caso contrario
 */
function enviarCorreoConAdjuntos($destinatario, $asunto, $mensaje, $archivos_adjuntos = []) {
    if (!defined('MAIL_USERNAME') || empty(MAIL_USERNAME)) {
        return false; // No está configurado el envío de correos
    }
    
    // Generar un boundary único para separar partes del email
    $boundary = md5(time());
    
    // Cabeceras
    $cabeceras = "MIME-Version: 1.0\r\n";
    $cabeceras .= "From: " . MAIL_FROM_NAME . " <" . MAIL_USERNAME . ">\r\n";
    $cabeceras .= "Content-Type: multipart/mixed; boundary=\"" . $boundary . "\"\r\n";
    
    // Mensaje
    $mensaje_email = "--" . $boundary . "\r\n";
    $mensaje_email .= "Content-Type: text/html; charset=utf-8\r\n";
    $mensaje_email .= "Content-Transfer-Encoding: 8bit\r\n\r\n";
    $mensaje_email .= $mensaje . "\r\n\r\n";
    
    // Agregar archivos adjuntos
    if (!empty($archivos_adjuntos)) {
        foreach ($archivos_adjuntos as $archivo) {
            if (file_exists($archivo)) {
                $contenido = file_get_contents($archivo);
                $contenido = chunk_split(base64_encode($contenido));
                $nombre_archivo = basename($archivo);
                $tipo_mime = mime_content_type($archivo);
                
                $mensaje_email .= "--" . $boundary . "\r\n";
                $mensaje_email .= "Content-Type: " . $tipo_mime . "; name=\"" . $nombre_archivo . "\"\r\n";
                $mensaje_email .= "Content-Transfer-Encoding: base64\r\n";
                $mensaje_email .= "Content-Disposition: attachment; filename=\"" . $nombre_archivo . "\"\r\n\r\n";
                $mensaje_email .= $contenido . "\r\n\r\n";
            }
        }
    }
    
    $mensaje_email .= "--" . $boundary . "--";
    
    // Enviar el correo
    if (defined('DEBUG_MODE') && DEBUG_MODE) {
        // En modo depuración, guardar el correo en un archivo de log
        file_put_contents(
            'email_log.txt', 
            date('Y-m-d H:i:s') . " - Para: $destinatario, Asunto: $asunto\n" . $mensaje_email . "\n\n", 
            FILE_APPEND
        );
        return true;
    } else {
        // Envío real del correo
        return mail($destinatario, $asunto, $mensaje_email, $cabeceras);
    }
}

/**
 * Genera un número de seguimiento único para solicitudes de repuestos
 * @param PDO $conn Conexión a la base de datos
 * @return string Número de seguimiento único
 */
function generarNumeroSeguimiento($conn) {
    $prefix = 'SR-' . date('Ym') . '-';
    
    // Obtener el último número de seguimiento del mes actual
    $stmt = $conn->prepare("
        SELECT numero_seguimiento 
        FROM solicitud_repuestos 
        WHERE numero_seguimiento LIKE :prefix 
        ORDER BY id DESC 
        LIMIT 1
    ");
    $prefix_search = $prefix . '%';
    $stmt->bindParam(':prefix', $prefix_search);
    $stmt->execute();
    
    $ultimo = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($ultimo && $ultimo['numero_seguimiento']) {
        // Extraer el número secuencial y aumentarlo
        $partes = explode('-', $ultimo['numero_seguimiento']);
        $secuencial = intval(end($partes)) + 1;
    } else {
        $secuencial = 1;
    }
    
    // Formatear el número secuencial con ceros a la izquierda
    return $prefix . str_pad($secuencial, 4, '0', STR_PAD_LEFT);
}

/**
 * Notifica a los administradores sobre una nueva solicitud de repuestos
 * @param PDO $conn Conexión a la base de datos
 * @param int $solicitud_id ID de la solicitud
 */
function notificarNuevaSolicitud($conn, $solicitud_id) {
    // Obtener detalles de la solicitud
    $stmt = $conn->prepare("
        SELECT sr.*, 
               u.nombre as usuario_nombre, u.apellido as usuario_apellido,
               m.codigo as maquina_codigo, m.nombre as maquina_nombre,
               sm.codigo_serie
        FROM solicitud_repuestos sr
        JOIN users u ON sr.user_id = u.id
        JOIN maquinas m ON sr.maquina_id = m.id
        LEFT JOIN series_maquinas sm ON sr.serie_id = sm.id
        WHERE sr.id = :id
    ");
    $stmt->bindParam(':id', $solicitud_id);
    $stmt->execute();
    $solicitud = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$solicitud) return;
    
    // Obtener administradores
    $stmt = $conn->prepare("
        SELECT email 
        FROM users 
        WHERE (rol = 'admin' OR rol = 'super_admin')
        AND activo = 1
    ");
    $stmt->execute();
    $admins = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    if (empty($admins)) return;
    
    // Construir mensaje
    $mensaje = "Nueva solicitud de repuestos:\n\n";
    $mensaje .= "Número de seguimiento: " . $solicitud['numero_seguimiento'] . "\n";
    $mensaje .= "Solicitante: " . $solicitud['usuario_nombre'] . " " . $solicitud['usuario_apellido'] . "\n";
    $mensaje .= "Máquina: " . $solicitud['maquina_codigo'] . " - " . $solicitud['maquina_nombre'] . "\n";
    if ($solicitud['codigo_serie']) {
        $mensaje .= "Serie: " . $solicitud['codigo_serie'] . "\n";
    }
    $mensaje .= "Fecha: " . date('d/m/Y', strtotime($solicitud['fecha'])) . "\n";
    $mensaje .= "Área: " . ucfirst(str_replace('_', ' ', $solicitud['area_origen'])) . "\n";
    $mensaje .= "Motivo: " . ucfirst(str_replace('_', ' ', $solicitud['motivo_solicitud'])) . "\n\n";
    $mensaje .= "Por favor, revise la solicitud en el sistema.";
    
    // Enviar notificación por email
    $asunto = "Nueva solicitud de repuestos - " . $solicitud['numero_seguimiento'];
    
    foreach ($admins as $email) {
        @mail($email, $asunto, $mensaje);
    }
}

/**
 * Agrupa los permisos por módulo
 * 
 * @param array $permisos Array de permisos
 * @return array Permisos agrupados por módulo
 */
function agruparPermisosPorModulo($permisos) {
    $grupos = [
        'rendiciones' => [],
        'ordenes_compra' => [],
        'maquinas' => [],
        'repuestos' => [],
        'usuarios' => [],
        'devoluciones' => [], // Agregamos el módulo de devoluciones
        'general' => []
    ];
    
    foreach ($permisos as $permiso) {
        if (strpos($permiso['nombre'], 'rendicion') !== false) {
            $grupos['rendiciones'][] = $permiso;
        } elseif (strpos($permiso['nombre'], 'orden_compra') !== false) {
            $grupos['ordenes_compra'][] = $permiso;
        } elseif (strpos($permiso['nombre'], 'maquina') !== false) {
            $grupos['maquinas'][] = $permiso;
        } elseif (strpos($permiso['nombre'], 'repuesto') !== false) {
            $grupos['repuestos'][] = $permiso;
        } elseif (strpos($permiso['nombre'], 'usuario') !== false) {
            $grupos['usuarios'][] = $permiso;
        } elseif (strpos($permiso['nombre'], 'devolucion') !== false) { // Agregamos la condición para devoluciones
            $grupos['devoluciones'][] = $permiso;
        } else {
            $grupos['general'][] = $permiso;
        }
    }
    
    return array_filter($grupos); // Eliminar grupos vacíos
}

/**
 * Envía una notificación por correo cuando se crea o modifica una rendición
 * @param PDO $conn Conexión a la base de datos
 * @param int $rendicion_id ID de la rendición
 * @param string $tipo_accion 'crear' o 'editar'
 * @return bool True si se envió correctamente, false en caso contrario
 */
function notificarRendicion($conn, $rendicion_id, $tipo_accion = 'crear') {
    try {
        // Obtener detalles de la rendición
        $stmt = $conn->prepare("
            SELECT r.*, 
                   u.nombre as usuario_nombre, u.apellido as usuario_apellido, u.email as usuario_email,
                   t.nombre as tercero_nombre, t.apellido as tercero_apellido,
                   e.nombre as estado_nombre
            FROM rendiciones r
            JOIN users u ON r.user_id = u.id
            LEFT JOIN users t ON r.tercero_id = t.id
            LEFT JOIN estados e ON r.estado_id = e.id
            WHERE r.id = :id
        ");
        $stmt->bindParam(':id', $rendicion_id);
        $stmt->execute();
        $rendicion = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$rendicion) return false;
        
        // Generar el código de la rendición
        $codigo = generarCodigo('REN', $rendicion_id);
        
        // Construir el asunto del correo
        $accion = ($tipo_accion == 'crear') ? 'Nueva' : 'Actualización de';
        $asunto = "$accion rendición - $codigo";
        
        // Construir el mensaje HTML
        $mensaje = "<html><body>";
        $mensaje .= "<h2>$accion Rendición de Gastos</h2>";
        $mensaje .= "<p><strong>Código:</strong> $codigo</p>";
        $mensaje .= "<p><strong>Fecha:</strong> " . date('d/m/Y', strtotime($rendicion['fecha'])) . "</p>";
        $mensaje .= "<p><strong>Estado:</strong> " . $rendicion['estado_nombre'] . "</p>";
        $mensaje .= "<p><strong>Monto Total:</strong> " . formatoMoneda($rendicion['monto_total']) . "</p>";
        $mensaje .= "<p><strong>Fondo a Rendir:</strong> " . formatoMoneda($rendicion['fondo_a_rendir']) . "</p>";
        
        if (!empty($rendicion['tercero_nombre'])) {
            $mensaje .= "<p><strong>Rendido por:</strong> " . $rendicion['tercero_nombre'] . " " . $rendicion['tercero_apellido'] . "</p>";
            $mensaje .= "<p><strong>Ingresado por:</strong> " . $rendicion['usuario_nombre'] . " " . $rendicion['usuario_apellido'] . "</p>";
        } else {
            $mensaje .= "<p><strong>Rendido por:</strong> " . $rendicion['usuario_nombre'] . " " . $rendicion['usuario_apellido'] . "</p>";
        }
        
        $mensaje .= "<p><strong>Descripción:</strong> " . nl2br($rendicion['descripcion']) . "</p>";
        
        // Obtener y agregar el detalle de la rendición
        $stmt = $conn->prepare("
            SELECT rc.*, cg.nombre as categoria_nombre 
            FROM rendicion_categoria rc
            JOIN categorias_gastos cg ON rc.categoria_id = cg.id
            WHERE rc.rendicion_id = :id
        ");
        $stmt->bindParam(':id', $rendicion_id);
        $stmt->execute();
        $detalles = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (!empty($detalles)) {
            $mensaje .= "<h3>Detalle de gastos:</h3>";
            $mensaje .= "<table border='1' cellpadding='5' cellspacing='0' style='border-collapse: collapse;'>";
            $mensaje .= "<tr><th>Categoría</th><th>Monto</th><th>Detalle</th><th>Tipo Doc.</th></tr>";
            
            foreach ($detalles as $detalle) {
                $mensaje .= "<tr>";
                $mensaje .= "<td>" . $detalle['categoria_nombre'] . "</td>";
                $mensaje .= "<td align='right'>" . formatoMoneda($detalle['monto']) . "</td>";
                $mensaje .= "<td>" . ($detalle['detalle'] ?? '-') . "</td>";
                $mensaje .= "<td>" . ucfirst($detalle['tipo_documento']) . "</td>";
                $mensaje .= "</tr>";
            }
            
            $mensaje .= "</table>";
        }
        
        $mensaje .= "</body></html>";
        
        // Generar el PDF
        $pdf_path = 'temp/' . $codigo . '.pdf';
        $url = "generate_pdf.php?tipo=rendicion&id=$rendicion_id&save=true";
        file_get_contents($url);
        
        // Obtener destinatarios
        $destinatarios = [];
        
        // Agregar el usuario que creó la rendición
        if (!empty($rendicion['usuario_email'])) {
            $destinatarios[] = $rendicion['usuario_email'];
        }
        
        // Agregar administradores
        $stmt = $conn->prepare("
            SELECT email 
            FROM users 
            WHERE (rol = 'admin' OR rol = 'super_admin')
            AND activo = 1
            AND email IS NOT NULL
            AND email != ''
        ");
        $stmt->execute();
        $admins = $stmt->fetchAll(PDO::FETCH_COLUMN);
        $destinatarios = array_merge($destinatarios, $admins);
        
        // Enviar el correo a cada destinatario
        foreach (array_unique($destinatarios) as $email) {
            enviarCorreoConAdjuntos($email, $asunto, $mensaje, [$pdf_path]);
        }
        
        // Limpiar el archivo temporal
        if (file_exists($pdf_path)) {
            unlink($pdf_path);
        }
        
        return true;
    } catch (Exception $e) {
        error_log('Error al enviar notificación de rendición: ' . $e->getMessage());
        return false;
    }
}

/**
 * Verifica si el usuario tiene acceso a un módulo específico
 * @param string $modulo Nombre del módulo
 * @return bool True si tiene acceso, false en caso contrario
 */
function tieneAccesoModulo($modulo) {
    global $conn;
    if (!isset($_SESSION['user_id'])) return false;
    
    $user_id = $_SESSION['user_id'];
    $user_rol = $_SESSION['user_rol'] ?? '';
    
    // Super admin tiene acceso a todo
    if ($user_rol === 'super_admin') return true;
    
    // Mapeo de permisos necesarios por módulo
    $permisos_modulo = [
        'rendiciones' => ['ver_rendiciones', 'crear_rendicion', 'editar_rendicion', 'aprobar_rendiciones'],
        'ordenes_compra' => ['ver_ordenes_compra', 'crear_orden_compra', 'editar_orden_compra', 'aprobar_ordenes_compra'],
        'maquinas' => ['ver_maquinas', 'crear_maquina', 'editar_maquina'],
        'repuestos' => ['ver_repuestos', 'crear_repuesto', 'editar_repuesto', 'ver_solicitudes_repuesto'],
        'devoluciones' => ['ver_devoluciones', 'crear_devolucion', 'editar_devolucion', 'aprobar_devoluciones'],
        'usuarios' => ['gestionar_usuarios'],
        'reportes' => ['ver_reportes'],
        'sistema' => ['acceso_admin_sistema', 'limpiar_cache', 'ver_logs_sistema', 'gestionar_respaldos']
    ];
    
    // Si el módulo no está definido, denegar acceso
    if (!isset($permisos_modulo[$modulo])) return false;
    
    // Verificar si tiene al menos uno de los permisos del módulo
    $permisos = $permisos_modulo[$modulo];
    $permisos_str = "'" . implode("','", $permisos) . "'";
    
    $stmt = $conn->prepare("
        SELECT COUNT(*) 
        FROM user_permisos up 
        JOIN permisos p ON up.permiso_id = p.id 
        WHERE up.user_id = :user_id 
        AND p.nombre IN ($permisos_str)
    ");
    
    $stmt->bindParam(':user_id', $user_id);
    $stmt->execute();
    
    return $stmt->fetchColumn() > 0;
}

/**
 * Protege una página verificando los permisos necesarios
 * @param string $modulo Nombre del módulo
 * @param string $accion Acción específica (opcional)
 */
function protegerModulo($modulo, $accion = '') {
    if (!tieneAccesoModulo($modulo)) {
        $_SESSION['flash_message'] = "No tiene permisos para acceder a este módulo";
        $_SESSION['flash_type'] = 'danger';
        header('Location: dashboard.php');
        exit;
    }
    
    if ($accion && !tienePermiso($accion)) {
        $_SESSION['flash_message'] = "No tiene permisos para realizar esta acción";
        $_SESSION['flash_type'] = 'danger';
        header('Location: ' . $modulo . '.php');
        exit;
    }
}

/**
 * Obtiene los módulos a los que el usuario tiene acceso
 * @return array Array de módulos accesibles
 */
function obtenerModulosAccesibles() {
    $modulos = [
        'documentos' => [
            'nombre' => 'Documentos',
            'icon' => 'description',
            'submodulos' => [
                'rendiciones' => [
                    'nombre' => 'Rendiciones',
                    'url' => 'rendiciones.php',
                    'icon' => 'receipt'
                ],
                'ordenes_compra' => [
                    'nombre' => 'Órdenes de Compra',
                    'url' => 'ordenes_compra.php',
                    'icon' => 'shopping_cart'
                ],
                'devoluciones' => [
                    'nombre' => 'Devoluciones',
                    'url' => 'devoluciones.php',
                    'icon' => 'assignment_return'
                ]
            ]
        ],
        'inventario' => [
            'nombre' => 'Inventario',
            'icon' => 'inventory',
            'submodulos' => [
                'maquinas' => [
                    'nombre' => 'Máquinas',
                    'url' => 'maquinas.php',
                    'icon' => 'precision_manufacturing'
                ],
                'repuestos' => [
                    'nombre' => 'Repuestos',
                    'url' => 'repuestos.php',
                    'icon' => 'build'
                ]
            ]
        ],
        'administracion' => [
            'nombre' => 'Administración',
            'icon' => 'admin_panel_settings',
            'submodulos' => [
                'usuarios' => [
                    'nombre' => 'Usuarios',
                    'url' => 'usuarios.php',
                    'icon' => 'people'
                ],
                'reportes' => [
                    'nombre' => 'Reportes',
                    'url' => 'reportes.php',
                    'icon' => 'assessment'
                ],
                'sistema' => [
                    'nombre' => 'Sistema',
                    'url' => 'admin_sistema.php',
                    'icon' => 'settings'
                ]
            ]
        ]
    ];
    
    // Filtrar módulos según permisos
    foreach ($modulos as $key => &$modulo) {
        foreach ($modulo['submodulos'] as $subkey => $submodulo) {
            if (!tieneAccesoModulo($subkey)) {
                unset($modulo['submodulos'][$subkey]);
            }
        }
        // Si no hay submodulos accesibles, eliminar el módulo principal
        if (empty($modulo['submodulos'])) {
            unset($modulos[$key]);
        }
    }
    
    return $modulos;
}

/**
 * Genera el menú de navegación según los permisos del usuario
 * @return string HTML del menú
 */
function generarMenu() {
    $modulos = obtenerModulosAccesibles();
    $html = '';
    
    foreach ($modulos as $modulo) {
        $html .= '<div class="menu-section">';
        $html .= '<div class="menu-header">';
        $html .= '<i class="material-icons">' . $modulo['icon'] . '</i>';
        $html .= '<span>' . $modulo['nombre'] . '</span>';
        $html .= '</div>';
        
        $html .= '<ul class="menu-items">';
        foreach ($modulo['submodulos'] as $submodulo) {
            $active = (basename($_SERVER['PHP_SELF']) === basename($submodulo['url'])) ? 'active' : '';
            $html .= '<li class="' . $active . '">';
            $html .= '<a href="' . $submodulo['url'] . '">';
            $html .= '<i class="material-icons">' . $submodulo['icon'] . '</i>';
            $html .= '<span>' . $submodulo['nombre'] . '</span>';
            $html .= '</a>';
            $html .= '</li>';
        }
        $html .= '</ul>';
        $html .= '</div>';
    }
    
    return $html;
}

/**
 * Verifica si un usuario tiene acceso a un archivo específico
 * @param string $file_path Ruta del archivo
 * @param int $user_id ID del usuario
 * @return bool
 */
function tieneAccesoArchivo($file_path, $user_id) {
    global $conn;
    
    // Verificar si es super_admin
    $query = "SELECT rol FROM users WHERE id = ?";
    $stmt = $conn->prepare($query);
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $user = $result->fetch_assoc();
    
    if ($user['rol'] === 'super_admin') {
        return true;
    }

    // Determinar el tipo de archivo y su ID basado en la ruta
    if (strpos($file_path, 'rendicion_') !== false) {
        $tipo = 'rendicion';
        preg_match('/rendicion_(\d+)/', $file_path, $matches);
        $doc_id = $matches[1] ?? 0;
    } elseif (strpos($file_path, 'devolucion_') !== false) {
        $tipo = 'devolucion';
        preg_match('/devolucion_(\d+)/', $file_path, $matches);
        $doc_id = $matches[1] ?? 0;
    } elseif (strpos($file_path, 'orden_') !== false) {
        $tipo = 'orden_compra';
        preg_match('/orden_(\d+)/', $file_path, $matches);
        $doc_id = $matches[1] ?? 0;
    } else {
        return false; // Si no podemos identificar el tipo de archivo
    }

    // Verificar permisos según el tipo de documento
    switch ($tipo) {
        case 'rendicion':
            $query = "SELECT user_id FROM rendiciones WHERE id = ?";
            $permiso_requerido = 'ver_rendiciones';
            break;
        case 'devolucion':
            $query = "SELECT user_id FROM devoluciones WHERE id = ?";
            $permiso_requerido = 'ver_devoluciones';
            break;
        case 'orden_compra':
            $query = "SELECT user_id FROM ordenes_compra WHERE id = ?";
            $permiso_requerido = 'ver_ordenes_compra';
            break;
        default:
            return false;
    }

    // Verificar si el usuario tiene el permiso general
    if (tienePermiso($permiso_requerido)) {
        return true;
    }

    // Verificar si el usuario es el propietario del documento
    $stmt = $conn->prepare($query);
    $stmt->bind_param("i", $doc_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $doc = $result->fetch_assoc();

    return $doc['user_id'] === $user_id;
}

/**
 * Protege el acceso a archivos
 * @param string $file_path Ruta del archivo solicitado
 */
function protegerArchivo($file_path) {
    session_start();
    
    if (!isset($_SESSION['user_id'])) {
        header('HTTP/1.0 403 Forbidden');
        die('Acceso denegado');
    }

    if (!tieneAccesoArchivo($file_path, $_SESSION['user_id'])) {
        header('HTTP/1.0 403 Forbidden');
        die('No tiene permisos para acceder a este archivo');
    }
}

/**
 * Genera una URL segura para la descarga de archivos
 * @param string $file_path Ruta del archivo
 * @return string URL segura para la descarga
 */
function generarUrlArchivo($file_path) {
    return 'download.php?file=' . urlencode($file_path);
}