<?php
/**
 * Gestión de devoluciones de dinero
 * CRUD completo para devoluciones con soporte para subir archivos
 */

// Definir el título de la página
$page_title = 'Gestión de Devoluciones';

// Incluir archivos de configuración y funciones
require_once 'includes/config.php';
require_once 'includes/functions.php';

// Verificar sesión
verificarSesion();
$user_id = $_SESSION['user_id'];
$user_rol = $_SESSION['user_rol'] ?? '';
$user_nombre = $_SESSION['user_nombre'] ?? '';
$user_apellido = $_SESSION['user_apellido'] ?? '';
$is_admin = ($user_rol === 'admin' || $user_rol === 'super_admin');

// Determinar la acción a realizar
$action = $_GET['action'] ?? 'list';
$id = isset($_GET['id']) ? intval($_GET['id']) : 0;

// Conexión a la base de datos
try {
    $conn = getDatabaseConnection();
    
    // Obtener orígenes de devolución
    $stmt = $conn->prepare("SELECT id, nombre FROM origenes_devolucion WHERE activo = 1");
    $stmt->execute();
    $origenes = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Obtener estados
    $estados = obtenerEstados($conn);
    $estados_map = [];
    foreach ($estados as $estado) {
        $estados_map[$estado['id']] = [
            'nombre' => $estado['nombre'],
            'color' => $estado['color']
        ];
    }
    
    // Procesar formulario de creación o edición (incluyendo archivos)
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($action === 'new' || $action === 'edit')) {
        // Datos generales de la devolución
        $descripcion = trim($_POST['descripcion']);
        $observaciones = trim($_POST['observaciones']);
        $origen_id = intval($_POST['origen_id']);
        $fecha = $_POST['fecha'];
        $monto_total = floatval($_POST['monto_total']);
        $archivos = isset($_FILES['soportes']) ? $_FILES['soportes'] : null;
        
        // Validación básica
        $errores = [];
        
        if (empty($descripcion)) {
            $errores[] = "El motivo es obligatorio";
        }
        
        if (empty($fecha)) {
            $errores[] = "La fecha es obligatoria";
        }
        
        if ($origen_id <= 0) {
            $errores[] = "Debe seleccionar un canal de devolución";
        }
        
        if ($monto_total <= 0) {
            $errores[] = "El monto debe ser un número positivo";
        }
        
        // Validar archivos (si se subieron)
        $archivos_subidos = [];
        if ($archivos && !empty($archivos['name'][0])) {
            $tipos_permitidos = ['image/jpeg', 'image/png', 'application/pdf'];
            $tamano_maximo = 5 * 1024 * 1024; // 5 MB
            
            foreach ($archivos['name'] as $index => $nombre) {
                if ($archivos['error'][$index] === UPLOAD_ERR_OK) {
                    $tipo = $archivos['type'][$index];
                    $tamano = $archivos['size'][$index];
                    $tmp_name = $archivos['tmp_name'][$index];
                    
                    // Validar tipo de archivo
                    if (!in_array($tipo, $tipos_permitidos)) {
                        $errores[] = "El archivo '$nombre' no es un tipo permitido (solo JPEG, PNG o PDF)";
                        continue;
                    }
                    
                    // Validar tamaño
                    if ($tamano > $tamano_maximo) {
                        $errores[] = "El archivo '$nombre' excede el tamaño máximo de 5 MB";
                        continue;
                    }
                    
                    // Sanitizar nombre del archivo
                    $nombre_sanitizado = preg_replace('/[^A-Za-z0-9\-\_\.]/', '_', $nombre);
                    $archivos_subidos[] = [
                        'nombre' => $nombre_sanitizado,
                        'tmp_name' => $tmp_name,
                        'tipo' => $tipo
                    ];
                } elseif ($archivos['error'][$index] !== UPLOAD_ERR_NO_FILE) {
                    $errores[] = "Error al subir el archivo '$nombre': " . $archivos['error'][$index];
                }
            }
        }
        
        // Si no hay errores, proceder a guardar
        if (empty($errores)) {
            $conn->beginTransaction();
            
            try {
                // Si es una edición, verificar permisos y plazo de 24 horas
                if ($action === 'edit' && $id > 0) {
                    // Obtener la devolución actual
                    $stmt = $conn->prepare("SELECT * FROM devoluciones WHERE id = :id");
                    $stmt->bindParam(':id', $id);
                    $stmt->execute();
                    $devolucion = $stmt->fetch();
                    
                    // Verificar que la devolución exista
                    if (!$devolucion) {
                        throw new Exception("La devolución no existe");
                    }
                    
                    // Verificar que sea el propietario o administrador
                    if ($devolucion['user_id'] != $user_id && !$is_admin) {
                        throw new Exception("No tiene permisos para editar esta devolución");
                    }
                    
                    // Verificar estado (solo se puede editar si está en estado Enviada o Rechazada)
                    if ($devolucion['estado_id'] != 1 && $devolucion['estado_id'] != 4) {
                        throw new Exception("No se puede editar una devolución que ya ha sido aprobada o está en revisión");
                    }
                    
                    // Verificar plazo de 24 horas (excepto para administradores)
                    if (!$is_admin && !estaEnTiempoEdicion($devolucion['fecha_creacion'])) {
                        throw new Exception("Solo puede editar devoluciones dentro de las 24 horas posteriores a su creación");
                    }
                    
                    // Actualizar la devolución
                    $stmt = $conn->prepare("
                        UPDATE devoluciones 
                        SET descripcion = :descripcion, 
                            observaciones = :observaciones,
                            origen_id = :origen_id,
                            fecha = :fecha, 
                            monto_total = :monto_total, 
                            ultima_modificacion = NOW()
                        WHERE id = :id
                    ");
                    
                    $stmt->bindParam(':descripcion', $descripcion);
                    $stmt->bindParam(':observaciones', $observaciones);
                    $stmt->bindParam(':origen_id', $origen_id);
                    $stmt->bindParam(':fecha', $fecha);
                    $stmt->bindParam(':monto_total', $monto_total);
                    $stmt->bindParam(':id', $id);
                    $stmt->execute();
                    
                    // Registrar actividad
                    registrarActividad($conn, $user_id, 'editar', 'Edición de devolución', $id, 'devolucion');
                    
                    $mensaje = "Devolución actualizada correctamente";
                } else {
                    // Crear nueva devolución
                    $stmt = $conn->prepare("
                        INSERT INTO devoluciones 
                        (user_id, descripcion, observaciones, origen_id, fecha, monto_total, estado_id, fecha_creacion, ultima_modificacion) 
                        VALUES (:user_id, :descripcion, :observaciones, :origen_id, :fecha, :monto_total, 1, NOW(), NOW())
                    ");
                    
                    $stmt->bindParam(':user_id', $user_id);
                    $stmt->bindParam(':descripcion', $descripcion);
                    $stmt->bindParam(':observaciones', $observaciones);
                    $stmt->bindParam(':origen_id', $origen_id);
                    $stmt->bindParam(':fecha', $fecha);
                    $stmt->bindParam(':monto_total', $monto_total);
                    $stmt->execute();
                    
                    $id = $conn->lastInsertId();
                    
                    // Registrar actividad
                    registrarActividad($conn, $user_id, 'crear', 'Creación de devolución', $id, 'devolucion');
                    
                    $mensaje = "Devolución creada correctamente";
                }
                
                // Procesar archivos subidos
                if (!empty($archivos_subidos)) {
                    // Crear directorio para la devolución si no existe
                    $carpeta_devolucion = "uploads/devolucion_$id/";
                    if (!file_exists($carpeta_devolucion)) {
                        mkdir($carpeta_devolucion, 0777, true);
                    }
                    
                    foreach ($archivos_subidos as $archivo) {
                        $nombre_archivo = time() . '_' . $archivo['nombre'];
                        $ruta_destino = $carpeta_devolucion . $nombre_archivo;
                        
                        // Mover el archivo al directorio
                        if (!move_uploaded_file($archivo['tmp_name'], $ruta_destino)) {
                            throw new Exception("Error al mover el archivo '$nombre_archivo'");
                        }
                        
                        // Registrar el archivo en la base de datos
                        $stmt = $conn->prepare("
                            INSERT INTO devolucion_soportes 
                            (devolucion_id, nombre_archivo, ruta_archivo, fecha_subida) 
                            VALUES (:devolucion_id, :nombre_archivo, :ruta_archivo, NOW())
                        ");
                        $stmt->bindParam(':devolucion_id', $id);
                        $stmt->bindParam(':nombre_archivo', $archivo['nombre']);
                        $stmt->bindParam(':ruta_archivo', $ruta_destino);
                        $stmt->execute();
                    }
                }
                
                $conn->commit();
                
                // Redireccionar a la lista con mensaje de éxito
                $_SESSION['flash_message'] = $mensaje;
                $_SESSION['flash_type'] = 'success';
                header('Location: devoluciones.php');
                exit;
                
            } catch (Exception $e) {
                $conn->rollBack();
                $error_message = $e->getMessage();
            }
        } else {
            $error_message = implode("<br>", $errores);
        }
    }
    
    // Procesar eliminación de un archivo de soporte (solo para usuarios autorizados)
    if ($action === 'delete_file' && isset($_GET['file_id']) && ($is_admin || $devolucion['user_id'] == $user_id)) {
        $file_id = intval($_GET['file_id']);
        
        try {
            // Obtener información del archivo
            $stmt = $conn->prepare("SELECT * FROM devolucion_soportes WHERE id = :file_id AND devolucion_id = :devolucion_id");
            $stmt->bindParam(':file_id', $file_id);
            $stmt->bindParam(':devolucion_id', $id);
            $stmt->execute();
            $soporte = $stmt->fetch();
            
            if (!$soporte) {
                throw new Exception("Archivo no encontrado");
            }
            
            // Verificar permisos
            $stmt = $conn->prepare("SELECT user_id, estado_id FROM devoluciones WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $devolucion = $stmt->fetch();
            
            if (!$devolucion || ($devolucion['user_id'] != $user_id && !$is_admin)) {
                throw new Exception("No tiene permisos para eliminar este archivo");
            }
            
            // Solo permitir eliminar si la devolución está en estado Enviada o Rechazada
            if ($devolucion['estado_id'] != 1 && $devolucion['estado_id'] != 4) {
                throw new Exception("No se puede eliminar archivos de una devolución que ya ha sido aprobada o está en revisión");
            }
            
            // Eliminar archivo del sistema de archivos
            if (file_exists($soporte['ruta_archivo'])) {
                unlink($soporte['ruta_archivo']);
            }
            
            // Eliminar registro de la base de datos
            $stmt = $conn->prepare("DELETE FROM devolucion_soportes WHERE id = :file_id");
            $stmt->bindParam(':file_id', $file_id);
            $stmt->execute();
            
            // Registrar actividad
            registrarActividad($conn, $user_id, 'eliminar', 'Eliminación de archivo de soporte de devolución', $id, 'devolucion');
            
            $_SESSION['flash_message'] = "Archivo eliminado correctamente";
            $_SESSION['flash_type'] = 'success';
            header("Location: devoluciones.php?action=view&id=$id");
            exit;
            
        } catch (Exception $e) {
            $error_message = "Error al eliminar el archivo: " . $e->getMessage();
        }
    }
    
    // Procesar aprobación o rechazo
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && $action === 'review' && $is_admin) {
        $decision = $_POST['decision'];
        $comentario = trim($_POST['comentario']);
        
        if ($decision != 'aprobar' && $decision != 'rechazar') {
            $error_message = "Acción no válida";
        } else {
            try {
                // Actualizar estado de la devolución
                $estado_id = ($decision === 'aprobar') ? 3 : 4; // 3=Aprobada, 4=Rechazada
                
                $stmt = $conn->prepare("
                    UPDATE devoluciones 
                    SET estado_id = :estado_id, 
                        revisor_id = :revisor_id, 
                        comentario_revision = :comentario,
                        fecha_revision = NOW()
                    WHERE id = :id
                ");
                
                $stmt->bindParam(':estado_id', $estado_id);
                $stmt->bindParam(':revisor_id', $user_id);
                $stmt->bindParam(':comentario', $comentario);
                $stmt->bindParam(':id', $id);
                $stmt->execute();
                
                // Registrar actividad
                $tipo_accion = ($decision === 'aprobar') ? 'aprobar' : 'rechazar';
                $descripcion = ($decision === 'aprobar') ? 'Aprobación de devolución' : 'Rechazo de devolución';
                registrarActividad($conn, $user_id, $tipo_accion, $descripcion, $id, 'devolucion');
                
                $mensaje = ($decision === 'aprobar') ? 
                    "Devolución aprobada correctamente" : 
                    "Devolución rechazada correctamente";
                
                $_SESSION['flash_message'] = $mensaje;
                $_SESSION['flash_type'] = 'success';
                header('Location: devoluciones.php');
                exit;
            } catch (Exception $e) {
                $error_message = "Error al procesar la solicitud: " . $e->getMessage();
            }
        }
    }
    
    // Acción para eliminar devolución (solo admin)
    if ($action === 'delete' && $is_admin && $id > 0) {
        try {
            // Obtener la devolución para verificar
            $stmt = $conn->prepare("SELECT * FROM devoluciones WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $devolucion = $stmt->fetch();
            
            if (!$devolucion) {
                throw new Exception("La devolución no existe");
            }
            
            // Verificar que no esté aprobada o en revisión
            if ($devolucion['estado_id'] == 2 || $devolucion['estado_id'] == 3 || $devolucion['estado_id'] == 5) {
                throw new Exception("No se puede eliminar una devolución que está en revisión, aprobada o pagada");
            }
            
            // Eliminar devolución (los archivos y registros en devolucion_soportes se eliminan automáticamente por ON DELETE CASCADE)
            $stmt = $conn->prepare("DELETE FROM devoluciones WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            
            // Eliminar carpeta de archivos (por si acaso ON DELETE CASCADE no elimina archivos físicos)
            $carpeta_devolucion = "uploads/devolucion_$id/";
            if (file_exists($carpeta_devolucion)) {
                array_map('unlink', glob("$carpeta_devolucion*.*"));
                rmdir($carpeta_devolucion);
            }
            
            // Registrar actividad
            registrarActividad($conn, $user_id, 'eliminar', 'Eliminación de devolución', $id, 'devolucion');
            
            $_SESSION['flash_message'] = "Devolución eliminada correctamente";
            $_SESSION['flash_type'] = 'success';
            header('Location: devoluciones.php');
            exit;
        } catch (Exception $e) {
            $error_message = "Error al eliminar la devolución: " . $e->getMessage();
        }
    }
    
} catch (PDOException $e) {
    $error_message = 'Error de conexión a la base de datos: ' . $e->getMessage();
}

// Incluir el header
require_once 'includes/header.php';
?>

<!-- Mensaje de error si existe -->
<?php if (isset($error_message)): ?>
<div class="alert alert-danger">
    <?php echo $error_message; ?>
</div>
<?php endif; ?>

<?php
// Mostrar contenido según la acción
switch ($action) {
    case 'new':
        // Formulario para crear nueva devolución
        ?>
        <div class="card">
            <div class="panel-header">
                <h2 class="panel-title">Nueva Solicitud de Devolución</h2>
                <a href="devoluciones.php" class="btn btn-primary btn-sm">
                    <i class="material-icons">arrow_back</i> Volver a la Lista
                </a>
            </div>
            
            <form method="POST" action="devoluciones.php?action=new" id="form-devolucion" enctype="multipart/form-data">
                <style>
                    .form-row {
                        display: flex;
                        flex-wrap: wrap;
                        gap: 20px;
                        margin-bottom: 20px;
                    }
                    .form-col {
                        flex: 1;
                        min-width: 250px;
                    }
                    .form-control:focus {
                        border-color: #1e88e5;
                        box-shadow: 0 0 0 0.15rem rgba(30, 136, 229, 0.25);
                    }
                    .form-control:required {
                        border-left: 3px solid #1e88e5;
                    }
                    .input-sm {
                        padding: 8px 10px;
                        height: auto;
                    }
                </style>
                
                <div class="form-row">
                    <div class="form-col">
                        <label for="fecha" class="form-label">Fecha de Devolución</label>
                        <input type="date" id="fecha" name="fecha" class="form-control input-sm" required
                               value="<?php echo isset($_POST['fecha']) ? $_POST['fecha'] : date('Y-m-d'); ?>">
                    </div>
                    
                    <div class="form-col">
                        <label for="monto_total" class="form-label">Monto a Devolver</label>
                        <div class="input-group">
                            <span class="input-group-text">$</span>
                            <input type="number" id="monto_total" name="monto_total" class="form-control input-sm" 
                                   min="1" step="1" placeholder="0" required autocomplete="off"
                                   value="<?php echo isset($_POST['monto_total']) ? htmlspecialchars($_POST['monto_total']) : ''; ?>">
                        </div>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-col">
                        <label for="origen_id" class="form-label">Canal de Devolución</label>
                        <select id="origen_id" name="origen_id" class="form-control input-sm" required>
                            <option value="">Seleccione un canal</option>
                            <?php foreach ($origenes as $origen): ?>
                            <option value="<?php echo $origen['id']; ?>" <?php echo isset($_POST['origen_id']) && $_POST['origen_id'] == $origen['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($origen['nombre']); ?>
                            </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-col">
                        <label for="descripcion" class="form-label">Motivo de la Devolución</label>
                        <input type="text" id="descripcion" name="descripcion" class="form-control input-sm" 
                               placeholder="Ej: Transferencia duplicada, saldo a favor, devolución de tienda" required autocomplete="off"
                               value="<?php echo isset($_POST['descripcion']) ? htmlspecialchars($_POST['descripcion']) : ''; ?>">
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-col">
                        <label for="observaciones" class="form-label">Observaciones</label>
                        <textarea id="observaciones" name="observaciones" class="form-control input-sm" 
                                placeholder="Información adicional importante sobre la devolución" rows="3"><?php echo isset($_POST['observaciones']) ? htmlspecialchars($_POST['observaciones']) : ''; ?></textarea>
                        <small class="text-muted">Puede incluir información adicional relevante para la devolución</small>
                    </div>
                </div>
                
                <!-- Sección para subir archivos -->
                <div class="card" style="margin-top: 20px;">
                    <div class="panel-header">
                        <h3 class="panel-title">Archivos de Soporte</h3>
                    </div>
                    <div class="form-group">
                        <label for="soportes" class="form-label">Subir Archivos (JPEG, PNG, PDF - Máx. 5MB)</label>
                        <input type="file" id="soportes" name="soportes[]" class="form-control" multiple
                               accept="image/jpeg,image/png,application/pdf">
                        <small class="text-muted">Puede adjuntar comprobantes de pago, recibos, facturas u otros documentos que respalden la solicitud de devolución.</small>
                    </div>
                </div>
                
                <div style="margin-top: 20px; text-align: center;">
                    <button type="submit" class="btn btn-primary">
                        <i class="material-icons">save</i> Guardar Devolución
                    </button>
                </div>
            </form>
        </div>
        <?php
        break;
        
    case 'edit':
        // Formulario para editar devolución existente
        try {
            // Obtener datos de la devolución
            $stmt = $conn->prepare("
                SELECT d.* 
                FROM devoluciones d
                WHERE d.id = :id " . (!$is_admin ? "AND d.user_id = :user_id" : "")
            );
            
            $stmt->bindParam(':id', $id);
            if (!$is_admin) {
                $stmt->bindParam(':user_id', $user_id);
            }
            
            $stmt->execute();
            $devolucion = $stmt->fetch();
            
            if (!$devolucion) {
                throw new Exception("Devolución no encontrada o sin permisos para editarla");
            }
            
            // Verificar estado (solo se puede editar si está en estado Enviada o Rechazada)
            if ($devolucion['estado_id'] != 1 && $devolucion['estado_id'] != 4) {
                throw new Exception("No se puede editar una devolución que ya ha sido aprobada o está en revisión");
            }
            
            // Verificar plazo de 24 horas (excepto para administradores)
            if (!$is_admin && !estaEnTiempoEdicion($devolucion['fecha_creacion'])) {
                throw new Exception("Solo puede editar devoluciones dentro de las 24 horas posteriores a su creación");
            }
            
            // Obtener archivos de soporte
            $stmt = $conn->prepare("SELECT * FROM devolucion_soportes WHERE devolucion_id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $soportes = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            ?>
            <div class="card">
                <div class="panel-header">
                    <h2 class="panel-title">Editar Solicitud de Devolución</h2>
                    <a href="devoluciones.php" class="btn btn-primary btn-sm">
                        <i class="material-icons">arrow_back</i> Volver a la Lista
                    </a>
                </div>
                
                <form method="POST" action="devoluciones.php?action=edit&id=<?php echo $id; ?>" id="form-devolucion" enctype="multipart/form-data">
                    <style>
                        .form-row {
                            display: flex;
                            flex-wrap: wrap;
                            gap: 20px;
                            margin-bottom: 20px;
                        }
                        .form-col {
                            flex: 1;
                            min-width: 250px;
                        }
                        .form-control:focus {
                            border-color: #1e88e5;
                            box-shadow: 0 0 0 0.15rem rgba(30, 136, 229, 0.25);
                        }
                        .form-control:required {
                            border-left: 3px solid #1e88e5;
                        }
                        .input-sm {
                            padding: 8px 10px;
                            height: auto;
                        }
                    </style>
                    
                    <div class="form-row">
                        <div class="form-col">
                            <label for="fecha" class="form-label">Fecha de Devolución</label>
                            <input type="date" id="fecha" name="fecha" class="form-control input-sm" required
                                   value="<?php echo $devolucion['fecha']; ?>">
                        </div>
                        
                        <div class="form-col">
                            <label for="monto_total" class="form-label">Monto a Devolver</label>
                            <div class="input-group">
                                <span class="input-group-text">$</span>
                                <input type="number" id="monto_total" name="monto_total" class="form-control input-sm" 
                                       min="1" step="1" required
                                       value="<?php echo $devolucion['monto_total']; ?>">
                            </div>
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-col">
                            <label for="origen_id" class="form-label">Canal de Devolución</label>
                            <select id="origen_id" name="origen_id" class="form-control input-sm" required>
                                <option value="">Seleccione un canal</option>
                                <?php foreach ($origenes as $origen): ?>
                                <option value="<?php echo $origen['id']; ?>" <?php echo $devolucion['origen_id'] == $origen['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($origen['nombre']); ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="form-col">
                            <label for="descripcion" class="form-label">Motivo de la Devolución</label>
                            <input type="text" id="descripcion" name="descripcion" class="form-control input-sm" 
                                   placeholder="Ej: Transferencia duplicada, saldo a favor, devolución de tienda" required
                                   value="<?php echo htmlspecialchars($devolucion['descripcion']); ?>">
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-col">
                            <label for="observaciones" class="form-label">Observaciones</label>
                            <textarea id="observaciones" name="observaciones" class="form-control input-sm" 
                                    placeholder="Información adicional importante sobre la devolución" rows="3"><?php echo htmlspecialchars($devolucion['observaciones']); ?></textarea>
                            <small class="text-muted">Puede incluir información adicional relevante para la devolución</small>
                        </div>
                    </div>
                    
                    <!-- Sección para archivos de soporte -->
                    <div class="card" style="margin-top: 20px;">
                        <div class="panel-header">
                            <h3 class="panel-title">Archivos de Soporte</h3>
                        </div>
                        <?php if (!empty($soportes)): ?>
                        <div class="table-responsive">
                            <table>
                                <thead>
                                    <tr>
                                        <th>Archivo</th>
                                        <th>Fecha Subida</th>
                                        <th>Acciones</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($soportes as $soporte): ?>
                                    <tr>
                                        <td>
                                            <a href="<?php echo $soporte['ruta_archivo']; ?>" target="_blank">
                                                <?php echo htmlspecialchars($soporte['nombre_archivo']); ?>
                                            </a>
                                        </td>
                                        <td><?php echo date('d/m/Y H:i', strtotime($soporte['fecha_subida'])); ?></td>
                                        <td>
                                            <a href="devoluciones.php?action=delete_file&id=<?php echo $id; ?>&file_id=<?php echo $soporte['id']; ?>" 
                                               onclick="return confirm('¿Está seguro de eliminar este archivo?')"
                                               class="btn btn-danger btn-sm">
                                                <i class="material-icons">delete</i>
                                            </a>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        <?php else: ?>
                        <p>No hay archivos de soporte subidos.</p>
                        <?php endif; ?>
                        <div class="form-group" style="margin-top: 20px;">
                            <label for="soportes" class="form-label">Subir Nuevos Archivos (JPEG, PNG, PDF - Máx. 5MB)</label>
                            <input type="file" id="soportes" name="soportes[]" class="form-control" multiple
                                   accept="image/jpeg,image/png,application/pdf">
                        </div>
                    </div>
                    
                    <div style="margin-top: 20px; text-align: center;">
                        <button type="submit" class="btn btn-primary">
                            <i class="material-icons">save</i> Actualizar Devolución
                        </button>
                    </div>
                </form>
            </div>
            <?php
        } catch (Exception $e) {
            echo '<div class="alert alert-danger">' . $e->getMessage() . '</div>';
            echo '<div style="text-align: center; margin-top: 20px;">
                      <a href="devoluciones.php" class="btn btn-primary">
                          <i class="material-icons">arrow_back</i> Volver a la Lista
                      </a>
                  </div>';
        }
        break;
        
    case 'view':
        // Ver detalles de una devolución
        try {
            // Obtener datos de la devolución
            $stmt = $conn->prepare("
                SELECT d.*, o.nombre as origen_nombre,
                       u.nombre, u.apellido, u.rut, u.cargo,
                       rev.nombre as revisor_nombre, rev.apellido as revisor_apellido
                FROM devoluciones d
                JOIN origenes_devolucion o ON d.origen_id = o.id
                JOIN users u ON d.user_id = u.id
                LEFT JOIN users rev ON d.revisor_id = rev.id
                WHERE d.id = :id " . (!$is_admin ? "AND (d.user_id = :user_id OR d.revisor_id = :user_id)" : "")
            );
            
            $stmt->bindParam(':id', $id);
            if (!$is_admin) {
                $stmt->bindParam(':user_id', $user_id);
            }
            
            $stmt->execute();
            $devolucion = $stmt->fetch();
            
            if (!$devolucion) {
                throw new Exception("Devolución no encontrada o sin permisos para verla");
            }
            
            // Obtener archivos de soporte
            $stmt = $conn->prepare("SELECT * FROM devolucion_soportes WHERE devolucion_id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $soportes = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Generar código
            $codigo = generarCodigo('DEV', $id);
            
            ?>
            <div class="card">
                <div class="panel-header">
                    <h2 class="panel-title">Detalle de Devolución: <?php echo $codigo; ?></h2>
                    <div class="button-group">
                        <a href="devoluciones.php" class="btn btn-primary btn-sm">
                            <i class="material-icons">arrow_back</i> Volver a la Lista
                        </a>
                        
                        <a href="generate_pdf.php?tipo=devolucion&id=<?php echo $id; ?>" class="btn btn-<?php echo ($devolucion['estado_id'] == 3 || $devolucion['estado_id'] == 5) ? 'success' : 'warning'; ?> btn-sm" target="_blank">
                            <i class="material-icons">picture_as_pdf</i> Generar PDF
                        </a>
                        
                        <?php if (($devolucion['estado_id'] == 1 || $devolucion['estado_id'] == 4) && 
                                  ($devolucion['user_id'] == $user_id || $is_admin)): ?>
                            <a href="devoluciones.php?action=edit&id=<?php echo $id; ?>" class="btn btn-warning btn-sm">
                                <i class="material-icons">edit</i> Editar
                            </a>
                        <?php endif; ?>
                        
                        <?php if ($is_admin && ($devolucion['estado_id'] == 1 || $devolucion['estado_id'] == 4)): ?>
                            <a href="devoluciones.php?action=delete&id=<?php echo $id; ?>" 
                               onclick="return confirm('¿Está seguro de eliminar esta devolución?')"
                               class="btn btn-danger btn-sm">
                                <i class="material-icons">delete</i> Eliminar
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <div class="row" style="display: flex; flex-wrap: wrap; margin: 0 -10px;">
                        <div class="col" style="flex: 1; padding: 0 10px; min-width: 250px;">
                            <div class="card" style="height: 100%;">
                                <div class="panel-header">
                                    <h3 class="panel-title">Información General</h3>
                                </div>
                                <table>
                                    <tr>
                                        <th style="width: 40%;">Código:</th>
                                        <td><?php echo $codigo; ?></td>
                                    </tr>
                                    <tr>
                                        <th>Fecha:</th>
                                        <td><?php echo formatoFecha($devolucion['fecha']); ?></td>
                                    </tr>
                                    <tr>
                                        <th>Estado:</th>
                                        <td>
                                            <span class="status-badge badge-estado-<?php echo $devolucion['estado_id']; ?>">
                                                <?php echo $estados_map[$devolucion['estado_id']]['nombre'] ?? 'Desconocido'; ?>
                                            </span>
                                        </td>
                                    </tr>
                                    <tr>
                                        <th>Monto Total:</th>
                                        <td><?php echo formatoMoneda($devolucion['monto_total']); ?></td>
                                    </tr>
                                    <tr>
                                        <th>Fecha Creación:</th>
                                        <td><?php echo date('d/m/Y H:i', strtotime($devolucion['fecha_creacion'])); ?></td>
                                    </tr>
                                    <tr>
                                        <th>Motivo:</th>
                                        <td><?php echo nl2br(htmlspecialchars($devolucion['descripcion'])); ?></td>
                                    </tr>
                                    <?php if (!empty($devolucion['observaciones'])): ?>
                                    <tr>
                                        <th>Observaciones:</th>
                                        <td><?php echo nl2br(htmlspecialchars($devolucion['observaciones'])); ?></td>
                                    </tr>
                                    <?php endif; ?>
                                    <tr>
                                        <th>Canal:</th>
                                        <td><?php echo htmlspecialchars($devolucion['origen_nombre']); ?></td>
                                    </tr>
                                </table>
                            </div>
                        </div>
                        
                        <div class="col" style="flex: 1; padding: 0 10px; min-width: 250px;">
                            <div class="card" style="height: 100%;">
                                <div class="panel-header">
                                    <h3 class="panel-title">Información del Usuario</h3>
                                </div>
                                <table>
                                    <tr>
                                        <th style="width: 40%;">Nombre:</th>
                                        <td><?php echo htmlspecialchars($devolucion['nombre'] . ' ' . $devolucion['apellido']); ?></td>
                                    </tr>
                                    <tr>
                                        <th>RUT:</th>
                                        <td><?php echo htmlspecialchars($devolucion['rut']); ?></td>
                                    </tr>
                                    <tr>
                                        <th>Cargo:</th>
                                        <td><?php echo htmlspecialchars($devolucion['cargo']); ?></td>
                                    </tr>
                                    <?php if ($devolucion['revisor_id']): ?>
                                    <tr>
                                        <th>Revisado por:</th>
                                        <td><?php echo htmlspecialchars($devolucion['revisor_nombre'] . ' ' . $devolucion['revisor_apellido']); ?></td>
                                    </tr>
                                    <tr>
                                        <th>Fecha Revisión:</th>
                                        <td><?php echo date('d/m/Y H:i', strtotime($devolucion['fecha_revision'])); ?></td>
                                    </tr>
                                    <?php endif; ?>
                                    <?php if (!empty($devolucion['comentario_revision'])): ?>
                                    <tr>
                                        <th>Comentario:</th>
                                        <td><?php echo nl2br(htmlspecialchars($devolucion['comentario_revision'])); ?></td>
                                    </tr>
                                    <?php endif; ?>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Sección para mostrar archivos de soporte -->
                <div class="card" style="margin-top: 20px;">
                    <div class="panel-header">
                        <h3 class="panel-title">Archivos de Soporte</h3>
                    </div>
                    <?php if (!empty($soportes)): ?>
                    <div class="table-responsive">
                        <table>
                            <thead>
                                <tr>
                                    <th>#</th>
                                    <th>Archivo</th>
                                    <th>Fecha Subida</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($soportes as $index => $soporte): ?>
                                <tr>
                                    <td><?php echo $index + 1; ?></td>
                                    <td>
                                        <a href="<?php echo $soporte['ruta_archivo']; ?>" target="_blank">
                                            <?php echo htmlspecialchars($soporte['nombre_archivo']); ?>
                                        </a>
                                    </td>
                                    <td><?php echo date('d/m/Y H:i', strtotime($soporte['fecha_subida'])); ?></td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    <?php else: ?>
                    <p>No hay archivos de soporte subidos.</p>
                    <?php endif; ?>
                </div>
                
                <?php if ($is_admin && $devolucion['estado_id'] == 1): // Si está pendiente y es admin ?>
                <div class="card" style="margin-top: 20px;">
                    <div class="panel-header">
                        <h3 class="panel-title">Revisión de Devolución</h3>
                    </div>
                    
                    <form method="POST" action="devoluciones.php?action=review&id=<?php echo $id; ?>">
                        <div class="form-group">
                            <label for="decision" class="form-label">Decisión</label>
                            <select id="decision" name="decision" class="form-control" required>
                                <option value="">Seleccione una opción</option>
                                <option value="aprobar">Aprobar</option>
                                <option value="rechazar">Rechazar</option>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="comentario" class="form-label">Comentario (opcional)</label>
                            <textarea id="comentario" name="comentario" class="form-control" 
                                      rows="3" placeholder="Ingrese comentarios sobre la revisión"></textarea>
                        </div>
                        
                        <div style="text-align: center;">
                            <button type="submit" class="btn btn-primary">
                                <i class="material-icons">check_circle</i> Enviar Revisión
                            </button>
                        </div>
                    </form>
                </div>
                <?php endif; ?>
            </div>
            <?php
        } catch (Exception $e) {
            echo '<div class="alert alert-danger">' . $e->getMessage() . '</div>';
            echo '<div style="text-align: center; margin-top: 20px;">
                      <a href="devoluciones.php" class="btn btn-primary">
                          <i class="material-icons">arrow_back</i> Volver a la Lista
                      </a>
                  </div>';
        }
        break;
        
    default:
        // Lista de devoluciones
        try {
            // Filtros de búsqueda
            $estado = isset($_GET['estado']) ? intval($_GET['estado']) : 0;
            $fecha_desde = isset($_GET['fecha_desde']) ? $_GET['fecha_desde'] : '';
            $fecha_hasta = isset($_GET['fecha_hasta']) ? $_GET['fecha_hasta'] : '';
            $busqueda = isset($_GET['busqueda']) ? trim($_GET['busqueda']) : '';
            
            // Construir consulta base
            $sql = "
                SELECT d.*, u.nombre, u.apellido, u.rut, o.nombre as origen_nombre
                FROM devoluciones d
                JOIN users u ON d.user_id = u.id
                JOIN origenes_devolucion o ON d.origen_id = o.id
                WHERE 1=1
            ";
            
            $params = [];
            
            // Si no es admin, solo ver sus propias devoluciones
            if (!$is_admin) {
                $sql .= " AND d.user_id = :user_id";
                $params[':user_id'] = $user_id;
            }
            
            // Aplicar filtros
            if ($estado > 0) {
                $sql .= " AND d.estado_id = :estado";
                $params[':estado'] = $estado;
            }
            
            if (!empty($fecha_desde)) {
                $sql .= " AND d.fecha >= :fecha_desde";
                $params[':fecha_desde'] = $fecha_desde;
            }
            
            if (!empty($fecha_hasta)) {
                $sql .= " AND d.fecha <= :fecha_hasta";
                $params[':fecha_hasta'] = $fecha_hasta;
            }
            
            if (!empty($busqueda)) {
                $sql .= " AND (d.descripcion LIKE :busqueda 
                               OR u.nombre LIKE :busqueda 
                               OR u.apellido LIKE :busqueda 
                               OR u.rut LIKE :busqueda)";
                $params[':busqueda'] = '%' . $busqueda . '%';
            }
            
            // Ordenar por fecha de creación (más reciente primero)
            $sql .= " ORDER BY d.fecha_creacion DESC";
            
            // Preparar y ejecutar consulta
            $stmt = $conn->prepare($sql);
            foreach ($params as $param => $value) {
                $stmt->bindValue($param, $value);
            }
            
            $stmt->execute();
            $devoluciones = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            ?>
            <div class="card">
                <div class="panel-header">
                    <h2 class="panel-title">Devoluciones de Dinero</h2>
                    <?php if (tienePermiso('crear_devolucion') || $is_admin): ?>
                    <a href="devoluciones.php?action=new" class="btn btn-primary btn-sm">
                        <i class="material-icons">add</i> Nueva Devolución
                    </a>
                    <?php endif; ?>
                </div>
                
                <!-- Filtros de búsqueda -->
                <div class="card" style="margin-bottom: 20px;">
                    <div class="panel-header">
                        <h3 class="panel-title">Filtros de Búsqueda</h3>
                    </div>
                    
                    <form method="GET" action="devoluciones.php" class="form-inline" style="display: flex; flex-wrap: wrap; gap: 10px;">
                        <div style="flex: 1; min-width: 200px;">
                            <label for="estado" class="form-label">Estado</label>
                            <select id="estado" name="estado" class="form-control">
                                <option value="0">Todos</option>
                                <?php foreach ($estados as $est): ?>
                                <option value="<?php echo $est['id']; ?>" <?php echo $estado == $est['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($est['nombre']); ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div style="flex: 1; min-width: 200px;">
                            <label for="fecha_desde" class="form-label">Fecha Desde</label>
                            <input type="date" id="fecha_desde" name="fecha_desde" class="form-control" 
                                   value="<?php echo $fecha_desde; ?>">
                        </div>
                        
                        <div style="flex: 1; min-width: 200px;">
                            <label for="fecha_hasta" class="form-label">Fecha Hasta</label>
                            <input type="date" id="fecha_hasta" name="fecha_hasta" class="form-control" 
                                   value="<?php echo $fecha_hasta; ?>">
                        </div>
                        
                        <div style="flex: 2; min-width: 300px;">
                            <label for="busqueda" class="form-label">Búsqueda</label>
                            <input type="text" id="busqueda" name="busqueda" class="form-control" 
                                   placeholder="Descripción, usuario, RUT..." value="<?php echo htmlspecialchars($busqueda); ?>">
                        </div>
                        
                        <div style="display: flex; align-items: flex-end; gap: 10px;">
                            <button type="submit" class="btn btn-primary">
                                <i class="material-icons">search</i> Buscar
                            </button>
                            
                            <a href="devoluciones.php" class="btn btn-danger">
                                <i class="material-icons">clear</i> Limpiar
                            </a>
                        </div>
                    </form>
                </div>
                
                <style>
                /* Estilos para botones y tablas */
                .button-group {
                    display: flex;
                    gap: 10px;
                    align-items: center;
                }
                
                .btn {
                    display: inline-flex;
                    align-items: center;
                    gap: 5px;
                    padding: 8px 15px;
                    border-radius: 4px;
                    font-size: 14px;
                    font-weight: 500;
                    text-decoration: none;
                    cursor: pointer;
                    border: none;
                    transition: all 0.2s;
                }
                
                .btn-sm {
                    padding: 6px 12px;
                    font-size: 13px;
                }
                
                .btn-primary {
                    background-color: #1e88e5;
                    color: white;
                }
                
                .btn-success {
                    background-color: #28a745;
                    color: white;
                }
                
                .btn-warning {
                    background-color: #ffa726;
                    color: white;
                }
                
                .btn-danger {
                    background-color: #ef5350;
                    color: white;
                }
                
                .btn:hover {
                    opacity: 0.9;
                    transform: translateY(-1px);
                }
                
                .btn:active {
                    transform: translateY(0);
                }
                
                .btn i {
                    font-size: 18px;
                }
                
                /* Estilos para tablas */
                .table-responsive {
                    overflow-x: auto;
                    margin: 15px 0;
                }
                
                table {
                    width: 100%;
                    border-collapse: collapse;
                    margin-bottom: 1rem;
                    background-color: transparent;
                }
                
                th, td {
                    padding: 12px;
                    vertical-align: top;
                    border-top: 1px solid #dee2e6;
                }
                
                th {
                    font-weight: 500;
                    text-align: left;
                    background-color: #f8f9fa;
                }
                
                tbody tr:nth-of-type(odd) {
                    background-color: rgba(0,0,0,.02);
                }
                
                tbody tr:hover {
                    background-color: rgba(0,0,0,.04);
                }
                
                /* Estilos para badges de estado */
                .status-badge {
                    display: inline-block;
                    padding: 4px 8px;
                    border-radius: 4px;
                    font-size: 12px;
                    font-weight: 500;
                    text-transform: uppercase;
                }
                
                .badge-estado-1 { /* Enviada */
                    background-color: #fff3cd;
                    color: #856404;
                }
                
                .badge-estado-2 { /* En revisión */
                    background-color: #cce5ff;
                    color: #004085;
                }
                
                .badge-estado-3 { /* Aprobada */
                    background-color: #d4edda;
                    color: #155724;
                }
                
                .badge-estado-4 { /* Rechazada */
                    background-color: #f8d7da;
                    color: #721c24;
                }
                
                .badge-estado-5 { /* Pagada */
                    background-color: #e2e3e5;
                    color: #383d41;
                }
                
                /* Estilos para íconos de acciones */
                .action-icon {
                    display: inline-flex;
                    align-items: center;
                    justify-content: center;
                    width: 28px;
                    height: 28px;
                    border-radius: 50%;
                    margin-right: 5px;
                    transition: all 0.2s;
                    color: #555;
                    background-color: #f0f0f0;
                    text-decoration: none;
                }
                
                .action-icon:hover {
                    background-color: #e0e0e0;
                    transform: scale(1.1);
                }
                
                </style>
                
                <div class="table-responsive">
                    <table class="devoluciones-table">
                        <thead>
                            <tr>
                                <th>Código</th>
                                <th>Fecha</th>
                                <th>Motivo</th>
                                <th>Canal</th>
                                <th>Monto</th>
                                <?php if ($is_admin): ?>
                                <th>Usuario</th>
                                <?php endif; ?>
                                <th>Estado</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($devoluciones)): ?>
                            <tr>
                                <td colspan="<?php echo $is_admin ? '8' : '7'; ?>" style="text-align: center;">
                                    No se encontraron devoluciones
                                </td>
                            </tr>
                            <?php else: ?>
                                <?php foreach ($devoluciones as $devolucion): ?>
                                <?php $codigo = generarCodigo('DEV', $devolucion['id']); ?>
                                <tr>
                                    <td><?php echo $codigo; ?></td>
                                    <td><?php echo formatoFecha($devolucion['fecha']); ?></td>
                                    <td><?php echo htmlspecialchars(substr($devolucion['descripcion'], 0, 40)) . (strlen($devolucion['descripcion']) > 40 ? '...' : ''); ?></td>
                                    <td><?php echo htmlspecialchars($devolucion['origen_nombre']); ?></td>
                                    <td><?php echo formatoMoneda($devolucion['monto_total']); ?></td>
                                    <?php if ($is_admin): ?>
                                    <td><?php echo htmlspecialchars($devolucion['nombre'] . ' ' . $devolucion['apellido']); ?></td>
                                    <?php endif; ?>
                                    <td>
                                        <span class="status-badge badge-estado-<?php echo $devolucion['estado_id']; ?>">
                                            <?php echo $estados_map[$devolucion['estado_id']]['nombre'] ?? 'Desconocido'; ?>
                                        </span>
                                    </td>
                                    <td>
                                        <a href="devoluciones.php?action=view&id=<?php echo $devolucion['id']; ?>" 
                                           title="Ver detalles" class="action-icon view-icon">
                                            <i class="material-icons">visibility</i>
                                        </a>
                                        
                                        <a href="generate_pdf.php?tipo=devolucion&id=<?php echo $devolucion['id']; ?>" 
                                           title="Generar PDF" class="action-icon pdf-icon" target="_blank">
                                            <i class="material-icons">picture_as_pdf</i>
                                        </a>
                                        
                                        <?php if (($devolucion['estado_id'] == 1 || $devolucion['estado_id'] == 4) && 
                                                 ($devolucion['user_id'] == $user_id || $is_admin)): ?>
                                        <a href="devoluciones.php?action=edit&id=<?php echo $devolucion['id']; ?>" 
                                           title="Editar" class="action-icon edit-icon">
                                            <i class="material-icons">edit</i>
                                        </a>
                                        <?php endif; ?>
                                        
                                        <?php if ($is_admin && ($devolucion['estado_id'] == 1 || $devolucion['estado_id'] == 4)): ?>
                                        <a href="devoluciones.php?action=delete&id=<?php echo $devolucion['id']; ?>" 
                                           onclick="return confirm('¿Está seguro de eliminar esta devolución?')"
                                           title="Eliminar" class="action-icon delete-icon">
                                            <i class="material-icons">delete</i>
                                        </a>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <?php
        } catch (PDOException $e) {
            echo '<div class="alert alert-danger">Error al cargar devoluciones: ' . $e->getMessage() . '</div>';
        }
        break;
}
?>

<?php
// Incluir el footer
require_once 'includes/footer.php';
?>