<?php
/**
 * Módulo de Gestión de Categorías
 */

$page_title = 'Gestión de Categorías';

require_once 'includes/config.php';
require_once 'includes/functions.php';
require_once 'includes/permisos.php';

// Verificar sesión y permisos
verificarSesion();
$user_id = $_SESSION['user_id'];
$user_rol = $_SESSION['user_rol'] ?? '';

// Solo permitir acceso a super_admin
if ($user_rol !== 'super_admin') {
    redirigirConMensaje('dashboard.php', "Acceso denegado. Solo administradores del sistema pueden acceder a esta sección.", 'danger');
    exit;
}

// Verificar permiso específico
if (!tienePermiso('configuracion_sistema')) {
    redirigirConMensaje('dashboard.php', "No tiene los permisos necesarios para acceder a esta sección.", 'danger');
    exit;
}

$conn = getDatabaseConnection();
$mensaje = '';
$tipo_mensaje = '';

// Procesar formulario
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    try {
        $conn->beginTransaction();
        
        switch ($action) {
            case 'crear':
                $nombre = filter_var($_POST['nombre'], FILTER_SANITIZE_STRING);
                $tipo = filter_var($_POST['tipo'], FILTER_SANITIZE_STRING);
                $codigo = filter_var($_POST['codigo'], FILTER_SANITIZE_STRING);
                $descripcion = filter_var($_POST['descripcion'], FILTER_SANITIZE_STRING);
                
                // Guardar en la tabla específica según el tipo
                switch ($tipo) {
                    case 'gasto':
                        $stmt = $conn->prepare("
                            INSERT INTO categorias_gastos (nombre, descripcion, activo, codigo)
                            VALUES (:nombre, :descripcion, 1, :codigo)
                        ");
                        break;
                        
                    case 'repuesto':
                        $stmt = $conn->prepare("
                            INSERT INTO categorias_repuestos (nombre, descripcion, activo)
                            VALUES (:nombre, :descripcion, 1)
                        ");
                        break;
                        
                    default:
                        // Para otros tipos, usar la tabla genérica
                        $stmt = $conn->prepare("
                            INSERT INTO categorias_personalizadas (nombre, tipo, codigo, descripcion)
                            VALUES (:nombre, :tipo, :codigo, :descripcion)
                        ");
                        $stmt->bindParam(':tipo', $tipo);
                        break;
                }
                
                $stmt->bindParam(':nombre', $nombre);
                $stmt->bindParam(':descripcion', $descripcion);
                if ($tipo === 'gasto' || $tipo === 'otro') {
                    $stmt->bindParam(':codigo', $codigo);
                }
                
                $stmt->execute();
                
                registrarActividad($conn, $user_id, 'crear_categoria', 
                    "Creación de categoría: $nombre ($tipo)");
                
                $mensaje = "Categoría creada correctamente.";
                $tipo_mensaje = "success";
                break;
                
            case 'editar':
                $id = filter_var($_POST['id'], FILTER_SANITIZE_NUMBER_INT);
                $nombre = filter_var($_POST['nombre'], FILTER_SANITIZE_STRING);
                $tipo = filter_var($_POST['tipo'], FILTER_SANITIZE_STRING);
                $codigo = filter_var($_POST['codigo'], FILTER_SANITIZE_STRING);
                $descripcion = filter_var($_POST['descripcion'], FILTER_SANITIZE_STRING);
                $activo = isset($_POST['activo']) ? 1 : 0;
                
                // Actualizar en la tabla específica según el tipo
                switch ($tipo) {
                    case 'gasto':
                        $stmt = $conn->prepare("
                            UPDATE categorias_gastos 
                            SET nombre = :nombre,
                                descripcion = :descripcion,
                                codigo = :codigo,
                                activo = :activo
                            WHERE id = :id
                        ");
                        $stmt->bindParam(':codigo', $codigo);
                        break;
                        
                    case 'repuesto':
                        $stmt = $conn->prepare("
                            UPDATE categorias_repuestos 
                            SET nombre = :nombre,
                                descripcion = :descripcion,
                                activo = :activo
                            WHERE id = :id
                        ");
                        break;
                        
                    default:
                        $stmt = $conn->prepare("
                            UPDATE categorias_personalizadas 
                            SET nombre = :nombre,
                                codigo = :codigo,
                                descripcion = :descripcion,
                                activo = :activo
                            WHERE id = :id
                        ");
                        $stmt->bindParam(':codigo', $codigo);
                        break;
                }
                
                $stmt->bindParam(':id', $id);
                $stmt->bindParam(':nombre', $nombre);
                $stmt->bindParam(':descripcion', $descripcion);
                $stmt->bindParam(':activo', $activo);
                
                $stmt->execute();
                
                registrarActividad($conn, $user_id, 'editar_categoria', 
                    "Edición de categoría ID: $id");
                
                $mensaje = "Categoría actualizada correctamente.";
                $tipo_mensaje = "success";
                break;
                
            case 'eliminar':
                $id = filter_var($_POST['id'], FILTER_SANITIZE_NUMBER_INT);
                $tipo = filter_var($_POST['tipo'], FILTER_SANITIZE_STRING);
                
                // Verificar si la categoría está en uso según su tipo
                switch ($tipo) {
                    case 'gasto':
                        $stmt = $conn->prepare("
                            SELECT COUNT(*) 
                            FROM rendicion_categoria 
                            WHERE categoria_id = :id
                        ");
                        break;
                        
                    case 'repuesto':
                        $stmt = $conn->prepare("
                            SELECT COUNT(*) 
                            FROM solicitud_repuesto_detalle 
                            WHERE repuesto_id IN (
                                SELECT id FROM repuestos WHERE categoria_id = :id
                            )
                        ");
                        break;
                        
                    default:
                        $stmt = $conn->prepare("
                            SELECT COUNT(*) 
                            FROM categorias_personalizadas 
                            WHERE id = :id
                        ");
                        break;
                }
                
                $stmt->execute([':id' => $id]);
                
                if ($stmt->fetchColumn() > 0) {
                    throw new Exception("No se puede eliminar la categoría porque está en uso.");
                }
                
                // Eliminar de la tabla correspondiente
                switch ($tipo) {
                    case 'gasto':
                        $stmt = $conn->prepare("DELETE FROM categorias_gastos WHERE id = :id");
                        break;
                    case 'repuesto':
                        $stmt = $conn->prepare("DELETE FROM categorias_repuestos WHERE id = :id");
                        break;
                    default:
                        $stmt = $conn->prepare("DELETE FROM categorias_personalizadas WHERE id = :id");
                        break;
                }
                
                $stmt->execute([':id' => $id]);
                
                registrarActividad($conn, $user_id, 'eliminar_categoria', 
                    "Eliminación de categoría ID: $id");
                
                $mensaje = "Categoría eliminada correctamente.";
                $tipo_mensaje = "success";
                break;
        }
        
        $conn->commit();
    } catch (Exception $e) {
        $conn->rollBack();
        $mensaje = "Error: " . $e->getMessage();
        $tipo_mensaje = "danger";
    }
}

// Obtener tipos de categorías disponibles
$tipos_categoria = [
    'gasto' => 'Gastos',
    'repuesto' => 'Repuestos',
    'documento' => 'Documentos',
    'otro' => 'Otros'
];

// Obtener categorías de todas las tablas
$categorias = [];

try {
    // Obtener categorías de gastos
    $stmt = $conn->query("
        SELECT 
            id,
            nombre,
            descripcion,
            activo,
            codigo,
            'gasto' as tipo
        FROM categorias_gastos
        ORDER BY nombre
    ");
    while ($cat = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $categorias['gasto'][] = $cat;
    }

    // Obtener categorías de repuestos
    $stmt = $conn->query("
        SELECT 
            id,
            nombre,
            descripcion,
            activo,
            NULL as codigo,
            'repuesto' as tipo
        FROM categorias_repuestos
        ORDER BY nombre
    ");
    while ($cat = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $categorias['repuesto'][] = $cat;
    }

    // Obtener otras categorías personalizadas
    $stmt = $conn->query("
        SELECT 
            id,
            nombre,
            descripcion,
            activo,
            codigo,
            tipo
        FROM categorias_personalizadas
        WHERE tipo NOT IN ('gasto', 'repuesto')
        ORDER BY tipo, nombre
    ");
    while ($cat = $stmt->fetch(PDO::FETCH_ASSOC)) {
        if (!isset($categorias[$cat['tipo']])) {
            $categorias[$cat['tipo']] = [];
        }
        $categorias[$cat['tipo']][] = $cat;
    }
} catch (PDOException $e) {
    $mensaje = "Error al obtener las categorías: " . $e->getMessage();
    $tipo_mensaje = "danger";
}

// Incluir el header
require_once 'includes/header.php';
?>

<style>
/* Estilos generales */
.card {
    background: #fff;
    border: none;
    box-shadow: 0 2px 15px rgba(0,0,0,0.08);
    border-radius: 12px;
}

.card-header {
    background: linear-gradient(45deg, #1e88e5, #1976d2);
    color: white;
    border-radius: 12px 12px 0 0 !important;
    padding: 20px;
}

.card-title {
    margin: 0;
    font-size: 1.5rem;
    font-weight: 500;
}

.card-body {
    padding: 25px;
}

/* Estilos de las secciones de tipo */
.tipo-section {
    background: #fff;
    border-radius: 10px;
    padding: 20px;
    margin-bottom: 30px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.05);
}

.tipo-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 25px;
    padding-bottom: 15px;
    border-bottom: 2px solid #f0f0f0;
}

.tipo-header h3 {
    display: flex;
    align-items: center;
    gap: 10px;
    margin: 0;
    color: #1e88e5;
    font-size: 1.3rem;
}

.tipo-header h3 i {
    background: #e3f2fd;
    padding: 8px;
    border-radius: 8px;
    color: #1e88e5;
}

.tipo-header .badge {
    font-size: 0.9rem;
    padding: 8px 12px;
    border-radius: 20px;
    background: #e3f2fd;
    color: #1e88e5;
}

/* Estilos de las tarjetas de categoría */
.categoria-card {
    background: #fff;
    border-radius: 10px;
    padding: 20px;
    margin-bottom: 15px;
    box-shadow: 0 3px 10px rgba(0,0,0,0.05);
    transition: all 0.3s ease;
    border: 1px solid #f0f0f0;
}

.categoria-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(0,0,0,0.1);
    border-color: #1e88e5;
}

.categoria-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 15px;
}

.categoria-header h4 {
    font-size: 1.1rem;
    color: #2c3e50;
    margin: 0;
    font-weight: 500;
}

.categoria-actions {
    display: flex;
    gap: 8px;
}

.categoria-actions .btn {
    padding: 6px;
    border-radius: 8px;
    transition: all 0.2s ease;
}

.categoria-actions .btn:hover {
    transform: scale(1.05);
}

.categoria-actions .btn-info {
    background: #1e88e5;
    border: none;
}

.categoria-actions .btn-danger {
    background: #dc3545;
    border: none;
}

.categoria-info {
    color: #666;
    font-size: 0.95rem;
}

.categoria-codigo {
    font-family: 'Roboto Mono', monospace;
    background: #f8f9fa;
    padding: 4px 8px;
    border-radius: 4px;
    color: #2c3e50;
    font-size: 0.9rem;
    border: 1px solid #e9ecef;
}

.categoria-estado {
    display: inline-flex;
    align-items: center;
    padding: 5px 12px;
    border-radius: 20px;
    font-size: 0.85rem;
    font-weight: 500;
    margin-top: 10px;
}

.categoria-estado.activo {
    background: #e8f5e9;
    color: #2e7d32;
}

.categoria-estado.inactivo {
    background: #ffebee;
    color: #c62828;
}

/* Estilos del botón de nueva categoría */
.btn-nueva-categoria {
    background: linear-gradient(45deg, #2196f3, #1976d2);
    border: none;
    padding: 10px 20px;
    border-radius: 8px;
    font-weight: 500;
    transition: all 0.3s ease;
    box-shadow: 0 2px 5px rgba(0,0,0,0.1);
}

.btn-nueva-categoria:hover {
    background: linear-gradient(45deg, #1976d2, #1565c0);
    transform: translateY(-1px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.15);
}

.btn-nueva-categoria i {
    margin-right: 8px;
}

/* Estilos de los modales */
.modal-content {
    border: none;
    border-radius: 12px;
    box-shadow: 0 5px 20px rgba(0,0,0,0.1);
}

.modal-header {
    background: linear-gradient(45deg, #1e88e5, #1976d2);
    color: white;
    border-radius: 12px 12px 0 0;
    padding: 20px;
}

.modal-title {
    font-weight: 500;
}

.modal-body {
    padding: 25px;
}

.form-group label {
    font-weight: 500;
    color: #2c3e50;
    margin-bottom: 8px;
}

.form-control {
    border-radius: 8px;
    border: 1px solid #e0e0e0;
    padding: 10px 15px;
}

.form-control:focus {
    border-color: #1e88e5;
    box-shadow: 0 0 0 0.2rem rgba(30,136,229,0.15);
}

.modal-footer {
    padding: 20px;
    border-top: 1px solid #f0f0f0;
}

.custom-switch .custom-control-label::before {
    height: 1.5rem;
    width: 2.75rem;
    border-radius: 1rem;
}

.custom-switch .custom-control-label::after {
    height: calc(1.5rem - 4px);
    width: calc(1.5rem - 4px);
    border-radius: 50%;
}

.custom-control-input:checked ~ .custom-control-label::before {
    background-color: #1e88e5;
    border-color: #1e88e5;
}

/* Estilos para mensajes de alerta */
.alert {
    border: none;
    border-radius: 10px;
    padding: 15px 20px;
    margin-bottom: 20px;
}

.alert-success {
    background-color: #e8f5e9;
    color: #2e7d32;
}

.alert-danger {
    background-color: #ffebee;
    color: #c62828;
}

/* Estilos para estado vacío */
.alert-info {
    background-color: #e3f2fd;
    color: #1565c0;
    border-radius: 8px;
    padding: 15px 20px;
    display: flex;
    align-items: center;
    gap: 10px;
}

.alert-info i {
    font-size: 1.2rem;
}

/* Animaciones */
@keyframes fadeIn {
    from { opacity: 0; transform: translateY(10px); }
    to { opacity: 1; transform: translateY(0); }
}

.categoria-card {
    animation: fadeIn 0.3s ease-out;
}

/* Estilos adicionales para los popups */
.modal-dialog-centered {
    display: flex;
    align-items: center;
    min-height: calc(100% - 3.5rem);
}

.modal-content {
    border: none;
    box-shadow: 0 3px 15px rgba(0,0,0,0.2);
}

.modal-header {
    display: flex;
    align-items: center;
    padding: 1rem 1.5rem;
}

.modal-header .material-icons {
    margin-right: 0.5rem;
    font-size: 1.5rem;
}

.modal-header.bg-danger {
    background: linear-gradient(45deg, #dc3545, #c82333) !important;
}

.modal-body {
    padding: 1.5rem;
}

.modal-footer {
    padding: 1rem 1.5rem;
    border-top: 1px solid #dee2e6;
}

.categoria-actions .btn {
    width: 32px;
    height: 32px;
    padding: 0;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 6px;
    transition: all 0.2s ease;
}

.categoria-actions .btn:hover {
    transform: translateY(-2px);
}

.categoria-actions .btn-info {
    background: linear-gradient(45deg, #2196f3, #1976d2);
}

.categoria-actions .btn-danger {
    background: linear-gradient(45deg, #dc3545, #c82333);
}

.categoria-actions .material-icons {
    font-size: 18px;
}

/* Animación para los modales */
.modal.fade .modal-dialog {
    transform: scale(0.8);
    transition: transform 0.2s ease-out;
}

.modal.show .modal-dialog {
    transform: scale(1);
}

/* Estilos para popups flotantes */
.popup-overlay {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 1050;
    opacity: 0;
    visibility: hidden;
    transition: all 0.3s ease;
}

.popup-overlay.show {
    opacity: 1;
    visibility: visible;
}

.popup-container {
    background: white;
    border-radius: 12px;
    width: 100%;
    max-width: 500px;
    transform: scale(0.7);
    transition: transform 0.3s ease;
    box-shadow: 0 5px 25px rgba(0,0,0,0.2);
}

.popup-overlay.show .popup-container {
    transform: scale(1);
}

.popup-header {
    background: linear-gradient(45deg, #1e88e5, #1976d2);
    color: white;
    padding: 20px;
    border-radius: 12px 12px 0 0;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.popup-header h5 {
    margin: 0;
    font-size: 1.1rem;
    display: flex;
    align-items: center;
    gap: 8px;
}

.popup-close {
    background: none;
    border: none;
    color: white;
    font-size: 1.5rem;
    padding: 0;
    cursor: pointer;
    opacity: 0.8;
    transition: opacity 0.2s;
}

.popup-close:hover {
    opacity: 1;
}

.popup-body {
    padding: 20px;
}

.popup-footer {
    padding: 15px 20px;
    border-top: 1px solid #eee;
    display: flex;
    justify-content: flex-end;
    gap: 10px;
}

.popup-footer .btn {
    min-width: 100px;
}

/* Estilos para los botones de acción */
.categoria-actions .btn {
    width: 36px;
    height: 36px;
}

.categoria-actions .btn i {
    font-size: 20px;
}

/* Animación para los popups */
@keyframes popupIn {
    from { transform: scale(0.7); opacity: 0; }
    to { transform: scale(1); opacity: 1; }
}

@keyframes popupOut {
    from { transform: scale(1); opacity: 1; }
    to { transform: scale(0.7); opacity: 0; }
}

.popup-container.animate-in {
    animation: popupIn 0.3s ease forwards;
}

.popup-container.animate-out {
    animation: popupOut 0.3s ease forwards;
}
</style>

<?php if ($mensaje): ?>
<div class="alert alert-<?php echo $tipo_mensaje; ?> alert-dismissible fade show">
    <?php echo $mensaje; ?>
    <button type="button" class="close" data-dismiss="alert">
        <span>&times;</span>
    </button>
</div>
<?php endif; ?>

<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <div>
            <h2 class="card-title mb-0">
                <i class="material-icons mr-2">category</i>
                Gestión de Categorías
            </h2>
            <small class="text-white-50">Administre las categorías del sistema</small>
        </div>
        <button type="button" class="btn btn-nueva-categoria" data-toggle="modal" data-target="#modalNuevaCategoria">
            <i class="material-icons">add</i>
            Nueva Categoría
        </button>
    </div>
    
    <div class="card-body">
        <?php foreach ($tipos_categoria as $tipo => $nombre_tipo): ?>
        <div class="tipo-section">
            <div class="tipo-header">
                <h3>
                    <i class="material-icons"><?php echo getTipoIcon($tipo); ?></i>
                    <?php echo $nombre_tipo; ?>
                </h3>
                <span class="badge badge-info">
                    <?php echo count($categorias[$tipo] ?? []); ?> categorías
                </span>
            </div>
            
            <?php if (isset($categorias[$tipo]) && !empty($categorias[$tipo])): ?>
                <?php foreach ($categorias[$tipo] as $categoria): ?>
                <div class="categoria-card">
                    <div class="categoria-header">
                        <h4><?php echo htmlspecialchars($categoria['nombre']); ?></h4>
                        <div class="categoria-actions">
                            <button type="button" class="btn btn-sm btn-info" 
                                    data-toggle="tooltip" 
                                    title="Editar categoría"
                                    onclick='editarCategoria(<?php echo json_encode([
                                        "id" => $categoria['id'],
                                        "tipo" => $categoria['tipo'],
                                        "nombre" => $categoria['nombre'],
                                        "codigo" => $categoria['codigo'],
                                        "descripcion" => $categoria['descripcion'],
                                        "activo" => $categoria['activo']
                                    ]); ?>)'>
                                <i class="material-icons">edit</i>
                            </button>
                            <button type="button" class="btn btn-sm btn-danger" 
                                    data-toggle="tooltip" 
                                    title="Eliminar categoría"
                                    onclick="mostrarPopupEliminar(<?php echo $categoria['id']; ?>, '<?php echo $tipo; ?>', '<?php echo htmlspecialchars($categoria['nombre']); ?>')">
                                <i class="material-icons">delete</i>
                            </button>
                        </div>
                    </div>
                    
                    <div class="categoria-info">
                        <?php if (!empty($categoria['codigo'])): ?>
                            <div class="mb-1">
                                Código: <span class="categoria-codigo"><?php echo htmlspecialchars($categoria['codigo']); ?></span>
                            </div>
                        <?php endif; ?>
                        
                        <?php if (!empty($categoria['descripcion'])): ?>
                            <div class="mb-1">
                                <?php echo htmlspecialchars($categoria['descripcion']); ?>
                            </div>
                        <?php endif; ?>
                        
                        <div>
                            <span class="categoria-estado <?php echo $categoria['activo'] ? 'activo' : 'inactivo'; ?>">
                                <?php echo $categoria['activo'] ? 'Activo' : 'Inactivo'; ?>
                            </span>
                        </div>
                    </div>
                </div>
                <?php endforeach; ?>
            <?php else: ?>
                <div class="alert alert-info">
                    No hay categorías definidas para <?php echo strtolower($nombre_tipo); ?>.
                </div>
            <?php endif; ?>
        </div>
        <?php endforeach; ?>
    </div>
</div>

<!-- Modal Nueva Categoría -->
<div class="modal fade" id="modalNuevaCategoria" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST">
                <input type="hidden" name="action" value="crear">
                
                <div class="modal-header">
                    <h5 class="modal-title">Nueva Categoría</h5>
                    <button type="button" class="close" data-dismiss="modal">
                        <span>&times;</span>
                    </button>
                </div>
                
                <div class="modal-body">
                    <div class="form-group">
                        <label>Nombre</label>
                        <input type="text" name="nombre" class="form-control" required>
                    </div>
                    
                    <div class="form-group">
                        <label>Tipo</label>
                        <select name="tipo" class="form-control" required>
                            <?php foreach ($tipos_categoria as $tipo => $nombre): ?>
                                <option value="<?php echo $tipo; ?>"><?php echo $nombre; ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label>Código</label>
                        <input type="text" name="codigo" class="form-control">
                        <small class="form-text text-muted">Opcional. Código único para la categoría.</small>
                    </div>
                    
                    <div class="form-group">
                        <label>Descripción</label>
                        <textarea name="descripcion" class="form-control" rows="3"></textarea>
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Guardar</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Popup de Edición -->
<div class="popup-overlay" id="popupEditar">
    <div class="popup-container">
        <form method="POST">
            <input type="hidden" name="action" value="editar">
            <input type="hidden" name="id" id="editar_id">
            <input type="hidden" name="tipo" id="editar_tipo">
            
            <div class="popup-header">
                <h5>
                    <i class="material-icons">edit</i>
                    Editar Categoría
                </h5>
                <button type="button" class="popup-close" onclick="cerrarPopup('popupEditar')">&times;</button>
            </div>
            
            <div class="popup-body">
                <div class="form-group">
                    <label>Nombre</label>
                    <input type="text" name="nombre" id="editar_nombre" class="form-control" required>
                </div>
                
                <div class="form-group codigo-field">
                    <label>Código</label>
                    <input type="text" name="codigo" id="editar_codigo" class="form-control">
                    <small class="text-muted">Código único para la categoría</small>
                </div>
                
                <div class="form-group">
                    <label>Descripción</label>
                    <textarea name="descripcion" id="editar_descripcion" class="form-control" rows="3"></textarea>
                </div>
                
                <div class="form-group">
                    <div class="custom-control custom-switch">
                        <input type="checkbox" name="activo" id="editar_activo" class="custom-control-input" value="1">
                        <label class="custom-control-label" for="editar_activo">Activo</label>
                    </div>
                </div>
            </div>
            
            <div class="popup-footer">
                <button type="button" class="btn btn-secondary" onclick="cerrarPopup('popupEditar')">Cancelar</button>
                <button type="submit" class="btn btn-primary">
                    <i class="material-icons">save</i>
                    Guardar
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Popup de Eliminación -->
<div class="popup-overlay" id="popupEliminar">
    <div class="popup-container">
        <form method="POST">
            <input type="hidden" name="action" value="eliminar">
            <input type="hidden" name="id" id="eliminar_id">
            <input type="hidden" name="tipo" id="eliminar_tipo">
            
            <div class="popup-header bg-danger">
                <h5>
                    <i class="material-icons">warning</i>
                    Confirmar Eliminación
                </h5>
                <button type="button" class="popup-close" onclick="cerrarPopup('popupEliminar')">&times;</button>
            </div>
            
            <div class="popup-body">
                <p class="mb-0">¿Está seguro que desea eliminar la categoría <strong id="nombreCategoriaEliminar"></strong>?</p>
                <small class="text-danger">Esta acción no se puede deshacer.</small>
            </div>
            
            <div class="popup-footer">
                <button type="button" class="btn btn-secondary" onclick="cerrarPopup('popupEliminar')">Cancelar</button>
                <button type="submit" class="btn btn-danger">
                    <i class="material-icons">delete_forever</i>
                    Eliminar
                </button>
            </div>
        </form>
    </div>
</div>

<script>
// Funciones para manejar los popups
function mostrarPopup(id) {
    document.getElementById(id).classList.add('show');
    document.body.style.overflow = 'hidden';
}

function cerrarPopup(id) {
    const overlay = document.getElementById(id);
    const container = overlay.querySelector('.popup-container');
    
    container.classList.add('animate-out');
    setTimeout(() => {
        overlay.classList.remove('show');
        container.classList.remove('animate-out');
        document.body.style.overflow = '';
    }, 300);
}

function editarCategoria(categoria) {
    document.getElementById('editar_id').value = categoria.id;
    document.getElementById('editar_tipo').value = categoria.tipo;
    document.getElementById('editar_nombre').value = categoria.nombre;
    document.getElementById('editar_codigo').value = categoria.codigo || '';
    document.getElementById('editar_descripcion').value = categoria.descripcion || '';
    document.getElementById('editar_activo').checked = categoria.activo == 1;
    
    // Mostrar/ocultar campo de código según el tipo
    const codigoField = document.querySelector('.codigo-field');
    if (categoria.tipo === 'gasto' || categoria.tipo === 'otro') {
        codigoField.style.display = 'block';
    } else {
        codigoField.style.display = 'none';
    }
    
    mostrarPopup('popupEditar');
}

function mostrarPopupEliminar(id, tipo, nombre) {
    document.getElementById('eliminar_id').value = id;
    document.getElementById('eliminar_tipo').value = tipo;
    document.getElementById('nombreCategoriaEliminar').textContent = nombre;
    mostrarPopup('popupEliminar');
}

// Cerrar popups con la tecla Escape
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        document.querySelectorAll('.popup-overlay.show').forEach(popup => {
            cerrarPopup(popup.id);
        });
    }
});

// Cerrar popups al hacer clic fuera
document.querySelectorAll('.popup-overlay').forEach(overlay => {
    overlay.addEventListener('click', function(e) {
        if (e.target === this) {
            cerrarPopup(this.id);
        }
    });
});

// Inicializar tooltips
$(function () {
    $('[data-toggle="tooltip"]').tooltip();
});
</script>

<?php
/**
 * Obtiene el icono correspondiente a cada tipo de categoría
 */
function getTipoIcon($tipo) {
    $icons = [
        'gasto' => 'attach_money',
        'repuesto' => 'build',
        'documento' => 'description',
        'otro' => 'category'
    ];
    
    return $icons[$tipo] ?? 'category';
}

// Incluir el footer
require_once 'includes/footer.php';
?> 